package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.Schema;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.ex.node.DocException;
import com.atlassian.adf.model.mark.Em;
import com.atlassian.adf.model.mark.Link;
import com.atlassian.adf.model.mark.Mark;
import com.atlassian.adf.model.mark.Strong;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.stream.Stream;

import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

/**
 * The root container node of every Atlassian Document Format (ADF) document.
 * <h2>About ADF</h2>
 * ADF stands for Atlassian Document Format. It is a structured JSON document, conforming to
 * a documented <a href="https://go.atlassian.com/adf-json-schema">JSON schema</a>, that represents
 * a piece of rich content to be displayed in an Atlassian Cloud application. It might represent a
 * comment on a Confluence page or the description of an issue in Jira.
 * <h2>About ADF's Structure</h2>
 * An ADF document is composed of a hierarchy of nodes. There are two categories of nodes:
 * block and {@link InlineContent inline}. Block nodes define the structural elements of the
 * document such as {@link Heading headings}, {@link Paragraph paragraphs}, {@link Table tables},
 * {@link Rule horizontal rule} separators, {@link BulletList lists}, and the like. Inline nodes
 * specify the document content such as text and images. Several node types can also accept
 * {@link Mark marks} that decorate that node type with formatting information or text markup
 * such as {@link Strong bold}, {@link Em italics}, and {@link Link links}.
 * <p>
 * A document is ordered; that is, there's a single sequential path through it: traversing
 * a document in sequence and concatenating the nodes yields content in the correct order.
 * Every independent ADF value must consist of a single well-formed {@code doc} node. The various
 * top-level block nodes (which in this library are those marked with the {@link DocContent}
 * interface) may be added to this {@code doc} to indicate the rich content that should be
 * displayed for it.
 * <h2>About This Library</h2>
 * This library is a manually constructed representation of the JSON schema that has been
 * published for ADF. It does not consult that schema when validating the content, so there
 * is always some risk that it will disagree with some aspect of the schema. However, if
 * any such discrepancies are found, please
 * <a href="https://bitbucket.org/atlassian/adf-builder-java/issues">report this as a bug</a>.
 * <p>
 * Additionally, any new features that are added to the schema have very limited support
 * in this library and may not be preserved across round trips if the library version is
 * too far behind. See {@link Schema#version()} for information about which JSON schema
 * version was consulted at the time this library was last checked against it.
 * <p>
 * Most of the elements that are available in this library share the following features:
 * <ul>
 *     <li>The various elements use static factories with appropriate names so that
 *     they can conveniently be used as static imports and combined intuitively like in
 *     the Java example below.</li>
 *     <li>Most elements are mutable. The only cases where they aren't are those
 *     that have little or no state, such that making them mutable does not really make
 *     any sense. For example, {@link Strong strong} marks don't have anything to mutate.</li>
 *     <li>The only erroneous state that the library usually tolerates is failing to add
 *     content to a node that requires it. Required parameters are demanded and validated
 *     by the static factory methods used to construct the node and that construction is
 *     not complete until all of them have been set. These have names like
 *     {@link Status.Partial.NeedsColor} that try to help indicate why the element is
 *     not fully constructed.</li>
 * </ul>
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link #doc(DocContent) doc}(
 *         {@link Paragraph#p(InlineContent[]) p}(
 *                 {@link Text#text(String) text}("Hello "),
 *                 {@link Text#text(String) text}("world").{@link Text#strong() strong}()
 *         )
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "version": 1,
 *     "type": "doc",
 *     "content": [
 *       {
 *         "type": "paragraph",
 *         "content": [
 *           {
 *             "type": "text",
 *             "text": "Hello "
 *           },
 *           {
 *             "type": "text",
 *             "text": "world",
 *             "marks": [
 *               {
 *                 "type": "strong"
 *               }
 *             ]
 *           }
 *         ]
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <strong>world</strong></p>
 * </div>
 *
 * @see #toMap()
 * @see #parse(Map)
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/nodes/doc/">Node - doc</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class Doc extends AbstractContentNode<Doc, DocContent> {
    /**
     * The JSON schema version that was current the last time that this library was updated for it.
     *
     * @deprecated Use {@link Schema#version()} instead. Since v0.18.0.
     */
    @Deprecated(forRemoval = true)
    @SuppressWarnings("unused")
    public static final String SCHEMA_VERSION = Schema.version();

    static Factory<Doc> FACTORY = new Factory<>(Type.DOC, Doc.class, Doc::parse);
    private static final Version DEFAULT_VERSION = Version.V1;

    private Version version = DEFAULT_VERSION;

    private Doc() {
    }

    /**
     * @return a new, empty document
     */
    public static Doc doc() {
        return new Doc();
    }

    /**
     * @return a new document with the given content
     */
    public static Doc doc(DocContent content) {
        return doc().content(content);
    }

    /**
     * @return a new document with the given content
     */
    public static Doc doc(DocContent... content) {
        return doc().content(content);
    }

    /**
     * @return a new document with the given content
     */
    public static Doc doc(Iterable<? extends DocContent> content) {
        return doc().content(content);
    }

    /**
     * @return a new document with the given content
     */
    public static Doc doc(Stream<? extends DocContent> content) {
        return doc().content(content);
    }

    /**
     * @return a new document with the given version
     */
    public static Doc doc(Version version) {
        return doc().version(version);
    }

    /**
     * @return a new document with the given version and content
     */
    public static Doc doc(Version version, DocContent content) {
        return doc(version).content(content);
    }

    /**
     * @return a new document with the given version and content
     */
    public static Doc doc(Version version, DocContent... content) {
        return doc(version).content(content);
    }

    /**
     * @return a new document with the given version and content
     */
    public static Doc doc(Version version, Iterable<? extends DocContent> content) {
        return doc(version).content(content);
    }

    /**
     * @return a new document with the given version and content
     */
    public static Doc doc(Version version, Stream<? extends DocContent> content) {
        return doc(version).content(content);
    }

    @Override
    public Doc copy() {
        return parse(toMap());
    }

    /**
     * Sets a new version for this document.
     *
     * @param version the new document version
     * @return {@code this}
     */
    public Doc version(Version version) {
        this.version = requireNonNull(version, "version");
        return this;
    }

    /**
     * Returns the version of the ADF standard used for this document.
     *
     * @return the version of the ADF standard used for this document.
     */
    public Version version() {
        return version;
    }

    @Override
    protected boolean contentNodeEquals(Doc other) {
        return version == other.version;
    }

    @Override
    protected int contentNodeHashCode() {
        return version.hashCode();
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("version", version);
    }

    @Override
    public String elementType() {
        return Type.DOC;
    }

    @Override
    protected void validateContentNodeForAppend(DocContent node) {
        super.validateContentNodeForAppend(node);
    }

    /**
     * Renders this document as a single map of string keys to their values, suitable for direct
     * translation into JSON.
     *
     * @return this document, as a map
     */
    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.VERSION, version.value())
                .add(Key.CONTENT, contentFieldMaps());
    }

    /**
     * Parses ADF from a single map of string keys to their values, suitable for direct interpretation
     * from JSON.
     *
     * @param map the map of keys to values representing the ADF
     * @return the parsed and validated document
     * @throws AdfException if there is a problem parsing the document
     */
    public static Doc parse(Map<String, ?> map) {
        // Other parse methods have no need to do this because the containing content node takes
        // care of it, but Doc is the top level so the caller can't do this for us.
        return frame("doc", () -> {
            String type = getTypeOrThrow(map);
            if (!Type.DOC.equals(type)) {
                throw new DocException.InvalidTopLevel(type);
            }

            Version version = Version.parse(getOrThrow(map, Key.VERSION));
            return doc(version)
                    .parseRequiredContentAllowEmpty(map, DocContent.class);
        });
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        int pos = sb.length();
        appendPlainTextContentJoinedWith('\n', sb);

        while (pos < sb.length()) {
            pos = sb.indexOf("\n\n", pos);
            if (pos == -1) break;
            ++pos;
            int stop = pos + 1;
            while (stop < sb.length() && sb.charAt(stop) == '\n') ++stop;
            sb.delete(pos, stop);
        }
    }

    public enum Version {
        V1(1);

        private final int value;

        Version(int value) {
            this.value = value;
        }

        public int value() {
            return value;
        }

        static Version parse(Number version) {
            return parse(asInt(version, "version"));
        }

        static Version parse(int version) {
            if (version == V1.value()) {
                return V1;
            }
            throw new DocException.UnsupportedVersion(version);
        }
    }
}
