package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.node.type.*;
import com.atlassian.adf.util.Factory;

import java.util.Map;
import java.util.UUID;
import java.util.stream.Stream;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * Decision lists provide lists that look much like a {@link BulletList bulletList}, but they cannot be
 * nested and there are differences in their visual representation. The decision items also require
 * identifier strings that can be used to associate them with information in other systems.
 *
 * <p>
 * Note: Jira does not currently support decision lists.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * {@link Doc#doc(DocContent...) doc}(
 *     {@link Paragraph#p(String) p}("Hello"),
 *     {@link #decisionList(String, DecisionItem...) decisionList}(
 *         "decision-list-id",
 *         {@link DecisionItem#decisionItem() decisionItem}()
 *                 .{@link DecisionItem.Partial.NeedsLocalId#localId(String) localId}("item-test-id")
 *                 .{@link DecisionItem.Partial.NeedsState#decided() decided}()
 *                 .{@link DecisionItem#content(String...) content}("Start using Document Format for apps")
 *     ),
 *     {@link Paragraph#p(String) p}("World")
 * );
 * <h3>ADF</h3>
 * <pre>{@code
 *  {
 *    "type": "doc",
 *    "version": 1,
 *    "content": [
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Hello"
 *          }
 *        ]
 *      },
 *      {
 *        "type": "decisionList",
 *        "attrs": {
 *          "localId": "decision-list-id"
 *        },
 *        "content": [
 *          {
 *            "type": "decisionItem",
 *            "attrs": {
 *              "localId": "item-test-id",
 *              "state": "DECIDED"
 *            },
 *            "content": [
 *              {
 *                "type": "text",
 *                "text": "Start using Document Format for apps"
 *              }
 *            ]
 *          }
 *        ]
 *      },
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "World"
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello</p>
 * <ol style="list-style-type: none; background-color: rgba(9, 30, 66, 0.04);">
 *     <li>
 *         <img src="data:image/png;base64,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" />
 *         Start using Document Format for apps
 *     </li>
 * </ol>
 * <p>World</p>
 * </div>
 * <p>
 * Note: This example output is not using AtlasKit to render the decision list, so while it gives a vague
 * impression of what a "decision list" is, it does not faithfully reproduce the actual presentation in
 * Atlassian products.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class DecisionList
        extends AbstractContentNode<DecisionList, DecisionItem>
        implements DocContent, LayoutColumnContent, NonNestableBlockContent, TableCellContent {

    static final Factory<DecisionList> FACTORY = new Factory<>(
            Type.DECISION_LIST,
            DecisionList.class,
            DecisionList::parse
    );

    private String localId;

    private DecisionList(String localId) {
        this.localId = requireNonNull(localId, "localId");
    }

    private DecisionList(UUID localId) {
        this(requireNonNull(localId, "localId").toString());
    }

    public static Partial.NeedsLocalId decisionList() {
        return new Partial.NeedsLocalId();
    }

    public static DecisionList decisionList(String localId) {
        return new DecisionList(localId);
    }

    public static DecisionList decisionList(String localId, DecisionItem content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(String localId, DecisionItem... content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(String localId, Iterable<? extends DecisionItem> content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(String localId, Stream<? extends DecisionItem> content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(UUID localId) {
        return new DecisionList(localId);
    }

    public static DecisionList decisionList(UUID localId, DecisionItem content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(UUID localId, DecisionItem... content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(UUID localId, Iterable<? extends DecisionItem> content) {
        return decisionList(localId).content(content);
    }

    public static DecisionList decisionList(UUID localId, Stream<? extends DecisionItem> content) {
        return decisionList(localId).content(content);
    }

    public DecisionList localId(String localId) {
        this.localId = requireNonNull(localId, "localId");
        return this;
    }

    public DecisionList localId(UUID localId) {
        return localId(requireNonNull(localId, "localId").toString());
    }

    public String localId() {
        return localId;
    }

    @Override
    public DecisionList copy() {
        return parse(toMap());
    }

    @Override
    public String elementType() {
        return Type.DECISION_LIST;
    }

    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        return mapWithType()
                .let(this::addContent)
                .add(Key.ATTRS, map(Attr.LOCAL_ID, localId));
    }

    @Override
    protected boolean contentNodeEquals(DecisionList other) {
        return localId.equals(other.localId);
    }

    @Override
    protected int contentNodeHashCode() {
        return localId.hashCode();
    }

    @Override
    protected void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("localId", localId);
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    private static DecisionList parse(Map<String, ?> map) {
        checkType(map, Type.DECISION_LIST);
        return decisionList(getAttrOrThrow(map, Attr.LOCAL_ID, String.class))
                .parseRequiredContent(map, DecisionItem.class);
    }


    /**
     * Types that represent a partially constructed {@link DecisionList decisionList}.
     */
    public interface Partial {
        /**
         * This partially constructed {@code decisionList} still needs its {@code localId} attribute set.
         */
        class NeedsLocalId {
            NeedsLocalId() {
            }

            public DecisionList localId(String localId) {
                return new DecisionList(localId);
            }
        }
    }
}
