package com.atlassian.adf.model.node;

import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.ex.node.TableCellException;
import com.atlassian.adf.model.node.type.TableCellContent;
import com.atlassian.adf.model.node.type.TableCellNode;
import com.atlassian.annotations.Internal;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.atlassian.adf.model.node.Paragraph.p;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.asInt;
import static com.atlassian.adf.util.ParserSupport.getAttr;
import static com.atlassian.adf.util.ParserSupport.getAttrInt;

/**
 * Either of the data-containing nodes that can be placed in a {@link TableRow}, meaning either a
 * {@link TableHeader tableHeader} or a {@link TableCell tableCell}.
 * <p>
 * Only the nodes that implement the {@link TableCellContent} marker interface are permitted as content.
 *
 * @param <N> the subclass's own type
 */
@Internal
public abstract class AbstractTableCellNode<N extends AbstractTableCellNode<N>>
        extends AbstractContentNode<N, TableCellContent>
        implements TableCellNode<N> {

    @Nullable
    protected String background;
    @Nullable
    protected Integer colspan;
    @Nullable
    protected int[] colwidth;
    @Nullable
    protected Integer rowspan;

    AbstractTableCellNode() {
    }

    @Override
    public Optional<String> background() {
        return Optional.ofNullable(background);
    }

    @Override
    public N background(@Nullable String background) {
        this.background = background;
        return self();
    }

    public N content(String content) {
        return content(p(content));
    }

    public N content(String... content) {
        return content(p(content));
    }

    @Override
    public Optional<Integer> rowspan() {
        return Optional.ofNullable(rowspan);
    }

    @Override
    public N rowspan(@Nullable Integer rowspan) {
        checkRowspan(rowspan);
        this.rowspan = rowspan;
        return self();
    }

    @Override
    public Optional<Integer> colspan() {
        return Optional.ofNullable(colspan);
    }

    @Override
    public Optional<int[]> colwidth() {
        return Optional.ofNullable(colwidth)
                .map(int[]::clone);
    }

    public N colspan(@Nullable Integer colspan) {
        return colspanAndColwidth(colspan, null);
    }

    public N colwidth(@Nullable int... colwidth) {
        return (colwidth != null)
                ? colspanAndColwidth(colwidth.length, colwidth)
                : colspanAndColwidth(null, null);
    }

    @Override
    public N colspanAndColwidth(@Nullable Integer colspan, @Nullable int[] colwidth) {
        checkColspanAndColwidth(colspan, colwidth);
        this.colspan = colspan;
        this.colwidth = colwidth;
        return self();
    }

    @Override
    protected void contentNodeValidate() {
        requireNotEmpty();
    }

    /**
     * Note: The {@code map} of attributes is always added, even if it is empty. This is a temporary patch to
     * mirror the behaviour of the TypeScript parser (which always includes an empty attributes {@code "attrs": {}}
     * field in table cells), in order to pass consistency checks during rollout. This change will be reverted when
     * {@code TableParser.OMIT_TABLE_ATTRIBUTES} is rolled out.
     */
    @Override
    public Map<String, ?> toMap() {
        requireNotEmpty();
        Map<String, ?> attrs = map()
                .addIfPresent(Attr.BACKGROUND, background)
                .addIfPresent(Attr.COLSPAN, colspan)
                .addMappedIfPresent(Attr.COLWIDTH, colwidth, cw -> Arrays.stream(cw)
                        .boxed()
                        .collect(Collectors.toList()))
                .addIfPresent(Attr.ROWSPAN, rowspan);
        return map(Key.TYPE, elementType())
        // Original code here was: addIf(!attrs.isEmpty(), Key.ATTRS, () -> attrs)
        // To be reverted to when TableParser.OMIT_TABLE_ATTRIBUTES is rolled out.
                .add(Key.ATTRS, attrs)
                .let(this::addContent);
    }

    private static int checkColspan(@Nullable Integer colspan) {
        if (colspan == null) {
            return 1;
        }
        if (colspan <= 0) {
            throw new TableCellException.ColspanNotPositive(colspan);
        }
        return colspan;
    }

    private static void checkColspanAndColwidth(@Nullable Integer colspan, @Nullable int[] colwidth) {
        int cols = checkColspan(colspan);
        if (colwidth == null) {
            return;
        }

        if (colwidth.length != cols) {
            throw new TableCellException.ColwidthDoesNotMatchColspan(colspan, colwidth);
        }

        boolean allZero = true;
        for (int i = 0; i < colwidth.length; ++i) {
            if (colwidth[i] < 0) {
                throw new TableCellException.ColwidthCannotBeNegative(i, colwidth[i]);
            }
            if (colwidth[i] > 0) {
                allZero = false;
            }
        }

        if (allZero) {
            throw new TableCellException.ColwidthMustHaveAtLeastOnePositiveValue();
        }
    }

    protected N parseTableNode(Map<String, ?> map) {
        parseRequiredContent(map, TableCellContent.class);
        this.background = getAttr(map, Attr.BACKGROUND, String.class).orElse(null);
        getAttrInt(map, Attr.ROWSPAN).ifPresent(this::rowspan);
        Integer colspan = getAttrInt(map, Attr.COLSPAN).orElse(null);
        int[] colwidth = parseColwidth(map);
        if (colspan != null || colwidth != null) {
            colspanAndColwidth(colspan, colwidth);
        }
        requireNotEmpty();
        return self();
    }

    @Nullable
    private int[] parseColwidth(Map<String, ?> map) {
        Object attr = getAttr(map, Attr.COLWIDTH).orElse(null);
        if (attr == null || attr instanceof int[]) {
            return (int[]) attr;
        }

        List<? extends Number> list;
        if (attr instanceof Number[]) {
            list = Arrays.stream((Number[]) attr).collect(Collectors.toList());
        } else if (attr instanceof List<?>) {
            list = unsafeCast(attr);
        } else {
            throw new AdfException.ValueTypeMismatch("colwidth", "int[]", attr.getClass().getSimpleName());
        }

        boolean allZero = true;
        int[] result = new int[list.size()];
        for (int i = 0; i < list.size(); ++i) {
            int value = asInt(list.get(i), "colwidth[" + i + ']');
            if (value < 0) {
                throw new TableCellException.ColwidthCannotBeNegative(i, value);
            } else if (value > 0) {
                allZero = false;
            }
            result[i] = value;
        }
        if (allZero) {
            throw new TableCellException.ColwidthMustHaveAtLeastOnePositiveValue();
        }
        return result;
    }

    @Override
    protected void validateContentNodeForAppend(TableCellContent node) {
        if (node instanceof CodeBlock) {
            ((CodeBlock) node).disableMarks(this);
        } else if (node instanceof Paragraph) {
            ((Paragraph) node).disableIndentation(this);
        } else if (node instanceof NestedExpand) {
            ((NestedExpand) node).disableMarks(this);
        }
    }

    @Override
    protected final boolean contentNodeEquals(N other) {
        return Objects.equals(background, other.background)
                && Objects.equals(colspan, other.colspan)
                && Arrays.equals(colwidth, other.colwidth)
                && Objects.equals(rowspan, other.rowspan);
    }

    @Override
    protected final int contentNodeHashCode() {
        return Objects.hash(background, colspan, rowspan, Arrays.hashCode(colwidth));
    }

    @Override
    protected final void appendContentNodeFields(ToStringHelper buf) {
        buf.appendField("background", background);
        buf.appendField("colspan", colspan);
        buf.appendField("colwidth", colwidth);
        buf.appendField("rowspan", rowspan);
    }

    private static void checkRowspan(@Nullable Integer rowspan) {
        if (rowspan != null && rowspan <= 0) {
            throw new TableCellException.RowspanNegative(rowspan);
        }
    }

    @Override
    public final void appendPlainText(StringBuilder sb) {
        appendPlainTextContentJoinedWith('\n', sb);
    }
}
