package com.atlassian.adf.model.node;

import com.atlassian.adf.model.Element;
import com.atlassian.adf.model.ex.AdfException;
import com.atlassian.adf.model.ex.node.ContentException;
import com.atlassian.adf.model.node.type.ContentNode;
import com.atlassian.adf.util.FieldMap;
import com.atlassian.annotations.Internal;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static com.atlassian.adf.model.ex.AdfException.frame;
import static com.atlassian.adf.model.node.NodeParserSupport.getNodeOfType;
import static com.atlassian.adf.util.Cast.unsafeCast;
import static com.atlassian.adf.util.Functions.iterateJoined;
import static com.atlassian.adf.util.Functions.mapToList;
import static com.atlassian.adf.util.ParserSupport.get;
import static java.util.Collections.singleton;
import static java.util.Collections.unmodifiableList;
import static java.util.Objects.requireNonNull;

/**
 * Nodes that use the {@link Node.Key#CONTENT "content"} key to specify a list of child nodes extend this type
 * to handle the common conventions across those node types.
 *
 * @param <C> the type of this container node that holds content
 * @param <N> the common superclass or interface for all node types that are permitted as content within this node
 */
@Internal
public abstract class AbstractContentNode<C extends AbstractContentNode<C, N>, N extends Node>
        extends AbstractNode<C>
        implements ContentNode<C, N> {

    protected final List<N> content = new ArrayList<>();

    /**
     * @return {@code true} if this content node is completely empty, which for many such nodes is not valid
     */
    public boolean isEmpty() {
        return content.isEmpty();
    }

    /**
     * Verifies that at least one content element has been provided.
     *
     * @throws ContentException.ContentRequired if this content node is still empty
     */
    protected void requireNotEmpty() {
        if (isEmpty()) {
            throw new ContentException.ContentRequired();
        }
    }

    /**
     * Called on each content node before it is added to validate that the node can be accepted.
     *
     * @param node the node that is about to be added
     */
    protected void validateContentNodeForAppend(N node) {
    }

    @Override
    public final List<N> content() {
        return unmodifiableList(content);
    }

    public final C content(N content) {
        return content(singleton(content).iterator());
    }

    // Arrays.asList(x).iterator() is always type-safe. There is no way to ignore the varargs warning here,
    // even with the @SafeVarargs annotation, due to inconsistencies in the compiler itself. This method is
    // really useful, so we are suppressing the warnings instead. This is justified because:
    //  * This implementation is safe and `final`.
    //  * There shouldn't really be any need for anyone to implement the parent interface directly.
    //  * If they did, they would still get a varargs warning of their very own to contend with, anyway.
    @SafeVarargs
    @SuppressWarnings("varargs")
    public final C content(N... content) {
        requireNonNull(content, "content");
        return content(Arrays.asList(content).iterator());
    }

    public final C content(Iterable<? extends N> content) {
        requireNonNull(content, "content");
        return content(content.iterator());
    }

    @Override
    public final C content(Stream<? extends N> content) {
        requireNonNull(content, "content");
        return content(content.iterator());
    }

    private C content(Iterator<? extends N> iter) {
        AdfException ex = null;
        int index = this.content.size();
        while (iter.hasNext()) {
            N node = iter.next();
            try {
                requireNonNull(node, "content item");
                validateContentNodeForAppend(node);
                this.content.add(node);
            } catch (AdfException e) {
                ex = fixException(ex, e, index);
            } catch (RuntimeException e) {
                ex = fixException(ex, new AdfException.UnexpectedRuntimeException(e), index);
            }

            ++index;
        }

        if (ex != null) {
            throw ex;
        }

        return self();
    }

    private static AdfException fixException(
            @Nullable AdfException originalException,
            AdfException newException,
            int index
    ) {
        newException.backtrace("[" + index + ']');
        if (originalException == null) {
            return newException;
        }
        originalException.addSuppressed(newException);
        return originalException;
    }

    @Override
    public Stream<Node> allNodes() {
        return Stream.concat(Stream.of(this), content.stream().flatMap(node -> {
            if (node instanceof ContentNode<?, ?>) return ((ContentNode<?, ?>) node).allNodes();
            if (node instanceof MediaSingle) return ((MediaSingle) node).allNodes();
            return Stream.of(node);
        }));
    }

    @Override
    public C clear() {
        content.clear();
        return self();
    }


    @Override
    public void removeIf(Predicate<? super N> predicate) {
        List<N> newContent = new ArrayList<>(content);
        if (newContent.removeIf(predicate)) {
            replaceContent(newContent);
        }
    }

    @Override
    public void transformContent(Function<? super N, ? extends N> transformer) {
        List<N> newContent = new ArrayList<>(content.size());
        boolean modified = false;

        for (N before : content) {
            N after = transformer.apply(before);
            if (after != before) {
                modified = true;
            }
            if (after != null) {
                newContent.add(after);
            }
        }

        if (modified) {
            replaceContent(newContent);
        }
    }

    // Unfortunately, `MediaSingle` necessarily complicates how all of this works. The rules:
    //
    //  1. It might satisfy `targetNodeClass`. If so, the `transformer` is applied to it directly through the
    //     normal path.
    //  2. Even though it doesn't implement `ContentNode`, it does contain 1 or 2 other nodes, and we need to
    //     give the transform the opportunity to be applied to them, too. That is covered by the helper
    //     method `MediaSingle.transformDescendants`.
    //  3. That method may return `true` to indicate that the entire `mediaSingle` node should be removed,
    //     so that has to be rechecked.

    @Override
    public <T extends Node> void transformDescendants(
            Class<T> targetNodeClass,
            Function<? super T, ? extends T> transformer
    ) {
        List<N> newContent = new ArrayList<>();
        boolean modified = false;

        for (N before : content) {
            N after = before;

            if (targetNodeClass.isInstance(before)) {
                after = unsafeCast(transformer.apply(targetNodeClass.cast(before)));
                if (after != before) modified = true;
            }

            if (after instanceof MediaSingle) {
                boolean removeIt = ((MediaSingle) after).transformDescendantsInternal(targetNodeClass, transformer);
                if (removeIt) {
                    after = null;
                    modified = true;
                }
            }

            if (after != null) {
                newContent.add(after);
                if (after instanceof ContentNode<?, ?>) {
                    ((ContentNode<?, ?>) after).transformDescendants(targetNodeClass, transformer);
                }
            }
        }

        if (modified) {
            replaceContent(newContent);
        }
    }

    public final void replaceContent(List<? extends N> newContent) {
        List<N> backup = new ArrayList<>(content);
        boolean ok = false;
        try {
            content.clear();
            content(newContent);
            ok = true;
        } finally {
            if (!ok) {
                content.clear();
                content.addAll(backup);
            }
        }
    }

    protected List<? extends Map<String, ?>> contentFieldMaps() {
        return mapToList(content, Element::toMap);
    }

    @Override
    public final void validate() {
        validateContentItems();
        contentNodeValidate();
    }

    protected void contentNodeValidate() {
    }

    /**
     * Validates that each of the content nodes is in a valid state.
     */
    protected final void validateContentItems() {
        content.forEach(Node::validate);
    }

    protected final void addContent(FieldMap map) {
        map.put(Key.CONTENT, contentFieldMaps());
    }

    protected final void addContentIfPresent(FieldMap map) {
        if (!content.isEmpty()) {
            map.put(Key.CONTENT, contentFieldMaps());
        }
    }

    /**
     * Allows content nodes that have their own fields to augment the {@code hashCode} implementation with
     * a hash of their own field values.
     * <p>
     * Implementations need not include the node's class; that is already covered by {@link AbstractNode#hashCode()}.
     * Implementations need not include the {@code content}, either; that is already covered by
     * {@link AbstractNode#nodeHashCode()}, which is expected to be this method's only consumer.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractContentNode} should maintain consistent
     * implementations of {@code contentNodeHashCode}, {@code contentNodeEquals}, and
     * {@code appendContentNodeFields}.
     *
     * @return the hash code of any additional field values that belong to a particular type of content node.
     * @see #contentNodeEquals(AbstractContentNode)
     * @see #appendContentNodeFields(ToStringHelper)
     */
    protected int contentNodeHashCode() {
        return 0;
    }

    /**
     * Allows content nodes that have their own fields to augment the {@code equals} implementation with
     * tests for their own field values.
     * <p>
     * Implementations need not check for identity, {@code null}, or a different node class;
     * those are already covered by {@link AbstractNode#equals(Object)}.
     * Implementations need not check the {@link #content()}, either; that is already covered by
     * {@link #nodeEquals(AbstractContentNode)}, which is expected to be this method's only consumer.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractContentNode} should maintain consistent
     * implementations of {@code contentNodeHashCode}, {@code contentNodeEquals}, and
     * {@code appendContentNodeFields}.
     *
     * @return {@code true} if all additional field values that belong to a particular type of content node
     * test as equal; {@code false} if differences are found
     * @see #contentNodeHashCode()
     * @see #appendContentNodeFields(ToStringHelper)
     */
    protected boolean contentNodeEquals(C other) {
        return true;
    }

    /**
     * Allows content nodes that have their own fields to augment the {@code toString()} implementation with
     * their own field values.
     * <p>
     * Each field's value should be provided by calling {@link ToStringHelper#appendField(String, Object)}.
     * The {@code value} may be {@code null}, in which case the field is omitted, for brevity. It will
     * handle array values gracefully, including arrays of primitive types.
     * <p>
     * Just as with the relationship between {@code hashCode}, {@code equals}, and {@code toString} for
     * ordinary Java classes, subclasses of {@code AbstractContentNode} should maintain consistent
     * implementations of {@code contentNodeHashCode}, {@code contentNodeEquals}, and
     * {@code appendContentNodeFields}.
     *
     * @param buf where the field values should be written
     * @see #contentNodeHashCode()
     * @see #contentNodeEquals(AbstractContentNode)
     */
    protected void appendContentNodeFields(ToStringHelper buf) {
    }

    /**
     * Implementation of {@code nodeHashCode()} that combines a hash of the {@link #content()}
     * with the value from {@link #contentNodeHashCode()}.
     *
     * @see #contentNodeHashCode()
     */
    @Override
    protected final int nodeHashCode() {
        return contentNodeHashCode() * 31
                + content.hashCode();
    }

    @Override
    protected final boolean nodeEquals(C other) {
        return contentNodeEquals(other)
                && content.equals(other.content);
    }

    @Override
    protected final void appendNodeFields(ToStringHelper buf) {
        appendContentNodeFields(buf);
        buf.appendContentField(content);
    }

    protected C parseOptionalContent(Map<String, ?> map, Class<N> contentClass) {
        getOptionalContentMaps(map)
                .ifPresent(list -> parseContentItems(list, contentClass));
        return self();
    }

    protected C parseRequiredContentAllowEmpty(Map<String, ?> map, Class<N> contentClass) {
        List<Map<String, ?>> list = getRequiredContentMapsAllowEmpty(map);
        if (!list.isEmpty()) {
            parseContentItems(list, contentClass);
        }
        return self();
    }

    protected C parseRequiredContent(Map<String, ?> map, Class<N> contentClass) {
        List<Map<String, ?>> list = getRequiredContentMaps(map);
        parseContentItems(list, contentClass);
        return self();
    }

    protected void parseContentItems(List<Map<String, ?>> maps, Class<N> contentClass) {
        List<N> content = new ArrayList<>(maps.size());
        int i = 0;
        for (Map<String, ?> map : maps) {
            frame(
                    "[" + i + ']',
                    () -> content.add(parseContentItem(map, contentClass))
            );
            ++i;
        }
        content(content);
    }

    protected N parseContentItem(Map<String, ?> map, Class<N> contentClass) {
        return getNodeOfType(contentClass, map, this);
    }

    static Optional<List<Map<String, ?>>> getOptionalContentMaps(Map<String, ?> map) {
        return Optional.ofNullable(get(map, Key.CONTENT));
    }

    static List<Map<String, ?>> getRequiredContentMaps(Map<String, ?> map) {
        List<Map<String, ?>> list = get(map, Key.CONTENT);
        if (list == null || list.isEmpty()) {
            throw new ContentException.ContentRequired();
        }
        return list;
    }

    static List<Map<String, ?>> getRequiredContentMapsAllowEmpty(Map<String, ?> map) {
        List<Map<String, ?>> list = get(map, Key.CONTENT);
        if (list == null) {
            throw new ContentException.FieldRequired();
        }
        return list;
    }

    @Override
    public void appendPlainText(StringBuilder sb) {
        appendPlainTextContent(sb, this);
    }

    /**
     * Intended for content nodes that do not need to add any structure to their plain-text output, this
     * renders each content item in turn with no special handling. If the {@code contentNode} is empty,
     * then this call has no effect.
     *
     * @param sb          the buffer to which the contents should be rendered as plain-text
     * @param contentNode the content node to be rendered as plain-text
     */
    static void appendPlainTextContent(StringBuilder sb, ContentNode<?, ?> contentNode) {
        contentNode.content().forEach(child -> child.appendPlainText(sb));
    }

    /**
     * Intended for content nodes that hold block elements, this will render their contents as plain-text
     * by joining the items with the given {@code delimiter} between them. If this node is empty, then this
     * call has no effect.
     *
     * @param delimiter the text (commonly {@code "\n"}, a newline) to use as a separator between content items
     * @param sb        the buffer to which the contents should be rendered as plain-text
     */
    protected void appendPlainTextContentJoinedWith(char delimiter, StringBuilder sb) {
        iterateJoined(
                content,
                child -> child.appendPlainText(sb),
                () -> sb.append(delimiter)
        );
    }

    /**
     * For content nodes that hold {@code InlineContent} (or some other marker interface like
     * {@code CaptionContent} that has a similar purpose), append that content to the given
     * buffer with appropriate handling of mention nodes.
     * <p>
     * Briefly, this means that if a mention node is present in the content, then a space is
     * added between it and the following node so that the mention name is not encroached upon
     * by the following text. We want the mention {@code @foo} followed by the text {@code bar}
     * to be {@code @foo bar}, not {@code @foobar}.
     *
     * @param sb the buffer to which the inline content (or similar) should be rendered as plain-text
     */
    protected void appendPlainTextInlineContent(StringBuilder sb) {
        int start = sb.length();
        boolean previousNodeWasMention = false;
        for (Node child : content) {
            int mark = sb.length();
            child.appendPlainText(sb);
            if (previousNodeWasMention) {
                if (sb.length() == mark || sb.charAt(mark) != ' ') {
                    sb.insert(mark, ' ');
                }
            }
            previousNodeWasMention = child instanceof Mention;
        }

        trim(sb, start);
    }

    /**
     * Trims a substring directly in the string builder's buffer to avoid creating an intermediate String,
     * trimming that, and writing it back to the original buffer.
     *
     * @param sb    the buffer containing the substring to be trimmed
     * @param start the beginning of the substring to be trimmed
     */
    protected void trim(StringBuilder sb, int start) {
        int pos = start;
        int end = sb.length();

        // Strip leading spaces (defined as all chars < 0x20 as for String.trim()) from the new buffer content
        while (pos < end && sb.charAt(pos) <= ' ') ++pos;
        if (pos > start) sb.delete(start, pos);

        // If that got everything, then there's nothing left to look at
        if (pos == end) return;

        // Otherwise see whether we need to strip anythiing from the end of the buffer
        pos = sb.length() - 1;
        if (sb.charAt(pos) > ' ') return;

        // If we do, then truncate the buffer appropriately
        --pos;
        while (pos > start && sb.charAt(pos) <= ' ') --pos;
        sb.setLength(pos + 1);
    }
}
