package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.mark.TextColorException;
import com.atlassian.adf.model.mark.type.FormattedTextMark;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.Text;
import com.atlassian.adf.model.node.type.InlineContent;
import com.atlassian.adf.util.Colors;
import com.atlassian.adf.util.Factory;

import java.awt.*;
import java.util.Map;
import java.util.regex.Pattern;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrOrThrow;
import static java.util.Objects.requireNonNull;

/**
 * The {@code textColor} mark adds {@code <span style="color: #ff0000;">...</span>} color styling.
 * This mark applies to {@link Text} nodes and can only be combined with other
 * {@link FormattedTextMark formatted-text} marks. As an additional restriction, it cannot be
 * combined with a {@link Link link} mark.
 * <h2>Rendering colors with theme support</h2>
 * Colors are stored in a hexadecimal format, but need to be displayed differently depending on
 * the current product theme, such as light and dark mode. Support for this is provided via a
 * mapping from each hexadecimal color to a design token from the Atlassian Design System.
 * <p>
 * Currently, such theming is only supported in the web user interface, where it is provided by
 * the {@code @atlaskit/editor-palette} package.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre>
 * {@link Paragraph#p(InlineContent[]) p}(
 *         {@link Text#text(String) text}("Hello "),
 *         {@link Text#text(String) text}("world").{@link Text#textColor(String)} textColor}("#97a0af"),
 *         {@link Text#text(String) text}("!")
 * );
 * </pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "paragraph",
 *     "content": [
 *       {
 *         "type": "text",
 *         "text": "Hello "
 *       },
 *       {
 *         "type": "text",
 *         "text": "world",
 *         "marks": [
 *           {
 *             "type": "textColor",
 *             "attrs": {
 *                 "color": "#97a0af"
 *             }
 *           }
 *         ]
 *       },
 *       {
 *         "type": "text",
 *         "text": "!"
 *       }
 *     ]
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Hello <span style="color: #97a0af;">world</span>!</p>
 * </div>
 *
 * @see <a href="https://atlaskit.atlassian.com/packages/editor/editor-palette">@atlaskit/editor-palette</a>
 * @see <a href="https://atlassian.design/components/tokens">Atlassian Design System &mdash; Tokens</a>
 * @see <a href="https://developer.atlassian.com/cloud/jira/platform/apis/document/marks/textColor/">Mark - textColor</a>
 */
@Documentation(state = Documentation.State.REVIEWED, date = "2023-07-26")
public class TextColor
        extends AbstractMark
        implements FormattedTextMark {

    private static final Pattern REGEX_VALID_COLOR = Pattern.compile("^#[0-9a-fA-F]{6}$");
    static final Factory<TextColor> FACTORY = new Factory<>(Type.TEXT_COLOR, TextColor.class, TextColor::parse);

    private String color;

    private TextColor(String color) {
        this.color = validateColor(color);
    }

    /**
     * @param color a color defined in HTML hexadecimal format, for example
     *              <span style="color: #daa520;">{@code #daa520}</span>.
     *              To display this color with correct contrast in different product themes,
     *              such as light and dark mode, use {@code @atlaskit/editor-palette} to map this color to
     *              a design token from the Atlassian Design System.
     * @return a {@code textColor} mark for the given color
     */
    public static TextColor textColor(String color) {
        return new TextColor(color);
    }

    public static TextColor textColor(Color color) {
        return textColor(Colors.hex6(color));
    }

    public static TextColor textColor(Colors.Named color) {
        return textColor(color.toHex());
    }

    @Override
    public TextColor copy() {
        return parse(toMap());
    }

    public String color() {
        return color;
    }

    public TextColor color(String color) {
        this.color = validateColor(color);
        return this;
    }

    public TextColor color(Color color) {
        return textColor(Colors.hex6(color));
    }

    @Override
    public String elementType() {
        return Type.TEXT_COLOR;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.COLOR, color));
    }

    @Override
    public boolean equals(Object o) {
        return this == o || (o instanceof TextColor && color.equals(((TextColor) o).color));
    }

    @Override
    public int hashCode() {
        return color.hashCode();
    }

    @Override
    public String toString() {
        return elementType() + '[' + color + ']';
    }

    private static TextColor parse(Map<String, ?> map) {
        checkType(map, Type.TEXT_COLOR);
        String color = getAttrOrThrow(map, Attr.COLOR);
        return textColor(color);
    }

    private static String validateColor(String color) {
        requireNonNull(color, "color");
        if (!REGEX_VALID_COLOR.matcher(color).matches()) {
            throw new TextColorException.InvalidColor(color);
        }
        return color;
    }
}
