package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.mark.IndentationException;
import com.atlassian.adf.model.mark.type.HeadingMark;
import com.atlassian.adf.model.mark.type.ParagraphMark;
import com.atlassian.adf.model.mark.type.PositionMark;
import com.atlassian.adf.model.node.Doc;
import com.atlassian.adf.model.node.Heading;
import com.atlassian.adf.model.node.Paragraph;
import com.atlassian.adf.model.node.type.DocContent;
import com.atlassian.adf.util.Factory;

import javax.annotation.concurrent.Immutable;
import java.util.Map;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.checkType;
import static com.atlassian.adf.util.ParserSupport.getAttrIntOrThrow;

/**
 * The {@code indentation} mark sets the level of indentation for a node, up to {@code 6}.
 * This mark applies to {@link Heading heading} and {@link Paragraph paragraph} nodes.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 *     {@link Doc#doc(DocContent[]) doc}(
 *         {@link Paragraph#p(String) p}("Normal"),
 *         {@link Paragraph#p(String) p}("Indented 1 level")
 *                 .{@link Paragraph#indentation(int) indentation}(1),
 *         {@link Paragraph#p(String) p}("Indented 2 levels")
 *                 .{@link Paragraph#mark(ParagraphMark) mark}({@link Indentation#indentation(int) indentation}(2))
 *         {@link Paragraph#p(String) p}("Normal again"),
 *     )
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 * {
 *    "type": "doc",
 *    "version": 1,
 *    "content": [
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Normal"
 *          }
 *        ]
 *      },
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Indented 1 level"
 *          }
 *        ],
 *        "marks": [
 *          {
 *            "type": "indentation",
 *            "attrs": {
 *              "level": 1
 *            }
 *          }
 *        ]
 *      },
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Indented 2 levels"
 *          }
 *        ],
 *        "marks": [
 *          {
 *            "type": "indentation",
 *            "attrs": {
 *              "level": 2
 *            }
 *          }
 *        ]
 *      },
 *      {
 *        "type": "paragraph",
 *        "content": [
 *          {
 *            "type": "text",
 *            "text": "Normal again"
 *          }
 *        ]
 *      }
 *    ]
 *  }
 * }</pre>
 * <h3>Result</h3>
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <p>Normal</p>
 * <p style="margin-left: 30px;">Indented 1 level</p>
 * <p style="margin-left: 60px;">Indented 2 levels</p>
 * <p>Normal again</p>
 * </div>
 */
@Immutable
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Indentation
        extends AbstractMark
        implements HeadingMark, ParagraphMark, PositionMark {

    static final Factory<Indentation> FACTORY = new Factory<>(Type.INDENTATION, Indentation.class, Indentation::parse);

    private final int level;

    private Indentation(int level) {
        if (level < 1 || level > 6) {
            throw new IndentationException.InvalidLevel(level);
        }
        this.level = level;
    }

    public static Indentation indentation(int level) {
        return new Indentation(level);
    }

    @Override
    public Indentation copy() {
        return this;
    }

    public int level() {
        return level;
    }

    @Override
    public boolean equals(Object o) {
        return this == o || (o instanceof Indentation && ((Indentation) o).level == level);
    }

    @Override
    public int hashCode() {
        return level;
    }

    @Override
    public String elementType() {
        return Type.INDENTATION;
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map(Attr.LEVEL, level));
    }

    @Override
    public String toString() {
        return elementType() + '[' + level + ']';
    }

    private static Indentation parse(Map<String, ?> map) {
        checkType(map, Type.INDENTATION);
        int level = getAttrIntOrThrow(map, Attr.LEVEL);
        return indentation(level);
    }
}