package com.atlassian.adf.model.mark;

import com.atlassian.adf.model.Documentation;
import com.atlassian.adf.model.ex.mark.BorderException;
import com.atlassian.adf.model.mark.type.MediaMark;
import com.atlassian.adf.model.node.Media;
import com.atlassian.adf.util.Char;
import com.atlassian.adf.util.Colors;
import com.atlassian.adf.util.Factory;

import java.awt.*;
import java.util.Map;
import java.util.Objects;

import static com.atlassian.adf.util.FieldMap.map;
import static com.atlassian.adf.util.ParserSupport.*;
import static java.util.Objects.requireNonNull;

/**
 * The {@code border} mark sets border styling. This mark applies to {@link Media media} nodes.
 * This mark applies to {@code media} nodes and indicates that the image should be decorated with a border
 * with the given {@code color} and {@code size} in pixels.
 * <p>
 * The {@code color} should be given as a 6- or 8-digit hexadecimal value prefixed with a hash mark; for example
 * <span style="color: #ff00cc">{@code #ff00cc}</span> or <span style="color: #00ffcc24">{@code #00ffcc24}</span>.
 * Where the 8-digit form is used, the final 2 hex digits represent the value for the alpha channel, which
 * controls the color's opacity.
 * <h2>Example</h2>
 * <h3>Java</h3>
 * <pre><code>
 * Media.{@link Media#fileMedia(String, String) fileMedia}(
 *         "6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5",
 *         "my project files"
 * ).{@link Media#border(String) border}("#ff00cc").{@link Media#size(Number, Number) size}(200, 183);
 * </code></pre>
 * <h3>ADF</h3>
 * <pre>{@code
 *   {
 *     "type": "media",
 *     "attrs": {
 *       "id": "6e7c7f2c-dd7a-499c-bceb-6f32bfbf30b5",
 *       "collection": "my project files",
 *       "type": "file",
 *       "width": 200,
 *       "height": 183
 *     }
 *   }
 * }</pre>
 * <h3>Result</h3>
 * <!-- Note: IntelliJ's JavaDocs viewer does not support this, so you won't see the border there -->
 * <div style="color: rgb(23, 43, 77); background-color: #ffffff;">
 * <img style="border: 2px solid #ff00cc;" width=200 height=183
 * src="data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAgAAAAIAQMAAAD+wSzIAAAABlBMVEX///+/v7+jQ3Y5AAAADklEQVQI12P4AIX8EAgALgAD/aNpbtEAAAAASUVORK5CYII"/>
 * <p>
 * </div>
 * Note: This example uses a data URI to embed the image instead of retrieving the image from media
 * services, which uses a different mechanism to supply the image content.
 */
@Documentation(state = Documentation.State.UNDOCUMENTED, date = "2023-07-26")
public class Border
        extends AbstractMark
        implements MediaMark {

    static final Factory<Border> FACTORY = new Factory<>(Type.BORDER, Border.class, Border::parse);

    /**
     * The default size, in pixels, that will be used for the border if it is not explicitly given.
     */
    public static final int DEFAULT_SIZE = 2;

    /**
     * The minimum size, in pixels, that may be used for a border.
     */
    public static final int MINIMUM_SIZE = 1;

    /**
     * The maximum size, in pixels, that may be used for a border.
     */
    public static final int MAXIMUM_SIZE = 3;

    /**
     * The default color that will be used for the border if it is not explicitly given.
     */
    public static final String DEFAULT_COLOR = "#091E4224";

    private int size;
    private String color;

    private Border(int size, String color) {
        this.size = size;
        this.color = color;
    }

    public static Border border() {
        return new Border(DEFAULT_SIZE, DEFAULT_COLOR);
    }

    public static Border border(String color) {
        return new Border(DEFAULT_SIZE, validateColor(color));
    }

    public static Border border(Color color) {
        return new Border(DEFAULT_SIZE, Colors.hex(color));
    }

    public static Border border(Colors.Named color) {
        return new Border(DEFAULT_SIZE, color.toHex());
    }

    public int size() {
        return size;
    }

    public Border size(int size) {
        this.size = validateSize(size);
        return this;
    }

    public String color() {
        return color;
    }

    public Border color(String color) {
        this.color = validateColor(color);
        return this;
    }

    public Border color(Color color) {
        this.color = Colors.hex(color);
        return this;
    }

    public Border color(Colors.Named color) {
        this.color = color.toHex();
        return this;
    }

    @Override
    public String elementType() {
        return Type.BORDER;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Border)) return false;
        Border other = (Border) o;
        return size == other.size && color.equals(other.color);
    }

    @Override
    public int hashCode() {
        return Objects.hash(size, color);
    }

    @Override
    public String toString() {
        return elementType() + "[size=" + size + ", color=" + color + ']';
    }

    @Override
    public Map<String, ?> toMap() {
        return mapWithType()
                .add(Key.ATTRS, map()
                        .add(Attr.SIZE, size)
                        .add(Attr.COLOR, color)
                );
    }

    private static Border parse(Map<String, ?> map) {
        checkType(map, Type.BORDER);
        String color = getAttrOrThrow(map, Attr.COLOR);
        int size = validateSize(getAttrIntOrThrow(map, Attr.SIZE));
        return new Border(size, color);
    }

    private static int validateSize(int size) {
        if (size >= MINIMUM_SIZE && size <= MAXIMUM_SIZE) return size;
        throw new BorderException.InvalidSize(size);
    }

    private static String validateColor(String color) {
        requireNonNull(color, Attr.COLOR);
        if (isValidColor(color)) return color;
        throw new BorderException.InvalidColor(color);
    }

    private static boolean isValidColor(String color) {
        int len = color.length();
        if (len != 7 && len != 9) return false;
        if (color.charAt(0) != '#') return false;
        for (int i = 1; i < len; ++i) {
            if (!Char.isHexit(color.charAt(i))) return false;
        }
        return true;
    }

    @Override
    public Border copy() {
        return parse(toMap());
    }
}
