package com.atlassian.adf.model.ex;

import javax.annotation.Nullable;
import java.net.URISyntaxException;
import java.util.Deque;
import java.util.LinkedList;
import java.util.Optional;
import java.util.function.Supplier;

import static com.atlassian.adf.util.Cast.unsafeCast;
import static java.util.Objects.requireNonNull;

/**
 * An exception reporting a problem with the structure of an ADF document.
 */
public abstract class AdfException extends RuntimeException {
    private static final long serialVersionUID = 1L;

    /**
     * Used in exception backtraces to introduce the node's {@code type} after any preceding information
     * about where the node was found.
     * <p>
     * <strong>Example</strong>: <code>doc[0].paragraph[0].text:marks[1].code</code><br>
     * The periods (<code>.</code>) before {@code paragraph} and {@code text} are node prefixes.
     */
    public static final String NODE_PREFIX = ".";

    /**
     * Used in exception backtraces to introduce the mark's {@code type} after any preceding information
     * about where the mark was found.
     * <p>
     * <strong>Example</strong>: <code>doc[0].paragraph[0].text:marks[1].code</code><br>
     * The period (<code>.</code>) before {@code code} is a mark prefix.
     */
    public static final String MARK_PREFIX = ".";

    /**
     * Used in exception backtraces to indicate that the problem occurs within a node's {@code marks} fields
     * rather than elsewhere in the node's information.
     * <p>
     * <strong>Example</strong>: <code>doc[0].paragraph[0].text:marks[1].code</code><br>
     * The <code>:marks</code> portion indicates that the {@code text} node's marks were at fault, and
     * specifically the one at index {@code [1]}, meaning the second mark.
     */
    public static final String MARKS_PREFIX = ":marks";

    private final Deque<String> adfBacktrace = new LinkedList<>();

    protected AdfException(String message) {
        super(message);
    }

    protected AdfException(String message, @Nullable Throwable cause) {
        super(message, cause);
    }

    /**
     * Wraps code that calculates a value with a catch block that will add the specified {@code frame}
     * to the {@link #backtrace(String)} if that code throws a runtime exception.
     *
     * @param frame    the frame information to be added to the backtrace if an exception is thrown
     * @param supplier the code that generates the requested value
     * @param <T>      the inferred return type of the {@code supplier}
     * @return the supplied value
     * @throws AdfException with {@code frame} added to the backtrace if any {@code RuntimeException} is thrown by
     *                      the {@code supplier}
     */
    public static <T> T frame(String frame, Supplier<T> supplier) {
        try {
            return supplier.get();
        } catch (AdfException e) {
            throw e.backtrace(frame);
        } catch (RuntimeException e) {
            throw new AdfException.UnexpectedRuntimeException(e).backtrace(frame);
        }
    }

    /**
     * Adds the given {@code frame} to the backtrace for this exception.
     * The backtrace is used to provide the {@link #getPath() path} when rendering an {@code AdfException}.
     *
     * @param frame the frame information to add to this exception's backtrace
     * @return {@code this}
     */
    public final AdfException backtrace(String frame) {
        synchronized (adfBacktrace) {
            adfBacktrace.addFirst(requireNonNull(frame, "frame"));
            return this;
        }
    }

    /**
     * Returns the current path information for this exception, if it is available.
     * The path is formed by concatenating all of the {@code frame} values that were provided by
     * {@link #frame(String, Supplier)} or {@link #backtrace(String)} calls, in reverse order.
     *
     * @return the path information for this exception, or {@code null} if none has been provided
     */
    @Nullable
    public String getPath() {
        synchronized (adfBacktrace) {
            return adfBacktrace.isEmpty()
                    ? null
                    : String.join("", adfBacktrace);
        }
    }

    @Override
    public String toString() {
        String path = getPath();
        if (path == null) {
            return super.toString();
        }
        return super.toString() + "  (path: " + path + ')';
    }

    /**
     * Used to wrap other runtime exceptions as an {@code AdfException}, primarily so that the
     * {@link #getPath() error path} can potentially be provided for it.
     */
    public static class UnexpectedRuntimeException extends AdfException {
        private static final long serialVersionUID = 1L;

        public UnexpectedRuntimeException(RuntimeException cause) {
            super("Caught unexpected runtime exception: " + cause, requireNonNull(cause, "cause"));
        }

        @Override
        public RuntimeException getCause() {
            return unsafeCast(super.getCause());
        }
    }

    /**
     * Thrown for any node or mark that fails to specify a {@code type} value.
     */
    public static class MissingType extends AdfException {
        private static final long serialVersionUID = 1L;

        public MissingType() {
            super("A 'type' field with a non-empty string value is mandatory for all objects within an ADF document");
        }
    }

    /**
     * Thrown to indicate that there is something wrong with a field (those values placed directly at the top level
     * of a node or mark) or attribute (those placed within the {@code attrs} object).
     */
    public static abstract class PropertyException extends AdfException {
        private static final long serialVersionUID = 1L;

        private final String propertyName;

        PropertyException(String propertyName, String message) {
            super(message);
            this.propertyName = requireNonNull(propertyName, "propertyName");
        }

        PropertyException(String propertyName, String message, @Nullable Throwable cause) {
            super(message, cause);
            this.propertyName = requireNonNull(propertyName, "propertyName");
        }

        /**
         * @return the name of the field or attribute that has a problem
         */
        public String propertyName() {
            return propertyName;
        }
    }

    /**
     * Thrown to indicate that a required field or attribute was not provided.
     * <p>
     * This code does not make an attempt to look for properties that are present in an unexpected location,
     * such as trying to specify the {@code color} attribute at the top level of a {@code textColor} mark
     * when it should be inside the {@code attrs}.
     */
    public static class MissingProperty extends PropertyException {
        private static final long serialVersionUID = 1L;

        public MissingProperty(String propertyName) {
            super(
                    propertyName,
                    "Required field or attribute is missing or in the wrong place: '" + propertyName + '\''
            );
        }
    }

    /**
     * Thrown to indicate that a required string value was specified as an empty string.
     */
    public static class EmptyProperty extends PropertyException {
        private static final long serialVersionUID = 1L;

        public EmptyProperty(String propertyName) {
            super(propertyName, "Field or attribute cannot be empty: '" + propertyName + '\'');
        }
    }

    /**
     * Thrown to indicate the error of attempting to add the same key to the map representation of a
     * node more than one time.
     */
    public static class DuplicateProperty extends PropertyException {
        private static final long serialVersionUID = 1L;

        public DuplicateProperty(String propertyName) {
            super(propertyName, "Duplicate field or attribute: '" + propertyName + '\'');
        }
    }

    /**
     * Thrown to indicate that a particular value could not be extracted from a field or attribute
     * because it was not of the correct type.
     * <p>
     * Note that this exception is only used for a few special cases where explicit checking of the type
     * was required. In most cases, a value of the wrong type will just throw a {@code ClassCastException},
     * instead.
     */
    public static class ValueTypeMismatch extends PropertyException {
        private static final long serialVersionUID = 1L;

        private final String expectedType;
        private final String actualType;

        public ValueTypeMismatch(String propertyName, String expectedType, String actualType) {
            super(propertyName,
                    "Field or attribute '" + propertyName + "' was expected to have type '" + expectedType +
                            "', but the actual value was of type '" + actualType + '\'');
            this.expectedType = requireNonNull(expectedType, "expectedType");
            this.actualType = requireNonNull(actualType, "actualType");
        }

        public String expectedType() {
            return expectedType;
        }

        public String actualType() {
            return actualType;
        }
    }

    /**
     * Thrown when a limited set of enumerated values are permitted, but a value was encountered that is
     * not in the allowed set. For example, {@code alignment} can take an {@code align} value of {@code center},
     * but cannot accept the British spelling {@code centre}. Them's the breaks!
     */
    public static class UnsupportedEnumValue extends AdfException {
        private static final long serialVersionUID = 1L;

        private final String enumName;

        @Nullable
        private final String enumValue;

        public UnsupportedEnumValue(String enumName, @Nullable String enumValue) {
            super("Invalid value supplied for enum '" + enumName + "': " +
                    Optional.ofNullable(enumValue).map(x -> "'" + x + "'").orElse("null"));
            this.enumName = enumName;
            this.enumValue = enumValue;
        }

        public String enumName() {
            return enumName;
        }

        @Nullable
        public String enumValue() {
            return enumValue;
        }
    }

    /**
     * Used as a runtime replacement for {@code URISyntaxException} when a parameter value
     * that must be a valid URI is checked.
     */
    public static class InvalidURI extends PropertyException {
        private static final long serialVersionUID = 1L;

        private final String uri;

        public InvalidURI(String propertyName, String uri) {
            super("Field or attribute '" + propertyName + "' must contain a valid URI", propertyName);
            this.uri = requireNonNull(uri, "uri");
        }

        public InvalidURI(String propertyName, String uri, @Nullable URISyntaxException cause) {
            super("Field or attribute '" + propertyName + "' must contain a valid URI", propertyName, cause);
            this.uri = requireNonNull(uri, "uri");
        }

        /**
         * Returns the malformed URL value that was provided.
         */
        public String uri() {
            return uri;
        }
    }
}
