package com.atlassian.adf.model;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

import static com.atlassian.adf.model.Element.nonNull;
import static com.atlassian.adf.util.Cast.unsafeCast;

/**
 * Represents an ADF element that isn't supported by this library and can only be preserved as-is,
 * not directly manipulated, because we don't know what it is, what it does, or what constraints
 * there might be around its use.
 */
public abstract class UnsupportedElement implements Element {
    private final String type;
    private final Map<String, ?> map;

    // Note: non-thread-safe caching of hash code is permissible because it is deterministic
    // and calculated from effectively-final state.
    private int hashCode;

    protected UnsupportedElement(Map<String, ?> map) {
        this.type = (String) nonNull(map.get(Key.TYPE), "type");
        this.map = immutableDeepCopy(map);
    }

    @Override
    public abstract UnsupportedElement copy();

    @Override
    public String elementType() {
        return type;
    }

    @Override
    public Map<String, ?> toMap() {
        return map;
    }

    @Override
    public final boolean isSupported() {
        return false;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        UnsupportedElement that = (UnsupportedElement) o;
        return hashCode() == that.hashCode() && Objects.equals(map, that.map);
    }

    @Override
    public int hashCode() {
        int h = hashCode;
        if (h == 0) {
            h = map.hashCode();
            hashCode = h;
        }
        return h;
    }

    @Override
    public String toString() {
        return "UnsupportedElement" + map;
    }

    @Override
    public void validate() {
        // No point in validating something that isn't supported anyway
    }

    private static Map<String, ?> immutableDeepCopy(Map<String, ?> map) {
        Map<String, Object> copy = new LinkedHashMap<>(map);
        for (Map.Entry<String, Object> entry : copy.entrySet()) {
            Object value = entry.getValue();
            if (value instanceof Map<?, ?>) {
                entry.setValue(immutableDeepCopy(unsafeCast(value)));
            }
        }

        // Note: Don't use Map.copyOf, because we want to preserve the original iteration order.
        return Collections.unmodifiableMap(copy);
    }
}
