package com.atlassian.adf.model;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.URL;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Schema reader for the JSON schema included in the source.
 */
public class Schema {
    /**
     * The JSON schema version that was current the last time that this library was updated for it.
     */
    private static final String VERSION = "30.2.1";
    private static final int BUFFER_SIZE = 8192;
    private static final String SCHEMA_JSON = "schema.json";

    @Nullable
    private static String schema;

    private Schema() {
        // static-only
    }

    /**
     * The current known JSON schema version, {@value #VERSION}.
     * @return {@value #VERSION}
     */
    public static String version() {
        return VERSION;
    }

    public static URL url() {
        URL url = Schema.class.getClassLoader().getResource(SCHEMA_JSON);
        if (url == null) throw new IllegalStateException("Unable to load schema: File not found");
        return url;
    }

    public static InputStream inputStream() throws IOException {
        return url().openStream();
    }

    public static String schema() {
        String cached = schema;
        if (cached == null) {
            try (
                    InputStream is = inputStream();
                    InputStreamReader isr = new InputStreamReader(is, UTF_8)
            ) {
                StringWriter sw = new StringWriter();
                char[] buf = new char[BUFFER_SIZE];
                while (true) {
                    int len = isr.read(buf);
                    if (len <= 0) break;
                    sw.write(buf, 0, len);
                }
                cached = sw.toString();
                schema = cached;
            } catch (IOException ioe) {
                throw new IllegalStateException("Unable to load schema: " + ioe, ioe);
            }
        }
        return cached;
    }
}
