package com.atlassian.activeobjects.spi;

import com.atlassian.tenancy.api.Tenant;
import com.atlassian.tenancy.api.TenantContext;

import javax.sql.DataSource;

import static java.util.Objects.requireNonNull;

/**
 * Compatibility implementation which calls {@link com.atlassian.activeobjects.spi.TenantAwareDataSourceProvider}
 * with the current tenant.
 *
 * Will throw {@link java.lang.IllegalStateException} if invoked when there is no tenant present.
 */
public class DataSourceProviderImpl implements DataSourceProvider {

    private final TenantAwareDataSourceProvider tenantAwareDataSourceProvider;
    private final TenantContext tenantContext;

    public DataSourceProviderImpl(
            final TenantAwareDataSourceProvider tenantAwareDataSourceProvider, final TenantContext tenantContext) {
        this.tenantAwareDataSourceProvider = requireNonNull(tenantAwareDataSourceProvider);
        this.tenantContext = requireNonNull(tenantContext);
    }

    @Override
    public DataSource getDataSource() {
        final Tenant tenant = tenantContext.getCurrentTenant();
        if (tenant == null) {
            throw new IllegalStateException("tenant / dataSource unavailable");
        }
        return tenantAwareDataSourceProvider.getDataSource(tenant);
    }

    @Override
    public DatabaseType getDatabaseType() {
        final Tenant tenant = tenantContext.getCurrentTenant();
        if (tenant == null) {
            throw new IllegalStateException("tenant / databaseType unavailable");
        }
        return tenantAwareDataSourceProvider.getDatabaseType(tenant);
    }

    @Override
    public String getSchema() {
        final Tenant tenant = tenantContext.getCurrentTenant();
        if (tenant == null) {
            throw new IllegalStateException("tenant / schema unavailable");
        }
        return tenantAwareDataSourceProvider.getSchema(tenant);
    }
}
