package com.appspiriment.baseclasses

import android.annotation.SuppressLint
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.res.Configuration
import android.net.ConnectivityManager
import android.os.Build
import android.os.Bundle
import android.util.DisplayMetrics
import android.view.Menu
import android.view.MenuItem
import android.view.View
import android.widget.LinearLayout
import androidx.appcompat.app.AppCompatActivity
import androidx.databinding.DataBindingUtil
import androidx.databinding.ViewDataBinding
import androidx.lifecycle.LiveData
import androidx.lifecycle.Observer
import com.appspiriment.androidutils.NetworkUtils
import com.appspiriment.androidutils.showMsgDialog
import com.appspiriment.androidutils.showToast
import com.appspiriment.baseclasses.utils.BaseViewStates
import com.appspiriment.baseclasses.utils.UpdateConstants
import com.google.android.gms.ads.AdListener
import com.google.android.gms.ads.AdRequest
import com.google.android.gms.ads.AdSize
import com.google.android.gms.ads.AdView
import com.google.android.material.snackbar.Snackbar
import com.google.android.play.core.appupdate.AppUpdateManager
import com.google.android.play.core.appupdate.AppUpdateManagerFactory
import com.google.android.play.core.install.model.ActivityResult.RESULT_IN_APP_UPDATE_FAILED
import com.google.android.play.core.install.model.AppUpdateType
import com.google.android.play.core.install.model.UpdateAvailability
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel
import kotlin.reflect.KClass


/*********************************************************
 * Class : BaseActivity
 * Purpose : Baseclass for Activities
 *  ******************************************************
 * Rework Details:
 * 1) {Author} :  {Date} : {Details}
 *********************************************************/
@Suppress("MemberVisibilityCanBePrivate")
@SuppressLint("Registered")
abstract class BaseActivity<
        out ViewModelType : BaseViewModel,
        out DataBindingType : ViewDataBinding>(
    viewModelClass: KClass<ViewModelType>,
    protected val layoutId: Int,
    protected val menuId: Int = R.menu.basedefaultmenu,
    protected val adMobBannerId: String? = null,
    protected val toolbarId: Int? = null,
    protected val isChildActivity: Boolean = false,
    protected val isUpdateActivity: Boolean = false
) : AppCompatActivity() {

    /***************************************
     * Declarations
     ***************************************/
    protected var menu: Menu? = null
    protected val viewModel by viewModel<ViewModelType>(viewModelClass)
    protected var isNetworkConnected = true
    protected val networkListener by lazy {
        object : BroadcastReceiver() {
            override fun onReceive(context: Context?, intent: Intent?) {
                NetworkUtils.isNetworkConnected(applicationContext).let {
                    if (isNetworkConnected != it) {
                        isNetworkConnected = it
                        initializeAdView()
                        onNetworkChanged()
                    }
                }
            }
        }
    }

    protected val intentFilter by lazy {
        IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION).apply {
            addAction(Intent.ACTION_AIRPLANE_MODE_CHANGED)
        }
    }

    protected val binding by lazy {
        DataBindingUtil.setContentView(
            this,
            layoutId
        ) as DataBindingType
    }

    protected val appUpdateManager: AppUpdateManager? by lazy {
        AppUpdateManagerFactory.create(applicationContext)
    }

    /***************************************
     * OnCreate
     ***************************************/
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        initializeBeforeBinding()

        binding.setVariable(BR.viewModel, viewModel)
        binding.lifecycleOwner = this

        if (toolbarId != null) {
            setSupportActionBar(findViewById(toolbarId))
            supportActionBar?.setDisplayHomeAsUpEnabled(isChildActivity)
            supportActionBar?.setHomeButtonEnabled(isChildActivity)
        }

        initializeViews()

        if (isUpdateActivity) updateAppIfAvailable()

        setBaseObservers()

        initializeAdView()

        initializationAfterObserving()

    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    override fun onCreateOptionsMenu(menu: Menu?): Boolean {
        menuInflater.inflate(menuId, menu)
        this.menu = menu
        return true
    }

    /**
     * *************************************
     * OnOptionsMenu Item Selected Method
     * **************************************
     */
    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        return if (item.itemId == android.R.id.home) {
            if (isHandleHomeButton())
                handleHomeButton()
            else
                super.onOptionsItemSelected(item)
        } else {
            handleOptionsItemClick(item.itemId)
        }
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    open fun initializeAdView() {
        findViewById<LinearLayout>(R.id.adView)?.let {
            it.visibility = if (isNetworkConnected) {
                populateAdView(it)
            } else {
                View.GONE
            }
        }
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    private fun populateAdView(container: LinearLayout): Int {
        return try {
            adMobBannerId?.let {
                val adRequest by inject<AdRequest>()
                container.removeAllViews()
                AdView(applicationContext).apply {
                    adUnitId = it
                    adSize = getBannerAdSize()
                    adListener = object : AdListener() {
                        override fun onAdFailedToLoad(p0: Int) {
                            val adview = when (adSize) {
                                AdSize.SMART_BANNER ->
                                    AdView(applicationContext).apply {
                                        adSize = AdSize.BANNER
                                    }
                                AdSize.BANNER -> null
                                else -> AdView(applicationContext).apply {
                                    adSize = AdSize.SMART_BANNER
                                }
                            }
                            adview?.run {
                                loadAd(AdRequest.Builder().build())
                                container.removeAllViews()
                                container.addView(this)
                            }
                        }
                    }
                    loadAd(adRequest)
                    container.addView(this)
                }
                View.VISIBLE
            } ?: View.GONE
        } catch (ignored: Exception) {
            View.GONE
        }
    }

    /***************************************
     * On Backpress
     ***************************************/
    open fun getBannerAdSize(): AdSize? {
        val display = windowManager.defaultDisplay
        val outMetrics = DisplayMetrics()
        display.getMetrics(outMetrics)
        val widthPixels = outMetrics.widthPixels.toFloat()
        val density = outMetrics.density
        val adWidth = (widthPixels / density).toInt()
        // Step 3 - Get adaptive ad size and return for setting on the ad view.
        return AdSize.getCurrentOrientationAnchoredAdaptiveBannerAdSize(
            this,
            adWidth
        )
    }

    /***************************************
     * OnCreate
     ***************************************/
    override fun onStart() {
        super.onStart()
        adMobBannerId?.let { registerReceiver(networkListener, intentFilter) }
    }

    /***************************************
     * OnCreate
     ***************************************/
    override fun onStop() {
        super.onStop()
        adMobBannerId?.let { unregisterReceiver(networkListener) }

    }

    /***************************************
     * Setting Observers
     ***************************************/
    open fun <T> setObserver(liveData: LiveData<T>, vararg functions: (value: T) -> Unit) {
        liveData.observe(this, Observer { value ->
            functions.forEach {
                it(value)
            }
        })
    }

    /***************************************
     * Setting Observers
     ***************************************/
    private fun setBaseObservers() {
        viewModel.viewState.observe(this, Observer { state ->
            run {
                if (state == BaseViewStates.STATE_NONE) return@run
                when (state) {
                    BaseViewStates.STATE_FINISH -> finish()
                    BaseViewStates.STATE_FINISH_WITH_DIALOG -> showConfirmFinishActivityDialog()
                    else -> observeViewState(state)
                }

                if (state != BaseViewStates.STATE_NONE)
                    viewModel.setViewState(BaseViewStates.STATE_NONE)
            }
        })

        setViewObservers()
    }


    /***************************************
     * Show Exit Confirmation
     ***************************************/
    open fun showConfirmFinishActivityDialog() {
        showMsgDialog(
            getString(R.string.app_name),
            "Are you sure want to exit from this application?",
            positiveButton = "EXIT", negativeButton = "CANCEL",
            finishActivityOnOk = true
        )
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    open fun showSnackbar(message: String? = null, messageResId: Int = -1) {
        try {
            val msgTxt = message?.let { it } ?: getString(messageResId)
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                Snackbar.make(binding.root, msgTxt, Snackbar.LENGTH_SHORT).show()
            } else showToast(msgTxt)
        } catch (e: Exception) {
        }
    }

    /***************************************
     * On Backpress
     ***************************************/
    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        initializeAdView()
    }


    /***************************************
     * On Backpress
     ***************************************/
    fun updateAppIfAvailable() {
        appUpdateManager?.appUpdateInfo?.addOnSuccessListener { appUpdateInfo ->
            if (appUpdateInfo.updateAvailability() == UpdateAvailability.UPDATE_AVAILABLE
                && appUpdateInfo.updatePriority() >= 0
            ) {
                appUpdateManager?.startUpdateFlowForResult(
                    appUpdateInfo,
                    AppUpdateType.IMMEDIATE, this, UpdateConstants.REQ_CODE_IMMD_UPDATE
                )
            }
        }
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    override fun onResume() {
        super.onResume()
        if (isUpdateActivity) {
            appUpdateManager?.appUpdateInfo?.addOnSuccessListener {
                if (it.updateAvailability() == UpdateAvailability.DEVELOPER_TRIGGERED_UPDATE_IN_PROGRESS) {
                    appUpdateManager?.startUpdateFlowForResult(
                        it,
                        AppUpdateType.IMMEDIATE,
                        this,
                        UpdateConstants.REQ_CODE_IMMD_UPDATE
                    )
                }
            }
        }
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    open fun handleHomeButton(): Boolean {
        if (isChildActivity) finish()
        return true
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        if (requestCode == UpdateConstants.REQ_CODE_IMMD_UPDATE) {
            when (resultCode) {
                RESULT_OK -> onUpdateSuccess()
                RESULT_CANCELED -> onUpdateCancelled()
                RESULT_IN_APP_UPDATE_FAILED -> onUpdateFailed()
            }
        }
    }

    /***************************************
     * On Backpress
     ***************************************/
    open fun onUpdateCancelled() {
        showMsgDialog("Update Required",
            "The app requires an update to function. Please update the app from playstore!",
            positiveButton = "OK",
            positiveClickListen = { finish() })
    }

    /***************************************
     * On Backpress
     ***************************************/
    open fun onUpdateFailed() {
        showMsgDialog("Update Failed",
            "The app requires an update to function. Please retry updating or try reinstalling the app from playstore!",
            positiveButton = "OK",
            positiveClickListen = { finish() })
    }


    /***************************************
     * On Backpress
     ***************************************/
    open fun observeViewState(state: Int) {}

    open fun onUpdateSuccess() {}
    open fun isHandleHomeButton() = false
    open fun setViewObservers() {}
    open fun initializeBeforeBinding() {}
    open fun initializeViews() {}
    open fun initializationAfterObserving() {}
    open fun handleOptionsItemClick(menuItemId: Int) = false
    open fun onNetworkChanged() {}
}
