package com.appspiriment.baseclasses.ads

import android.annotation.SuppressLint
import android.content.res.Configuration
import android.util.DisplayMetrics
import android.view.View
import android.widget.LinearLayout
import androidx.databinding.ViewDataBinding
import com.appspiriment.baseclasses.BaseViewModel
import com.appspiriment.baseclasses.NonAdBaseActivity
import com.google.android.gms.ads.AdListener
import com.google.android.gms.ads.AdRequest
import com.google.android.gms.ads.AdSize
import com.google.android.gms.ads.AdView
import kotlin.reflect.KClass


/*********************************************************
 * Class : BaseActivity
 * Purpose : Baseclass for Activities
 *  ******************************************************
 * Rework Details:
 * 1) {Author} :  {Date} : {Details}
 *********************************************************/
@Suppress("MemberVisibilityCanBePrivate")
@SuppressLint("Registered")
abstract class BaseActivity<
        out ViewModelType : BaseViewModel,
        out DataBindingType : ViewDataBinding>(
    viewModelClass: KClass<ViewModelType>,
    override val layoutId: Int,
    override val menuId: Int = R.menu.basedefaultmenu,
    protected val adMobBannerId: String? = null,
    override val toolbarId: Int? = null,
    override val isChildActivity: Boolean = false,
    override val isUpdateActivity: Boolean = false
) : NonAdBaseActivity<ViewModelType, DataBindingType>(
    viewModelClass = viewModelClass,
    layoutId = layoutId,
    menuId = menuId,
    toolbarId = toolbarId,
    isChildActivity = isChildActivity,
    isUpdateActivity = isUpdateActivity
) {

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    override fun initializeAdView() {
        findViewById<LinearLayout>(R.id.adView)?.let {
            it.visibility = if (isNetworkConnected) {
                populateAdView(it)
            } else {
                View.GONE
            }
        }
    }

    /***************************************
     * Show Exit Confirmation
     ***************************************/
    private fun populateAdView(container: LinearLayout): Int {
        return try {
            adMobBannerId?.let {
                val adRequest = AdRequest.Builder().build()
                container.removeAllViews()
                AdView(applicationContext).apply {
                    adUnitId = it
                    adSize = getBannerAdSize()
                    adListener = object : AdListener() {
                        override fun onAdFailedToLoad(p0: Int) {
                            val adview = when (adSize) {
                                AdSize.SMART_BANNER ->
                                    AdView(applicationContext).apply {
                                        adSize = AdSize.BANNER
                                    }
                                AdSize.BANNER -> null
                                else -> AdView(applicationContext).apply {
                                    adSize = AdSize.SMART_BANNER
                                }
                            }
                            adview?.run {
                                loadAd(AdRequest.Builder().build())
                                container.removeAllViews()
                                container.addView(this)
                            }
                        }
                    }
                    loadAd(adRequest)
                    container.addView(this)
                }
                View.VISIBLE
            } ?: View.GONE
        } catch (ignored: Exception) {
            View.GONE
        }
    }

    /***************************************
     * On Backpress
     ***************************************/
    open fun getBannerAdSize(): AdSize? {
        val display = windowManager.defaultDisplay
        val outMetrics = DisplayMetrics()
        display.getMetrics(outMetrics)
        val widthPixels = outMetrics.widthPixels.toFloat()
        val density = outMetrics.density
        val adWidth = (widthPixels / density).toInt()
        // Step 3 - Get adaptive ad size and return for setting on the ad view.
        return AdSize.getCurrentOrientationAnchoredAdaptiveBannerAdSize(
            this,
            adWidth
        )
    }

    /***************************************
     * OnCreate
     ***************************************/
    override fun onStart() {
        super.onStart()
        adMobBannerId?.let { registerReceiver(networkListener, intentFilter) }
    }

    /***************************************
     * OnCreate
     ***************************************/
    override fun onStop() {
        super.onStop()
        adMobBannerId?.let { unregisterReceiver(networkListener) }

    }

    /***************************************
     * On Backpress
     ***************************************/
    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        initializeAdView()
    }
}
