/*
 * Decompiled with CFR 0.152.
 */
package com.android.builder.utils;

import com.android.builder.utils.ExceptionConsumer;
import com.android.ide.common.util.ReadWriteProcessLock;
import com.android.ide.common.util.ReadWriteThreadLock;
import com.android.utils.FileUtils;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Maps;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

public class FileCache {
    private final File mCacheDirectory;
    private final LockingScope mLockingScope;
    private final AtomicInteger mMisses = new AtomicInteger(0);
    private final AtomicInteger mHits = new AtomicInteger(0);

    private FileCache(File cacheDirectory, LockingScope lockingScope) throws IOException {
        this.mCacheDirectory = cacheDirectory.getCanonicalFile();
        this.mLockingScope = lockingScope;
    }

    public static FileCache getInstanceWithInterProcessLocking(File cacheDirectory) throws IOException {
        return new FileCache(cacheDirectory, LockingScope.INTER_PROCESS);
    }

    public static FileCache getInstanceWithSingleProcessLocking(File cacheDirectory) throws IOException {
        return new FileCache(cacheDirectory, LockingScope.SINGLE_PROCESS);
    }

    public File getCacheDirectory() {
        return this.mCacheDirectory;
    }

    public QueryResult createFile(File outputFile, Inputs inputs, Callable<Void> fileCreator) throws ExecutionException, IOException {
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)outputFile, (File)this.mCacheDirectory) ? 1 : 0) != 0, (Object)String.format("Output file/directory '%1$s' must not be located in the cache directory '%2$s'", outputFile.getAbsolutePath(), this.mCacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)this.mCacheDirectory, (File)outputFile) ? 1 : 0) != 0, (Object)String.format("Output directory '%1$s' must not contain the cache directory '%2$s'", outputFile.getAbsolutePath(), this.mCacheDirectory.getAbsolutePath()));
        Preconditions.checkArgument((!outputFile.getCanonicalFile().equals(this.mCacheDirectory.getCanonicalFile()) ? 1 : 0) != 0, (Object)String.format("Output directory must not be the same as the cache directory '%1$s'", this.mCacheDirectory.getAbsolutePath()));
        File cacheEntryDir = this.getCacheEntryDir(inputs);
        File cachedFile = this.getCachedFile(cacheEntryDir);
        Callable<Void> createOutputFile = () -> {
            FileUtils.deletePath((File)outputFile);
            Files.createParentDirs((File)outputFile);
            try {
                fileCreator.call();
            }
            catch (Exception exception) {
                throw new FileCreatorException(exception);
            }
            return null;
        };
        Callable<Void> copyOutputFileToCachedFile = () -> {
            if (outputFile.exists()) {
                FileCache.copyFileOrDirectory(outputFile, cachedFile);
            }
            return null;
        };
        Callable<Void> copyCachedFileToOutputFile = () -> {
            FileUtils.deletePath((File)outputFile);
            Files.createParentDirs((File)outputFile);
            if (cachedFile.exists()) {
                FileCache.copyFileOrDirectory(cachedFile, outputFile);
            }
            return null;
        };
        Callable<Void> actionIfCacheHit = () -> (Void)this.doLocked(outputFile, LockingType.EXCLUSIVE, copyCachedFileToOutputFile);
        Callable<Void> actionIfCacheMissedOrCorrupted = () -> this.doLocked(outputFile, LockingType.EXCLUSIVE, () -> {
            createOutputFile.call();
            copyOutputFileToCachedFile.call();
            return null;
        });
        return this.queryCacheEntry(inputs, cacheEntryDir, actionIfCacheHit, actionIfCacheMissedOrCorrupted);
    }

    public QueryResult createFileInCacheIfAbsent(Inputs inputs, ExceptionConsumer<File> fileCreator) throws ExecutionException, IOException {
        File cacheEntryDir = this.getCacheEntryDir(inputs);
        File cachedFile = this.getCachedFile(cacheEntryDir);
        Callable<Void> actionIfCacheMissedOrCorrupted = () -> {
            try {
                fileCreator.accept(cachedFile);
            }
            catch (Exception exception) {
                throw new FileCreatorException(exception);
            }
            return null;
        };
        QueryResult queryResult = this.queryCacheEntry(inputs, cacheEntryDir, () -> null, actionIfCacheMissedOrCorrupted);
        return new QueryResult(queryResult.getQueryEvent(), queryResult.getCauseOfCorruption(), Optional.of(cachedFile));
    }

    private QueryResult queryCacheEntry(Inputs inputs, File cacheEntryDir, Callable<Void> actionIfCacheHit, Callable<Void> actionIfCacheMissedOrCorrupted) throws ExecutionException, IOException {
        try {
            return this.doLocked(this.mCacheDirectory, LockingType.SHARED, () -> {
                FileUtils.mkdirs((File)this.mCacheDirectory);
                QueryResult queryResult = this.doLocked(cacheEntryDir, LockingType.SHARED, () -> {
                    QueryResult result = this.checkCacheEntry(inputs, cacheEntryDir);
                    if (result.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                        this.mHits.incrementAndGet();
                        actionIfCacheHit.call();
                    }
                    return result;
                });
                if (queryResult.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                    return queryResult;
                }
                return this.doLocked(cacheEntryDir, LockingType.EXCLUSIVE, () -> {
                    QueryResult result = this.checkCacheEntry(inputs, cacheEntryDir);
                    if (result.getQueryEvent().equals((Object)QueryEvent.HIT)) {
                        this.mHits.incrementAndGet();
                        actionIfCacheHit.call();
                        return result;
                    }
                    if (result.getQueryEvent().equals((Object)QueryEvent.CORRUPTED)) {
                        FileUtils.deletePath((File)cacheEntryDir);
                    }
                    this.mMisses.incrementAndGet();
                    FileUtils.mkdirs((File)cacheEntryDir);
                    actionIfCacheMissedOrCorrupted.call();
                    Files.write((CharSequence)inputs.toString(), (File)this.getInputsFile(cacheEntryDir), (Charset)StandardCharsets.UTF_8);
                    return result;
                });
            });
        }
        catch (ExecutionException exception) {
            for (Throwable cause : Throwables.getCausalChain((Throwable)exception)) {
                if (cause instanceof FileCreatorException) {
                    throw exception;
                }
                if (cause instanceof IOException) {
                    throw new IOException(exception);
                }
                if (cause instanceof ExecutionException) continue;
                throw new RuntimeException(exception);
            }
            throw new RuntimeException("Unable to find root cause of ExecutionException, " + exception);
        }
    }

    private QueryResult checkCacheEntry(Inputs inputs, File cacheEntryDir) {
        String inputsInCacheEntry;
        if (!cacheEntryDir.exists()) {
            return new QueryResult(QueryEvent.MISSED);
        }
        File inputsFile = this.getInputsFile(cacheEntryDir);
        if (!inputsFile.exists()) {
            return new QueryResult(QueryEvent.CORRUPTED, new IllegalStateException(String.format("Inputs file '%s' does not exist", inputsFile.getAbsolutePath())));
        }
        try {
            inputsInCacheEntry = Files.toString((File)inputsFile, (Charset)StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            return new QueryResult(QueryEvent.CORRUPTED, e);
        }
        if (!inputs.toString().equals(inputsInCacheEntry)) {
            return new QueryResult(QueryEvent.CORRUPTED, new IllegalStateException(String.format("Expected contents '%s' but found '%s' in inputs file '%s'", inputs.toString(), inputsInCacheEntry, inputsFile.getAbsolutePath())));
        }
        return new QueryResult(QueryEvent.HIT);
    }

    private File getCacheEntryDir(Inputs inputs) {
        return new File(this.mCacheDirectory, inputs.getKey());
    }

    private File getCachedFile(File cacheEntryDir) {
        return new File(cacheEntryDir, "output");
    }

    private File getInputsFile(File cacheEntryDir) {
        return new File(cacheEntryDir, "inputs");
    }

    public File getFileInCache(Inputs inputs) {
        return this.getCachedFile(this.getCacheEntryDir(inputs));
    }

    public void delete() throws IOException {
        try {
            this.doLocked(this.mCacheDirectory, LockingType.EXCLUSIVE, () -> {
                FileUtils.deletePath((File)this.mCacheDirectory);
                return null;
            });
        }
        catch (ExecutionException exception) {
            if (exception.getCause() instanceof IOException) {
                throw new IOException(exception);
            }
            throw new RuntimeException(exception);
        }
    }

    <V> V doLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        if (this.mLockingScope == LockingScope.INTER_PROCESS) {
            return this.doInterProcessLocked(accessedFile, lockingType, action);
        }
        return this.doSingleProcessLocked(accessedFile, lockingType, action);
    }

    private <V> V doInterProcessLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        String lockFileName = Hashing.sha1().hashString((CharSequence)FileUtils.getCaseSensitivityAwareCanonicalPath((File)accessedFile), StandardCharsets.UTF_8).toString();
        File lockFile = accessedFile.getCanonicalFile().equals(this.mCacheDirectory) ? new File(System.getProperty("java.io.tmpdir"), lockFileName) : new File(this.mCacheDirectory, lockFileName);
        ReadWriteProcessLock readWriteProcessLock = new ReadWriteProcessLock(lockFile.toPath());
        ReadWriteProcessLock.Lock lock = lockingType == LockingType.SHARED ? readWriteProcessLock.readLock() : readWriteProcessLock.writeLock();
        lock.lock();
        try {
            V v = action.call();
            return v;
        }
        catch (Exception exception) {
            throw new ExecutionException(exception);
        }
        finally {
            lock.unlock();
        }
    }

    private <V> V doSingleProcessLocked(File accessedFile, LockingType lockingType, Callable<V> action) throws ExecutionException, IOException {
        ReadWriteThreadLock readWriteThreadLock = new ReadWriteThreadLock((Object)accessedFile.toPath().normalize());
        ReadWriteThreadLock.Lock lock = lockingType == LockingType.SHARED ? readWriteThreadLock.readLock() : readWriteThreadLock.writeLock();
        lock.lock();
        try {
            V v = action.call();
            return v;
        }
        catch (Exception exception) {
            throw new ExecutionException(exception);
        }
        finally {
            lock.unlock();
        }
    }

    int getMisses() {
        return this.mMisses.get();
    }

    int getHits() {
        return this.mHits.get();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("cacheDirectory", (Object)this.mCacheDirectory).add("lockingScope", (Object)this.mLockingScope).toString();
    }

    private static void copyFileOrDirectory(File from, File to) throws IOException {
        Preconditions.checkArgument((boolean)from.exists(), (Object)("Source path " + from.getAbsolutePath() + " does not exists."));
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)from, (File)to) ? 1 : 0) != 0);
        Preconditions.checkArgument((!FileUtils.isFileInDirectory((File)to, (File)from) ? 1 : 0) != 0);
        Preconditions.checkArgument((!from.getCanonicalFile().equals(to.getCanonicalFile()) ? 1 : 0) != 0);
        if (from.isFile()) {
            Files.createParentDirs((File)to);
            FileUtils.copyFile((File)from, (File)to);
        } else if (from.isDirectory()) {
            FileUtils.deletePath((File)to);
            FileUtils.copyDirectory((File)from, (File)to);
        }
    }

    public static enum QueryEvent {
        HIT,
        MISSED,
        CORRUPTED;

    }

    public static final class QueryResult {
        private final QueryEvent queryEvent;
        private final Optional<Throwable> causeOfCorruption;
        private final Optional<File> cachedFile;

        QueryResult(QueryEvent queryEvent, Optional<Throwable> causeOfCorruption, Optional<File> cachedFile) {
            Preconditions.checkState((queryEvent.equals((Object)QueryEvent.CORRUPTED) && causeOfCorruption.isPresent() || !queryEvent.equals((Object)QueryEvent.CORRUPTED) && !causeOfCorruption.isPresent() ? 1 : 0) != 0);
            this.queryEvent = queryEvent;
            this.causeOfCorruption = causeOfCorruption;
            this.cachedFile = cachedFile;
        }

        QueryResult(QueryEvent queryEvent, Throwable causeOfCorruption) {
            this(queryEvent, Optional.of(causeOfCorruption), Optional.empty());
        }

        QueryResult(QueryEvent queryEvent) {
            this(queryEvent, Optional.empty(), Optional.empty());
        }

        public QueryEvent getQueryEvent() {
            return this.queryEvent;
        }

        public Optional<Throwable> getCauseOfCorruption() {
            return this.causeOfCorruption;
        }

        public Optional<File> getCachedFile() {
            return this.cachedFile;
        }
    }

    public static enum Command {
        TEST,
        PREDEX_LIBRARY,
        PREPARE_LIBRARY;

    }

    public static final class Inputs {
        private static final String COMMAND = "COMMAND";
        private final Command command;
        private final LinkedHashMap<String, String> parameters;

        private Inputs(Builder builder) {
            this.command = builder.command;
            this.parameters = Maps.newLinkedHashMap((Map)builder.parameters);
        }

        public String toString() {
            return "COMMAND=" + this.command.name() + System.lineSeparator() + Joiner.on((String)System.lineSeparator()).withKeyValueSeparator("=").join(this.parameters);
        }

        public String getKey() {
            return Hashing.sha1().hashString((CharSequence)this.toString(), StandardCharsets.UTF_8).toString();
        }

        public static final class Builder {
            private final Command command;
            private final LinkedHashMap<String, String> parameters = Maps.newLinkedHashMap();

            public Builder(Command command) {
                this.command = command;
            }

            public Builder putFilePath(String name, File file) {
                this.parameters.put(name, file.getPath());
                return this;
            }

            public Builder putFileHash(String name, File file) throws IOException {
                Preconditions.checkArgument((boolean)file.isFile(), (Object)(file + " is not a file."));
                this.parameters.put(name, Files.hash((File)file, (HashFunction)Hashing.sha1()).toString());
                return this;
            }

            public Builder putString(String name, String value) {
                this.parameters.put(name, value);
                return this;
            }

            public Builder putBoolean(String name, boolean value) {
                this.parameters.put(name, String.valueOf(value));
                return this;
            }

            public Builder putLong(String name, long value) {
                this.parameters.put(name, String.valueOf(value));
                return this;
            }

            public Inputs build() {
                Preconditions.checkState((!this.parameters.isEmpty() ? 1 : 0) != 0, (Object)"Inputs must not be empty.");
                return new Inputs(this);
            }
        }
    }

    private static final class FileCreatorException
    extends ExecutionException {
        public FileCreatorException(Exception exception) {
            super(exception);
        }
    }

    static enum LockingType {
        SHARED,
        EXCLUSIVE;

    }

    private static enum LockingScope {
        INTER_PROCESS,
        SINGLE_PROCESS;

    }
}

