package android.support.test.rule.logging;

import android.os.Build;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.VisibleForTesting;
import android.support.test.InstrumentationRegistry;
import android.support.test.annotation.Beta;
import android.support.test.internal.util.Checks;

import org.junit.rules.ExternalResource;
import org.junit.runner.Description;
import org.junit.runners.model.Statement;

import java.io.File;

/**
 * Base class for logging rules to remove boiler-plate instance variables used by most logging
 * rules.
 */
@Beta
public abstract class LoggingBaseRule extends ExternalResource {

    private final String mPackageNameUnderTest;

    /**
     * This allows unit tests to test API level specific logic.
     */
    private int mAndroidRuntimeVersion = Build.VERSION.SDK_INT;

    /**
     * The file that "getprop" data was logged to. If this variable is null then data collection
     * has not been attempted.
     */
    private File mLogFile = null;

    private String mLogFileName;

    private File mLogFileOutputDirectory = null;

    private String mTestClass;

    private String mTestName;

    private int mTestRunNumber = 1;

    public LoggingBaseRule() {
        mPackageNameUnderTest = InstrumentationRegistry.getTargetContext().getPackageName();

        mLogFileName = getDefaultLogFileName();
    }

    /**
     * Request the log be written to a specific location.
     * <p/>
     * File logFileOutputDirectory directory to log output to
     * String fileName to override the default file name, if desired
     */
    public LoggingBaseRule(@NonNull File logFileOutputDirectory,
            @Nullable String fileName) {
        this();
        Checks.checkNotNull(logFileOutputDirectory,
                "Log directory must be provided when using this constructor");
        mLogFileOutputDirectory = logFileOutputDirectory;

        if (fileName != null) {
            mLogFileName = fileName;
        }
    }

    @VisibleForTesting
    void initialize() {
        if (mLogFileName != null) {
            if (null == mLogFileOutputDirectory) {
                mLogFile = RuleLoggingUtils.getTestFile(mTestClass, mTestName, mLogFileName,
                        mTestRunNumber);
            } else {
                mLogFile = new File(mLogFileOutputDirectory, mLogFileName);
            }
        }
    }

    @Override
    public final void after() {
        afterTest();
    }

    abstract void afterTest();

    @Override
    public final Statement apply(Statement base, Description description) {
        mTestName = description.getMethodName();
        mTestClass = description.getClassName();
        return super.apply(base, description);
    }

    @Override
    public final void before() {
        initialize();
        beforeTest();
    }

    abstract void beforeTest();

    @VisibleForTesting
    int getAndroidRuntimeVersion() {
        return mAndroidRuntimeVersion;
    }

    @VisibleForTesting
    void setAndroidRuntimeVersion(int version) {
        mAndroidRuntimeVersion = version;
    }

    /**
     * Return the default log file name for this rule class. A null return value indicates the rule
     * doesn't log to a file.
     */
    abstract String getDefaultLogFileName();

    @VisibleForTesting
    File getLogFile() {
        return mLogFile;
    }

    protected String getLogFileName() {
        return mLogFileName;
    }

    protected File getLogFileOutputDirectory() {
        return mLogFileOutputDirectory;
    }

    protected String getPackageNameUnderTest() {
        return mPackageNameUnderTest;
    }

    protected String getTestClass() {
        return mTestClass;
    }

    protected String getTestName() {
        return mTestName;
    }

    protected int getTestRunNumber() {
        return mTestRunNumber;
    }

    /**
     * Indicate to the {@code Rule} that this is the {@code testRunNumber}th time the test method
     * is being called in a test instrumentation. This method only needs to be called when a test
     * method is being called multiple times in a test instrumentation.
     *
     * @param testRunNumber indicates the nth run for the test method in zero-index form
     */
    public void setTestRunNumber(int testRunNumber) {
        Checks.checkState(testRunNumber >= 0, "Invalid test run number (" + testRunNumber + ")");
        mTestRunNumber = testRunNumber;
    }
}
