/*
 * Copyright 2015, Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.os.Trace;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.test.annotation.Beta;
import android.util.Log;

import org.junit.Rule;

import java.io.File;
import java.io.IOException;

import static android.support.test.rule.logging.RuleLoggingUtils.startProcess;
import static android.support.test.rule.logging.RuleLoggingUtils.startProcessAndLogToFile;

/**
 * When applied to a test class this {@code Rule} performs a network system information reset
 * before test method execution. The {@link org.junit.Rule} then collects network system
 * information after test method execution and logs it to a file on external storage. The default
 * file location is maintained at {@link RuleLoggingUtils#getTestDir(String, String, int)}.
 * The Rule performs these actions through the execution of a dumpsys on the {@code netstats}
 * service.
 * <p/>
 * The {@link Rule}'s {@link #before()} and {@link #after()} methods can also be used for logging
 * battery information across a test suite by manually invoking the methods in a
 * {@link org.junit.runner.notification.RunListener RunListener}.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public LogNetStatsRule mLogNetStatsRule = new LogNetStatsRule();
 * <code>
 */
@Beta
public class LogNetStatsRule extends LoggingBaseRule {

    private static final String TAG = "LogNetStatsRule";

    public LogNetStatsRule() {
    }

    public LogNetStatsRule(@NonNull File logFileOutputDirectory, @Nullable String fileName) {
        super(logFileOutputDirectory, fileName);
    }

    @Override
    String getDefaultLogFileName() {
        return "netstats.dumpsys.log";
    }

    @Override
    public void beforeTest() {
        Process process = null;
        try {
            process = startProcess(
                    new String[]{"dumpsys", "netstats", "--reset", getPackageNameUnderTest()});
            process.waitFor();
        } catch (InterruptedException | IOException exception) {
            Log.e(TAG, "Unable to reset netstats", exception);
        } finally {
            if (process != null) {
                process.destroy();
            }
        }
    }

    public void afterTest() {
        String[] commandParts;
        commandParts = new String[]{"dumpsys", "netstats"};

        try {
            Trace.beginSection("Netstats dumpsys");
            startProcessAndLogToFile(commandParts, getLogFile(), getAndroidRuntimeVersion());
        } finally {
            Trace.endSection();
        }
    }
}