/*
 * Copyright 2015, Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.os.Build;
import android.os.Trace;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.test.annotation.Beta;
import android.util.Log;

import org.junit.Rule;

import java.io.File;
import java.io.IOException;

import static android.support.test.rule.logging.RuleLoggingUtils.startProcess;
import static android.support.test.rule.logging.RuleLoggingUtils.startProcessAndLogToFile;

/**
 * When applied to a test class this {@code Rule} performs a graphics system information reset
 * before test method execution. The {@link Rule} then collects graphics system
 * information after test method execution and logs it to a file on external storage. The default
 * file location is maintained at {@link RuleLoggingUtils#getTestDir(String, String, int)}.
 * The Rule performs these actions through the execution of a dumpsys on the {@code gfxinfo}
 * service.
 * <p/>
 * The {@link Rule}'s {@link #before()} and {@link #after()} methods can also be used for logging
 * battery information across a test suite by manually invoking the methods in a
 * {@link org.junit.runner.notification.RunListener RunListener}.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public LogGraphicsStatsRule mLogGraphicsStatsRule = new LogGraphicsStatsRule();
 * <code>
 */
@Beta
public class LogGraphicsStatsRule extends LoggingBaseRule {

    private static final String TAG = "LogGraphicsStatsRule";

    private static final String DEFAULT_LOG_FILE_NAME = "gfxinfo.dumpsys.log";

    /**
     * {@inheritDoc}
     */
    public LogGraphicsStatsRule() {
    }

    /**
     * {@inheritDoc}
     */
    public LogGraphicsStatsRule(@NonNull File logFileOutputDirectory, @Nullable String fileName) {
        super(logFileOutputDirectory, fileName);
    }

    @Override
    public void afterTest() {
        String[] commandParts;
        if (getAndroidRuntimeVersion() >= Build.VERSION_CODES.M) {
            // The framestats parameter was added in Android M.
            commandParts =
                    new String[]{"dumpsys", "gfxinfo", getPackageNameUnderTest(), "framestats"};
        } else {
            commandParts = new String[]{"dumpsys", "gfxinfo", getPackageNameUnderTest()};
        }

        try {
            Trace.beginSection("Gfxinfo dumpsys");
            startProcessAndLogToFile(commandParts, getLogFile(), getAndroidRuntimeVersion());
        } finally {
            Trace.endSection();
        }
    }

    @Override
    public void beforeTest() {
        Process process = null;
        try {
            process = startProcess(
                    new String[]{"dumpsys", "gfxinfo", "--reset", getPackageNameUnderTest()});
            process.waitFor();
        } catch (InterruptedException | IOException exception) {
            Log.e(TAG, "Unable to reset reset gfxinfo", exception);
        } finally {
            if (process != null) {
                process.destroy();
            }
        }
    }

    @Override
    String getDefaultLogFileName() {
        return DEFAULT_LOG_FILE_NAME;
    }
}
