/*
 * Copyright (C) 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.support.test.rule.logging;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.test.annotation.Beta;

import org.junit.Rule;

import java.io.File;

import static android.support.test.rule.logging.RuleLoggingUtils.startCmdAndLogOutputPostL;

/**
 * When applied to a test class this {@code Rule} executes the "getprop" device command after
 * test method execution and logs it to a file on external storage. The default file location is
 * retrieved via {@link RuleLoggingUtils#getTestDir(String, String, int)}.
 * <p/>
 * The {@link Rule} logs the runtime build number, device name, build type, lcd density, cpu
 * type, among many other device state attributes. When used in conjunction with a device lab it
 * might assist in identifying trends in test failures for a subset of Android devices.
 * <p/>
 * The pertinent information output by this rule typically doesn't change between tests, therefore,
 * it is typically used manually by using the {@link #before()} and {@link #after()} methods in a
 * {@link org.junit.runner.notification.RunListener RunListener}.
 * <p/>
 * Usage:
 * <code>
 * \@Rule
 * public LogDeviceGetPropInfoRule mLogDeviceGetPropInfoRule = new LogDeviceGetPropInfoRule();
 * </code>
 */
@Beta
public class LogDeviceGetPropInfoRule extends LoggingBaseRule {

    private static final String DEFAULT_LOG_FILE_NAME = "deviceprops.log";

    /**
     * {@inheritDoc}
     */
    public LogDeviceGetPropInfoRule() {
    }

    /**
     * {@inheritDoc}
     */
    public LogDeviceGetPropInfoRule(@NonNull File logFileOutputDirectory,
            @Nullable String fileName) {
        super(logFileOutputDirectory, fileName);
    }

    @Override
    public void afterTest() {
        String[] commandParts = new String[]{"getprop"};
        startCmdAndLogOutputPostL(commandParts, getLogFile());
    }

    @Override
    public void beforeTest() {
        // Do nothing
    }

    String getDefaultLogFileName() {
        return DEFAULT_LOG_FILE_NAME;
    }
}
