package com.aliyun.core.utils;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

public class Configuration implements Cloneable {

    public static final String PROPERTY_HTTP_PROXY = "HTTP_PROXY";

    public static final String PROPERTY_HTTPS_PROXY = "HTTPS_PROXY";

    public static final String PROPERTY_NO_PROXY = "NO_PROXY";

    public static final String PROPERTY_ALIBABA_CLOUD_LOG_LEVEL = "ALIBABA_CLOUD_SDK_LOG_LEVEL";

    private static final String[] DEFAULT_CONFIGURATIONS = {
            PROPERTY_HTTP_PROXY,
            PROPERTY_HTTPS_PROXY,
            PROPERTY_NO_PROXY,
            PROPERTY_ALIBABA_CLOUD_LOG_LEVEL,
    };

    /*
     * Gets the global configuration shared by all client libraries.
     */
    private static final Configuration GLOBAL_CONFIGURATION = new Configuration();

    public static final Configuration NONE = new NoopConfiguration();

    private final ConcurrentMap<String, String> configurations;

    public Configuration() {
        this.configurations = new ConcurrentHashMap<>();
        loadBaseConfiguration(this);
    }

    private Configuration(ConcurrentMap<String, String> configurations) {
        this.configurations = new ConcurrentHashMap<>(configurations);
    }

    public static Configuration getGlobalConfiguration() {
        return GLOBAL_CONFIGURATION;
    }

    public String get(String name) {
        return getOrLoad(name);
    }

    public <T> T get(String name, T defaultValue) {
        return convertOrDefault(getOrLoad(name), defaultValue);
    }

    public <T> T get(String name, Function<String, T> converter) {
        String value = getOrLoad(name);
        if (StringUtils.isEmpty(value)) {
            return null;
        }

        return converter.apply(value);
    }

    private String getOrLoad(String name) {
        String value = configurations.get(name);
        if (value != null) {
            return value;
        }

        value = load(name);
        if (value != null) {
            configurations.put(name, value);
            return value;
        }

        return null;
    }

    private String load(String name) {
        String value = loadFromProperties(name);

        if (value != null) {
            return value;
        }

        return loadFromEnvironment(name);
    }

    String loadFromEnvironment(String name) {
        return System.getenv(name);
    }

    String loadFromProperties(String name) {
        return System.getProperty(name);
    }

    public Configuration put(String name, String value) {
        configurations.put(name, value);
        return this;
    }

    public String remove(String name) {
        return configurations.remove(name);
    }

    public boolean contains(String name) {
        return configurations.containsKey(name);
    }

    @SuppressWarnings("CloneDoesntCallSuperClone")
    public Configuration clone() {
        return new Configuration(configurations);
    }

    @SuppressWarnings("unchecked")
    private static <T> T convertOrDefault(String value, T defaultValue) {
        if (StringUtils.isEmpty(value)) {
            return defaultValue;
        }

        Object convertedValue;
        if (defaultValue instanceof Byte) {
            convertedValue = Byte.parseByte(value);
        } else if (defaultValue instanceof Short) {
            convertedValue = Short.parseShort(value);
        } else if (defaultValue instanceof Integer) {
            convertedValue = Integer.parseInt(value);
        } else if (defaultValue instanceof Long) {
            convertedValue = Long.parseLong(value);
        } else if (defaultValue instanceof Float) {
            convertedValue = Float.parseFloat(value);
        } else if (defaultValue instanceof Double) {
            convertedValue = Double.parseDouble(value);
        } else if (defaultValue instanceof Boolean) {
            convertedValue = Boolean.parseBoolean(value);
        } else {
            convertedValue = value;
        }

        return (T) convertedValue;
    }

    private void loadBaseConfiguration(Configuration configuration) {
        for (String config : DEFAULT_CONFIGURATIONS) {
            String value = load(config);
            if (value != null) {
                configuration.put(config, value);
            }
        }
    }
}
