package com.aliyun.core.http;

import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.Configuration;
import com.aliyun.core.utils.StringUtils;

import java.io.UnsupportedEncodingException;
import java.net.*;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.regex.Pattern;

public class ProxyOptions {
    private static final ClientLogger LOGGER = new ClientLogger(ProxyOptions.class);
    private static final String INVALID_CONFIGURATION_MESSAGE = "'configuration' cannot be 'Configuration.NONE'.";
    private static final String INVALID_ALIYUN_PROXY_URL = "Configuration {} is an invalid URL and is being ignored.";
    private static final String JAVA_PROXY_PREREQUISITE = "java.net.useSystemProxies";
    private static final String JAVA_PROXY_HOST = "proxyHost";
    private static final String JAVA_PROXY_PORT = "proxyPort";
    private static final String JAVA_PROXY_USER = "proxyUser";
    private static final String JAVA_PROXY_PASSWORD = "proxyPassword";
    private static final String JAVA_NON_PROXY_HOSTS = "http.nonProxyHosts";
    private static final String HTTPS = "https";
    private static final int DEFAULT_HTTPS_PORT = 443;
    private static final String HTTP = "http";
    private static final int DEFAULT_HTTP_PORT = 80;

    private static final List<BiFunction<Configuration, Boolean, ProxyOptions>> ENVIRONMENT_LOAD_ORDER = Arrays.asList(
            (configuration, resolveProxy)
                    -> attemptToLoadAliProxy(configuration, resolveProxy, Configuration.PROPERTY_HTTPS_PROXY),
            (configuration, resolveProxy)
                    -> attemptToLoadAliProxy(configuration, resolveProxy, Configuration.PROPERTY_HTTP_PROXY),
            (configuration, resolveProxy)
                    -> attemptToLoadJavaProxy(configuration, resolveProxy, HTTPS),
            (configuration, resolveProxy)
                    -> attemptToLoadJavaProxy(configuration, resolveProxy, HTTP)
    );

    private final InetSocketAddress address;
    private final Type type;
    private String scheme;
    private String username;
    private String password;
    private String nonProxyHosts;

    public ProxyOptions(Type type, InetSocketAddress address) {
        this.type = type;
        this.address = address;
    }

    public ProxyOptions(Type type, InetSocketAddress address, String scheme) {
        this.type = type;
        this.address = address;
        this.scheme = scheme;
    }

    public ProxyOptions setCredentials(String username, String password) {
        this.username = Objects.requireNonNull(username, "proxy 'username' cannot be null.");
        this.password = Objects.requireNonNull(password, "proxy 'password' cannot be null.");
        return this;
    }

    public ProxyOptions setNonProxyHosts(String nonProxyHosts) {
        this.nonProxyHosts = sanitizeNoProxy(nonProxyHosts);
        return this;
    }

    /**
     * The address of the proxy.
     */
    public InetSocketAddress getAddress() {
        return address;
    }

    /**
     * The type of the proxy.
     */
    public Type getType() {
        return type;
    }

    /**
     * The scheme of the proxy host : http or https.
     */
    public String getScheme() {
        return scheme;
    }

    /**
     * The proxy user name.
     */
    public String getUsername() {
        return this.username;
    }

    /**
     * The proxy password.
     */
    public String getPassword() {
        return this.password;
    }

    /**
     * The hosts that bypass the proxy.
     */
    public String getNonProxyHosts() {
        return this.nonProxyHosts;
    }

    public static ProxyOptions fromConfiguration(Configuration configuration) {
        return fromConfiguration(configuration, false);
    }

    public static ProxyOptions fromConfiguration(Configuration configuration, boolean createUnresolved) {
        if (configuration == Configuration.NONE) {
            throw LOGGER.logExceptionAsWarning(new IllegalArgumentException(INVALID_CONFIGURATION_MESSAGE));
        }

        Configuration proxyConfiguration = (configuration == null)
                ? Configuration.getGlobalConfiguration()
                : configuration;

        for (BiFunction<Configuration, Boolean, ProxyOptions> loader : ENVIRONMENT_LOAD_ORDER) {
            ProxyOptions proxyOptions = loader.apply(proxyConfiguration, createUnresolved);
            if (proxyOptions != null) {
                return proxyOptions;
            }
        }

        return null;
    }

    private static ProxyOptions attemptToLoadAliProxy(Configuration configuration, boolean createUnresolved,
                                                      String proxyProperty) {
        String proxyConfiguration = configuration.get(proxyProperty);

        // No proxy configuration setup.
        if (StringUtils.isEmpty(proxyConfiguration)) {
            return null;
        }

        try {
            URL proxyUrl = new URL(proxyConfiguration);
            int port = (proxyUrl.getPort() == -1) ? proxyUrl.getDefaultPort() : proxyUrl.getPort();

            InetSocketAddress socketAddress = (createUnresolved)
                    ? InetSocketAddress.createUnresolved(proxyUrl.getHost(), port)
                    : new InetSocketAddress(proxyUrl.getHost(), port);

            ProxyOptions proxyOptions = new ProxyOptions(Type.HTTP, socketAddress, proxyUrl.getProtocol());

            String nonProxyHostsString = configuration.get(Configuration.PROPERTY_NO_PROXY);
            if (!StringUtils.isEmpty(nonProxyHostsString)) {
                proxyOptions.nonProxyHosts = sanitizeNoProxy(nonProxyHostsString);
            }

            String userInfo = proxyUrl.getUserInfo();
            if (userInfo != null) {
                String[] usernamePassword = userInfo.split(":", 2);
                if (usernamePassword.length == 2) {
                    try {
                        proxyOptions.setCredentials(
                                URLDecoder.decode(usernamePassword[0], StandardCharsets.UTF_8.toString()),
                                URLDecoder.decode(usernamePassword[1], StandardCharsets.UTF_8.toString())
                        );
                    } catch (UnsupportedEncodingException e) {
                        return null;
                    }
                }
            }

            return proxyOptions;
        } catch (MalformedURLException ex) {
            LOGGER.warning(INVALID_ALIYUN_PROXY_URL, proxyProperty);
            return null;
        }
    }

    private static String sanitizeNoProxy(String noProxyString) {
        String[] nonProxyHosts = noProxyString.split(",");
        for (int i = 0; i < nonProxyHosts.length; i++) {
            String prefixWildcard = "";
            String suffixWildcard = "";
            String body = nonProxyHosts[i];
            if (body.startsWith(".*")) {
                prefixWildcard = ".*";
                body = body.substring(2);
            } else if (body.startsWith("*") || body.startsWith(".")) {
                prefixWildcard = ".*";
                body = body.substring(1);
            }
            if (body.endsWith(".*")) {
                suffixWildcard = ".*";
                body = body.substring(0, body.length() - 2);
            } else if (body.endsWith("*") || body.endsWith(".")) {
                suffixWildcard = ".*";
                body = body.substring(0, body.length() - 1);
            }

            nonProxyHosts[i] = prefixWildcard + Pattern.quote(body) + suffixWildcard;
        }

        return String.join("|", nonProxyHosts);
    }

    private static ProxyOptions attemptToLoadJavaProxy(Configuration configuration, boolean createUnresolved,
                                                       String type) {
        if (!Boolean.parseBoolean(configuration.get(JAVA_PROXY_PREREQUISITE))) {
            return null;
        }
        String host = configuration.get(type + "." + JAVA_PROXY_HOST);
        if (StringUtils.isEmpty(host)) {
            return null;
        }

        int port;
        try {
            port = Integer.parseInt(configuration.get(type + "." + JAVA_PROXY_PORT));
        } catch (NumberFormatException ex) {
            port = HTTPS.equals(type) ? DEFAULT_HTTPS_PORT : DEFAULT_HTTP_PORT;
        }

        InetSocketAddress socketAddress = (createUnresolved)
                ? InetSocketAddress.createUnresolved(host, port)
                : new InetSocketAddress(host, port);

        ProxyOptions proxyOptions = new ProxyOptions(Type.HTTP, socketAddress, type);

        String nonProxyHostsString = configuration.get(JAVA_NON_PROXY_HOSTS);
        if (!StringUtils.isEmpty(nonProxyHostsString)) {
            proxyOptions.nonProxyHosts = sanitizeJavaHttpNonProxyHosts(nonProxyHostsString);
        }

        String username = configuration.get(type + "." + JAVA_PROXY_USER);
        String password = configuration.get(type + "." + JAVA_PROXY_PASSWORD);

        if (username != null && password != null) {
            proxyOptions.setCredentials(username, password);
        }

        return proxyOptions;
    }

    private static String sanitizeJavaHttpNonProxyHosts(String nonProxyHostsString) {
        String[] nonProxyHosts = nonProxyHostsString.split("\\|");

        for (int i = 0; i < nonProxyHosts.length; i++) {
            String prefixWildcard = "";
            String suffixWildcard = "";
            String body = nonProxyHosts[i];

            if (body.startsWith("*")) {
                prefixWildcard = ".*";
                body = body.substring(1);
            }

            if (body.endsWith("*")) {
                suffixWildcard = ".*";
                body = body.substring(0, body.length() - 1);
            }
            nonProxyHosts[i] = prefixWildcard + Pattern.quote(body) + suffixWildcard;
        }

        return String.join("|", nonProxyHosts);
    }

    /**
     * The type of the proxy.
     */
    public enum Type {
        /**
         * HTTP proxy type.
         */
        HTTP(Proxy.Type.HTTP),

        /**
         * SOCKS4 proxy type.
         */
        SOCKS4(Proxy.Type.SOCKS),

        /**
         * SOCKS5 proxy type.
         */
        SOCKS5(Proxy.Type.SOCKS);

        private final Proxy.Type proxyType;

        Type(Proxy.Type proxyType) {
            this.proxyType = proxyType;
        }

        /**
         * Get the {@link Proxy.Type} equivalent of this type.
         *
         * @return the proxy type
         */
        public Proxy.Type toProxyType() {
            return proxyType;
        }
    }
}
