////////////////////////////////////////////////////////////////////////////////
//
//  Copyright (C) 2003-2006 Adobe Macromedia Software LLC and its licensors.
//  All Rights Reserved. The following is Source Code and is subject to all
//  restrictions on such code as contained in the End User License Agreement
//  accompanying this product.
//
////////////////////////////////////////////////////////////////////////////////

package mx.charts
{

import flash.events.Event;
import mx.charts.chartClasses.AxisLabelSet;
import mx.charts.chartClasses.NumericAxis;
import mx.core.mx_internal;

use namespace mx_internal;

/**
 *  The LogAxis class maps numeric values logarithmically
 *  between a minimum and maximum value along a chart axis.
 *  By default, it determines <code>minimum</code>, <code>maximum</code>,
 *  and <code>interval</code> values from the charting data
 *  to fit all of the chart elements on the screen.
 *  You can also explicitly set specific values for these properties.
 *  A LogAxis object cannot correctly render negative values,
 *  as Log10() of a negative number is <code>undefined</code>.
 *
 *  @mxml
 *  
 *  <p>The <code>&lt;mx:LogAxis&gt;</code> tag inherits all the properties
 *  of its parent classes and adds the following properties:</p>
 *  
 *  <pre>
 *  &lt;mx:LogAxis
 *    <strong>Properties</strong>
 *    interval="10"
 *    maximum="null"
 *    maximumLabelPrecision="null"
 *    minimum="null"
 *  /&gt;
 *  </pre>
 * 
 *  @see mx.charts.chartClasses.IAxis
 *
 *  @includeExample examples/LogAxisExample.mxml
 */
public class LogAxis extends NumericAxis
{
    include "../core/Version.as";

	//--------------------------------------------------------------------------
	//
	//  Constructor
	//
	//--------------------------------------------------------------------------
	
	/**
	 *  Constructor.
	 */
	public function LogAxis() 
	{
		super();

		computedInterval = 1;							
	}

	//----------------------------------
	//  interval
	//----------------------------------

    [Inspectable(category="General")]

	/**
	 *  Specifies the multiplier label values along the axis.
	 *  A value of 10 generates labels at 1, 10, 100, 1000, etc., 
	 *  while a value of 100 generates labels at 1, 100, 10000, etc.
	 *  Flex calculates the interval if this property
	 *  is set to <code>NaN</code>.
	 *  Intervals must be even powers of 10, and must be greater than or equal to 10.
	 *  The LogAxis rounds the interval down to an even power of 10, if necessary.
	 *
	 *  @default 10
	 */
	public function get interval():Number
	{
		return Math.pow(10, computedInterval);
	}

	/**
	 *  @private
	 */
	public function set interval(value:Number):void
	{
		if (!isNaN(value))
			value = Math.max(1, Math.floor(Math.log(value) / Math.LN10));

		if (isNaN(value))
			value = 1;

		computedInterval = value;
		invalidateCache();

		dispatchEvent(new Event("axisChange"));
	}

	//----------------------------------
	//  maximum
	//----------------------------------

    [Inspectable(category="General")]

	/**
	 *  Specifies the maximum value for an axis label. 
	 *  If <code>NaN</code>, Flex determines the maximum value
	 *  from the data in the chart. 
	 *  The maximum value must be an even power of 10.
	 *  The LogAxis rounds an explicit maximum value
	 *  up to an even power of 10, if necessary.
	 *  
	 *  @default NaN
	 */
	public function get maximum():Number
	{
		return Math.pow(10, computedMaximum);
	}
	
	/**
	 *  @private
	 */
	public function set maximum(value:Number):void
	{
		computedMaximum = Math.ceil(Math.log(value) / Math.LN10);
		assignedMaximum = computedMaximum;
		
		invalidateCache();

		dispatchEvent(new Event("mappingChange"));
		dispatchEvent(new Event("axisChange"));
	}

	//----------------------------------
	//  maximumLabelPrecision
	//----------------------------------

	/**
	 *  @private
	 */	
	private var _labelPrecision:Number;

	/**
	 * @private
	 */
	public function get maximumLabelPrecision():Number
	{
		return _labelPrecision;
	}

	/**
	 *  Specifies the maximum number of decimal places for representing fractional 
	 *  values on the labels generated by this axis. By default, the 
	 *  Axis autogenerates this value from the labels themselves.  A value of 0 
	 *  round to the nearest integer value, while a value of 2 rounds to the nearest 
	 *  1/100th of a value.
	 */
	public function set maximumLabelPrecision(value:Number):void
	{
		_labelPrecision = value;

		invalidateCache();
	}

	//----------------------------------
	//  minimum
	//----------------------------------

    [Inspectable(category="General")]

	/**
	 *  Specifies the minimum value for an axis label.
	 *  If <code>NaN</code>, Flex determines the minimum value
	 *  from the data in the chart. 
	 *  The minimum value must be an even power of 10.
	 *  The LogAxis will round an explicit minimum value
	 *  downward to an even power of 10 if necessary.
	 *  
	 *  @default NaN
	 */
	public function get minimum():Number
	{
		return Math.pow(10, computedMinimum);
	}
	
	/**
	 *  @private
	 */
	public function set minimum(value:Number):void
	{
		if(value == 0)
			assignedMinimum = NaN;
		else
		{			
			assignedMinimum = Math.floor(Math.log(value) / Math.LN10);
		}
		computedMinimum = assignedMinimum;

		invalidateCache();

		dispatchEvent(new Event("mappingChange"));
		dispatchEvent(new Event("axisChange"));
	}

	//--------------------------------------------------------------------------
	//
	//  Overridden methods: NumericAxis
	//
	//--------------------------------------------------------------------------

	/**
	 *  @private
	 */
	override protected function adjustMinMax(minValue:Number,
											 maxValue:Number):void
	{
		// esg: We always floor to the nearest power of 10
		//if (autoAdjust && isNaN(assignedMinimum)) 
			computedMinimum = Math.floor(computedMinimum);
		
		// esg: We always ceil o the nearest power of 10
		//if (autoAdjust && isNaN(assignedMaximum)) 
			computedMaximum = Math.ceil(computedMaximum);
	}

	/**
	 *  @private
	 */
	override public function mapCache(cache:Array /* of ChartItem */, field:String,
									  convertedField:String,
									  indexValues:Boolean = false):void
	{
		const ln10:Number = Math.LN10;

		var n:int = cache.length;
		var i:int;
		var v:Object;
		var vf:Number;
		var parseFunction:Function = this.parseFunction;
		
		if (parseFunction != null)
		{
			for (i = 0; i < n; i++)
			{
				v = cache[i];				
				v[convertedField] =
					Math.log(Number(parseFunction(v[field]))) / ln10;
			}		
		}
		else
		{
			for (i = 0; i < n && cache[i][field] == null; i++)
			{
			}
	
			if (i == n)
				return;
	
			if (cache[i][field] is String)
			{
				for (i = 0; i < n; i++)
				{
					v = cache[i];
					v[convertedField] = Math.log(Number(v[field])) / ln10;
				}
			}
			else if (cache[i][field] is XML ||
					 cache[i][field] is XMLList)
			{
				for (i = 0; i < n; i++)
				{
					v = cache[i];
					v[convertedField] =
						Math.log(Number(v[field].toString())) / ln10;
				}
			}
			else if (cache[i][field] is Number ||
					 cache[i][field] is int ||
					 cache[i][field] is uint)
			{
				for (i = 0; i < n; i++)
				{
					v = cache[i];
	
					v[convertedField] = v[field] == null ?
										NaN :
										Math.log(v[field]) / ln10;
				}
			}
			else
			{
				for (i = 0; i < n; i++)
				{
					v = cache[i];
	
					v[convertedField] = Math.log(parseFloat(v[field])) / ln10;
				}
			}
		}
	}	

	/**
	 *  @private
	 */
	override protected function buildLabelCache():Boolean
	{
		if (labelCache)
			return false;

		labelCache = [];

		var r:Number = computedMaximum - computedMinimum;		
		
		var labelBase:Number = labelMinimum - 
			Math.floor((labelMinimum - computedMinimum) / computedInterval) *
			computedInterval;
		
		var labelTop:Number = computedMaximum + 0.000001
		
		var labelFunction:Function = this.labelFunction;

		var i:Number;
		var v:Number;
		var roundedValue:Number;
		
		var roundBase:Number;
		if (!isNaN(_labelPrecision))
			roundBase = Math.pow(10, _labelPrecision);

		if (labelFunction != null)
		{
			var previousValue:Number = NaN;
			for (i = labelBase; i <= labelTop; i += computedInterval)
			{
				v = Math.pow(10, i);
				
				roundedValue = isNaN(_labelPrecision) ?
							   v :
							   Math.round(v * roundBase) / roundBase;
				
				labelCache.push(new AxisLabel((i - computedMinimum) / r, v,
					labelFunction(roundedValue, previousValue, this)));
				
				previousValue = v;
			}		
		}
		else
		{
			for (i = labelBase; i <= labelTop; i += computedInterval)
			{
				v = Math.pow(10, i);

				roundedValue = isNaN(_labelPrecision) ?
							   v :
							   Math.round(v * roundBase) / roundBase;
				
				labelCache.push(new AxisLabel((i - computedMinimum)/r, v,
					roundedValue.toString()));
			}						
		}

		return true;
	}

	/**
	 *  @private
	 */
	override protected function buildMinorTickCache():Array /* of Number */
	{
		var cache:Array /* of Number */ = [];

		var n:int = labelCache.length;
		for (var i:int = 0; i < labelCache.length; i++)
		{
			cache.push(labelCache[i].position);
		}

		return cache;
	}

	/**
	 *  @private
	 */
	override public function reduceLabels(intervalStart:AxisLabel,
										  intervalEnd:AxisLabel):AxisLabelSet
	{
		var intervalMultiplier:Number =
			Math.round((Math.log(Number(intervalEnd.value)) / Math.LN10) - 
			Math.log(Number(intervalStart.value)) / Math.LN10);
		intervalMultiplier =
			Math.floor(intervalMultiplier / computedInterval) + 1;
		
		var labels:Array /* of AxisLabel */ = [];
		var newMinorTicks:Array /* of Number */ = [];
		var newTicks:Array /* of Number */ = [];

		var r:Number = computedMaximum - computedMinimum;		
		
		var labelBase:Number = labelMinimum -
			Math.floor((labelMinimum - computedMinimum) / computedInterval) *
			computedInterval;
		
		var labelTop:Number = computedMaximum + 0.000001
		
		for (var i:int = 0; i < labelCache.length; i += intervalMultiplier)
		{
			var ci:AxisLabel = labelCache[i];
			labels.push(ci);
			newTicks.push(ci.position);
			newMinorTicks.push(ci.position);
		}		
				
		var labelSet:AxisLabelSet = new AxisLabelSet();
		labelSet.labels = labels;
		labelSet.minorTicks = newMinorTicks;
		labelSet.ticks = newTicks;
		labelSet.accurate = true;
		return labelSet;
	}

	/**
	 *  @private
	 */
	override public function invertTransform(value:Number):Object
	{
		update();

		return Math.pow(10,
			value * (computedMaximum - computedMinimum) + computedMinimum);
	}

	/**
	 *  @private
	 */
	override protected function guardMinMax(min:Number, max:Number):Array /* of int */
	{
		if (isNaN(min) || !isFinite(min))
			min = 0;
		if (isNaN(max) || !isFinite(max))
			max = min + 2;
		if (max == min)
			max = min + 2;

		return [min,max];
	}
}

}
