/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.http;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.StringBody;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ApacheHttpUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(ApacheHttpUtil.class);


    public static HttpUriRequest getApacheRequest(HttpRequest httpRequest) {
        HttpUriRequest apacheHttpRequest = null;
        switch (httpRequest.getHttpMethod()) {
            case GET:
                LOGGER.debug("Creating apache http get request for {}", httpRequest.getRequestKey());
                apacheHttpRequest = createGetRequest(httpRequest);
                break;
            case POST:
                LOGGER.debug("Creating apache http post request for {}", httpRequest.getRequestKey());
                RequestType requestType = httpRequest.getRequestType();
                if (requestType.equals(RequestType.REGULAR)) {
                    apacheHttpRequest = createPostRequest(httpRequest);
                } else {
                    apacheHttpRequest = createMultiPartRequest((MultiPartRequest) httpRequest);
                }
                break;
            case PUT:
                break;
        }
        return apacheHttpRequest;
    }

    private static HttpGet createGetRequest(HttpRequest httpRequest) {
        HttpGet apacheHttpGet = new HttpGet(httpRequest.getFinalUri());
        setRequestConfig(httpRequest, apacheHttpGet);
        httpRequest.getHeaders().forEach(apacheHttpGet::setHeader);
        return apacheHttpGet;
    }

    private static void setRequestConfig(HttpRequest httpRequest, HttpRequestBase httpRequestBase) {
        HttpRequestConfig requestConfig = httpRequest.getConfig();
        if (requestConfig != null) {
            RequestConfig.Builder apacheRequestConfigBuilder = RequestConfig.custom()
                    .setConnectTimeout(requestConfig.getConnectTimeout())
                    .setSocketTimeout(requestConfig.getSocketTimeout());
            if (requestConfig.getProxyHost() != null) {
                HttpHost proxy = new HttpHost(requestConfig.getProxyHost(), requestConfig.getProxyPort(), requestConfig.getProxyScheme());
                apacheRequestConfigBuilder.setProxy(proxy);
            }
            httpRequestBase.setConfig(apacheRequestConfigBuilder.build());
        }
    }

    private static HttpPost createPostRequest(HttpRequest httpRequest) {
        HttpPost apacheHttpPost = new HttpPost(httpRequest.getFinalUri());
        if (httpRequest.getEntity()!=null) {
            apacheHttpPost.setEntity(httpRequest.getEntity());
        }
        setRequestConfig(httpRequest, apacheHttpPost);
        httpRequest.getHeaders().forEach(apacheHttpPost::setHeader);
        return apacheHttpPost;
    }

    private static HttpPost createMultiPartRequest(MultiPartRequest multiPartRequest) {
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();

        HttpPost apacheHttpMultipartRequest = new HttpPost(multiPartRequest.getFinalUri());
        List<ByteArrayPart> byteArrayParts = multiPartRequest.getByteArrayParts();
        List<InputStreamPart> inputStreamParts = multiPartRequest.getInputStreams();

        StringBodyPart bodyPart = multiPartRequest.getStringBodyPart();
        if (bodyPart != null) {
            builder.addPart(bodyPart.getName(), new StringBody(bodyPart.getBody(),
                    ContentType.create(bodyPart.getContentType(), StandardCharsets.UTF_8)));
        }

        //can be removed
        if(byteArrayParts != null) {
            for (ByteArrayPart byteArrayPart : byteArrayParts) {
                ByteArrayBody byteArrayBody = new ByteArrayBody(byteArrayPart.getBody(), byteArrayPart.getFileName());
                builder.addPart(byteArrayPart.getName(), byteArrayBody);
            }
        }

        if(inputStreamParts != null) {
          LOGGER.debug("Added input stream part {} request", multiPartRequest.getRequestKey());
          for (InputStreamPart inputStreamPart : inputStreamParts) {
            builder.addBinaryBody(inputStreamPart.getName(), inputStreamPart.getInputStream(), ContentType.APPLICATION_OCTET_STREAM,
                                  inputStreamPart.getFileName());
          }
        }

        HttpEntity build = builder.build();
        apacheHttpMultipartRequest.setEntity(build);
        setRequestConfig(multiPartRequest, apacheHttpMultipartRequest);
        multiPartRequest.getHeaders().forEach(apacheHttpMultipartRequest::setHeader);


        return apacheHttpMultipartRequest;
    }

    static Map<String, String> getHeaders(Header[] headers) {
        return Arrays.stream(headers)
                .collect(Collectors.toMap(NameValuePair::getName, NameValuePair::getValue, (key1, key2) -> key1));
    }

}
