package com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.ExtractPDF;

import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.util.JsonUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.pdfops.options.extractpdf.ExtractRenditionsElementType;
import org.apache.commons.collections4.CollectionUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

public class StructuredData {
    private static  final String DEFAULT_FILE_NAME = "structuredData";
    private static final String PATH_DELIMINATOR = "/";
    private static final Map<ExtractRenditionsElementType, String> renditionToPathMapping = new HashMap<ExtractRenditionsElementType, String>() {
        {
            put(ExtractRenditionsElementType.TABLES, "table");
            put(ExtractRenditionsElementType.FIGURES, "figure");
        }
    };
    private String fileFormat;
    private JSONObject data;
    private Map<String, ExtractRenditionsElementType> renditionContentNameToType;

    public static StructuredData deserializeData(InputStream inputStream, ExtensionMediaTypeMapping mediaType) {
        String fileFormat = mediaType.getExtension();
        if (mediaType.equals(ExtensionMediaTypeMapping.JSON)) {
            Map<String, Object> jsonContent = JsonUtil.deserializeJsonStream(inputStream, Map.class);
            return new StructuredData(fileFormat, new JSONObject(jsonContent));
        }
        return null;
    }

    public StructuredData(String fileFormat, JSONObject data) {
        this.fileFormat = fileFormat;
        this.data = data;
        preprocessData();
    }

    public JSONObject getData() {
        return data;
    }

    public String getFileFormat() {
        return fileFormat;
    }

    public String getFileName() {
        return PathUtil.getFileNameWithExtension(DEFAULT_FILE_NAME, fileFormat);
    }

    public ExtractRenditionsElementType getRenditionElementType(String contentName) {
        if (renditionContentNameToType != null) {
            return renditionContentNameToType.get(contentName);
        }
        return null;
    }

    public void preprocessData() {
        JSONArray elements = new JSONArray();
        if (data.has("elements")) {
            elements = data.getJSONArray("elements");
        }
        for (Object object : elements) {
            JSONObject jsonObject = (JSONObject) object;
            updateElementsPath(jsonObject);
            if (jsonObject.has("Kids")) {
                JSONArray kidJsonArray = jsonObject.getJSONArray("Kids");
                for (Object kidJsonElement : kidJsonArray) {
                    updateElementsPath((JSONObject) kidJsonElement);
                }
            }
        }
    }

    public void updateElementsPath(JSONObject jsonElementData) {
        ExtractRenditionsElementType renditionType = pdfRenditionIdentifier(jsonElementData);
        if (renditionType != null) {
            JSONArray fileNames = jsonElementData.getJSONArray("filePaths");
            for (int j = 0; j < fileNames.length(); j++) {
                updateElementContentNameToType(fileNames.getString(j), renditionType);
                fileNames.put(j, PathUtil.getSubPath(renditionType.toString().toLowerCase(), fileNames.getString(j)));
            }
        }
    }

    private void updateElementContentNameToType(String name, ExtractRenditionsElementType extractElementType) {
        if (renditionContentNameToType == null) {
            renditionContentNameToType = new HashMap<>();
        }
        renditionContentNameToType.put(name, extractElementType);
    }

    private static ExtractRenditionsElementType pdfRenditionIdentifier(JSONObject pdfElement) {
        if (isRenditionElement(pdfElement)) {
            String path = pdfElement.getString("Path");
            String[] splitPath = path.split(PATH_DELIMINATOR);
            String identifier = splitPath[splitPath.length - 1];
            for (Map.Entry<ExtractRenditionsElementType, String> mapping : renditionToPathMapping.entrySet()) {
                if (identifier.toLowerCase().startsWith(mapping.getValue().toLowerCase())) {
                    return mapping.getKey();
                }
            }
        }
        return null;
    }

    private static boolean isRenditionElement(JSONObject pdfElement) {
        if (pdfElement != null && pdfElement.has("filePaths") && pdfElement.has("Path") &&
                CollectionUtils.isNotEmpty(pdfElement.getJSONArray("filePaths").toList()) &&
                StringUtil.isNotBlank(pdfElement.getString("Path"))) {
            return true;
        }
        return false;
    }
}
