/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal;

import com.adobe.pdfservices.operation.internal.auth.Authenticator;
import com.adobe.pdfservices.operation.internal.auth.AuthenticatorFactory;
import com.adobe.pdfservices.operation.internal.auth.ServiceAccountCredentialsWithUri;
import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.internal.http.HttpRequest;
import com.adobe.pdfservices.operation.internal.http.HttpRequestConfig;
import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.ClientConfig;
import com.adobe.pdfservices.operation.auth.Credentials;
import com.adobe.pdfservices.operation.auth.ServiceAccountCredentials;
import com.adobe.pdfservices.operation.internal.auth.SessionTokenAuthenticator;

public class InternalExecutionContext extends ExecutionContext {

    private InternalClientConfig clientConfig;

    private Authenticator authenticator;

    private CPFServiceRequestContext cpfServiceRequestContext;

    public InternalExecutionContext(Credentials credentials, ClientConfig clientConfig) {
        if (credentials instanceof ServiceAccountCredentials) {
            String cpfOpsCreateUri=((ServiceAccountCredentialsWithUri) credentials).getCpfOpsCreateUri();
            if (clientConfig instanceof InternalClientConfig)
                this.clientConfig = (InternalClientConfig) clientConfig;
            else
                this.clientConfig = new InternalClientConfig();
            this.clientConfig.validate();
            this.authenticator = AuthenticatorFactory.getAuthenticator(credentials);
            this.cpfServiceRequestContext
                    = new CPFServiceRequestContext(
                    (cpfOpsCreateUri!=null)?cpfOpsCreateUri:this.clientConfig.getOpsCreateUri()
            );
        } else {
            throw new IllegalArgumentException("Invalid ClientContext provided as argument");
        }
    }


    public InternalExecutionContext(String clientId, String accessToken, String cpfUri, ClientConfig clientConfig) {
        this.authenticator = new SessionTokenAuthenticator(clientId, accessToken);
        if (clientConfig instanceof InternalClientConfig)
            this.clientConfig = (InternalClientConfig) clientConfig;
        else
            this.clientConfig = new InternalClientConfig();

        this.clientConfig.validate();

        this.cpfServiceRequestContext = new CPFServiceRequestContext(
                (cpfUri!=null) ? cpfUri : this.clientConfig.getOpsCreateUri());
    }


    public InternalClientConfig getClientConfig() {
        return clientConfig;
    }

    public HttpRequest getBaseRequestFromRequestContext(RequestKey requestKey) {
        HttpRequest baseRequest = cpfServiceRequestContext.getBaseRequest(requestKey);
        return baseRequest.withAuthenticator(authenticator).withConfig(new HttpRequestConfig(clientConfig));
    }

    public void validate() {
        this.clientConfig.validate();
        if (this.authenticator == null) {
            throw new IllegalStateException("Authentication not initialized in the provided context");
        }
    }
}
