/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.util.Objects;

import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.io.FileRef;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileRefImpl extends FileRef {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileRef.class);

    private String mediaType;
    private String localSource;
    private URL sourceUrl;
    private String name;
    private InputStream inputStream;
    private String extension;
    private byte[] fileByteArray;

    public FileRefImpl(String localSource, String mediaType) {
        Objects.requireNonNull(localSource, "Source file path must not be null");
        Objects.requireNonNull(mediaType, "Source file mediaType must not be null");
        this.localSource = localSource;
        this.mediaType = mediaType;
        this.name = PathUtil.getFileName(localSource);
        initializeExtension();
    }

    public FileRefImpl(InputStream inputStream, String mediaType) {
        Objects.requireNonNull(inputStream, "Source inputStream must not be null");
        Objects.requireNonNull(mediaType, "Source file mediaType must not be null");
        this.inputStream = inputStream;
        this.mediaType = mediaType;
        initializeExtension();
    }

    public FileRefImpl(String localSource) {
        Objects.requireNonNull(localSource, "Source file path must not be null");
        this.localSource = localSource;
        this.name = PathUtil.getFileName(localSource);
        this.extension = PathUtil.getExtension(localSource);
        ExtensionMediaTypeMapping mediaTypeMapping = ExtensionMediaTypeMapping.getFromExtension(this.extension);
        this.mediaType = mediaTypeMapping == null ? null : mediaTypeMapping.getMediaType();

    }

    public FileRefImpl(URL url) {
        Objects.requireNonNull(url, "Source file URL must not be null");
        this.sourceUrl = url;
    }

    private void initializeExtension() {
        ExtensionMediaTypeMapping mediaTypeMapping = ExtensionMediaTypeMapping.getFromMimeType(this.getMediaType());
        if (mediaTypeMapping != null) {
            this.extension = mediaTypeMapping.getExtension();
        }
    }

    public String getLocalPath() {
        return localSource;
    }

    public String getName() {
        return name;
    }

    @Override
    public boolean equals(java.lang.Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        FileRefImpl fileRefImpl = (FileRefImpl) o;
        return Objects.equals(localSource, fileRefImpl.localSource) &&
                Objects.equals(extension, fileRefImpl.extension) &&
                Objects.equals(mediaType, fileRefImpl.mediaType) &&
                Objects.equals(inputStream, fileRefImpl.inputStream) &&
                Objects.equals(sourceUrl, fileRefImpl.sourceUrl) &&
                Objects.equals(name, fileRefImpl.name);
    }

    @Override
    public int hashCode() {
        return Objects.hash(localSource, extension, mediaType, inputStream, name);
    }

    public InputStream getAsStream() throws FileNotFoundException {
        if (localSource != null) {
            return new FileInputStream(localSource);
        } else if (inputStream != null) {
            return inputStream;
        }
        return null;
    }

    public byte[] getByteArray() throws IOException {
        if (fileByteArray == null) {
            InputStream is = this.getAsStream();
            try {
                fileByteArray = IOUtils.toByteArray(is);
            }
            finally {
                if (StringUtil.isNotBlank(this.getLocalPath())) {
                    //close stream if the file is provided from local source
                    is.close();
                }
            }
        }
        return fileByteArray;
    }


    public String getExtension() {
        return extension;
    }

    public URL getSourceURL() {
        return sourceUrl;
    }

    private void moveFile(String targetLocation) throws IOException {
        if (isOperationResult(this.localSource)) {

            LOGGER.info("Moving file at {} to target {}", this.localSource, targetLocation);
            String targetFileName = PathUtil.getFileNameWithExtension(PathUtil.getBaseName(targetLocation),
                    getExtension());
            String destinationPath = String.format("%s%s", PathUtil.getFullPath(targetLocation), targetFileName);
            FileUtil.moveFile(getLocalPath(), destinationPath);
        } else {
            LOGGER.error("Invalid use of saveAs(). Method invoked on FileRef instance which does not point to an operation result");
            throw new UnsupportedOperationException("Method saveAs only allowed on operation results");
        }
    }

    private boolean isOperationResult(String localSource) {
        return localSource != null && localSource.contains(PathUtil.getTemporaryDirectoryPath());
    }

    private void moveFile(OutputStream outputStream) throws IOException {
        if (isOperationResult(this.localSource)) {
            LOGGER.info("Writing file at {} to output stream", this.localSource);
            FileUtil.moveFileToStream(this.localSource, outputStream);
        } else {
            LOGGER.error("Invalid use of saveAs(). Method invoked on FileRef instance which does not point to an operation result");
            throw new UnsupportedOperationException("Method saveAs only allowed on operation results");
        }
    }

    public String getMediaType() {
        return mediaType;
    }

    @Override
    public void saveAs(String targetLocation) throws IOException {
        Objects.requireNonNull(targetLocation, "Target location must not be null");
        this.moveFile(targetLocation);
    }


    @Override
    public void saveAs(OutputStream outputStream) throws IOException {
        Objects.requireNonNull(outputStream, "OutputStream must not be null");
        this.moveFile(outputStream);

    }

    public FileRefImpl getCopy() throws IOException {
        byte[] byteArray  = Objects.nonNull(fileByteArray) ? fileByteArray : getByteArray();
        InputStream targetStream = new ByteArrayInputStream(byteArray);
        InputStream internalStream = new ByteArrayInputStream(byteArray);
        this.inputStream = internalStream;
        return new FileRefImpl(targetStream, this.mediaType);
    }
}
