package cielo.printer.client

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.graphics.Bitmap
import android.os.IBinder
import cielo.printer.client.di.contract.PrinterClientContract
import cielo.printer.service.IPrinterCommunicationService
import cielo.printer.service.IPrinterCommunicationServiceCallback
import cielo.printer.service.IPrinterOperationCallback
import io.reactivex.rxkotlin.subscribeBy
import io.reactivex.subjects.BehaviorSubject
import javax.inject.Inject

open class PrinterClientImpl @Inject
constructor(val context: Context) : PrinterClientContract {

    private var intent: Intent? = null

    private var printerServiceSubject: BehaviorSubject<IPrinterCommunicationService>?
            = BehaviorSubject.create<IPrinterCommunicationService>()

    private var printerListener: PrinterClientContract.PrinterServiceListener? = null

    private val serviceCallback = object : IPrinterCommunicationServiceCallback.Stub() {

        override fun onSuccess() {
            printerListener?.onSuccess()
        }

        override fun onError(code: Int, msg: String?) {
            printerListener?.onError(Exception(msg))
        }

        override fun onWithoutPaper() {
            printerListener?.onWithoutPaper()
        }

    }

    private val serviceConnection = object : ServiceConnection {
        override fun onServiceDisconnected(name: ComponentName) {
            printerServiceSubject?.onComplete()
        }

        override fun onServiceConnected(name: ComponentName, service: IBinder) {
            val printerService: IPrinterCommunicationService?
                    = IPrinterCommunicationService.Stub.asInterface(service)
            if (printerService != null) {
                printerServiceSubject?.onNext(printerService)
            }
            setCallback()
        }
    }

    init {
        prepareServiceIntent()
    }

    private fun prepareServiceIntent() {
        this.intent = Intent()
        this.intent!!.`package` = "cielo.printer.manager"
        this.intent!!.action = "PRINT"
    }

    override fun bind() {
        context.startService(intent)
        context.bindService(intent, serviceConnection, Context.BIND_AUTO_CREATE)
    }

    override fun unbind() {
        context.stopService(intent)
        context.unbindService(serviceConnection)
    }

    override fun printText(string: String) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printText(string)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun printText(string: Array<String>, style: Map<String, Int>) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printColumnsTextWithAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun printText(string: Array<String>, style: List<Map<String, Int>>) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printColumnsTextWithListAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun printText(string: String, style: Map<String, Int>) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printTextWithAttributes(string, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun printBitmap(bitmap: Bitmap) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printBitmap(bitmap)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun printBitmap(bitmap: Bitmap, style: Map<String, Int>) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.printBitmapWithAttributes(bitmap, style)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun setPrinterListener(printerListener: PrinterClientContract.PrinterServiceListener) {
        this.printerListener = printerListener
    }

    private fun setCallback() {
        printerServiceSubject?.subscribeBy(onNext = {
            it.setCallback(serviceCallback)
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun restartCounterLinesPrinted(printerListener: PrinterClientContract.PrinterOperationListener) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.restartCounterLinesPrinted(prepareOperationListener(printerListener))
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    override fun getCounterLinesPrinted(printerListener: PrinterClientContract.PrinterOperationListener) {
        printerServiceSubject?.subscribeBy(onNext = {
            it.getCounterLinesPrinted(prepareOperationListener(printerListener))
        }, onError = {
            this.printerListener?.onError(it)
        })
    }

    private fun prepareOperationListener(printerListener: PrinterClientContract.PrinterOperationListener): IPrinterOperationCallback.Stub {
        return object : IPrinterOperationCallback.Stub() {
            override fun onSuccess() {
                printerListener.onSuccess()
            }

            override fun onResult(result: Long) {
                printerListener.onResult(result)
            }

            override fun onError(code: Int, msg: String?) {
                printerListener.onError(Exception(msg))
            }
        }
    }

}