package cc.livvy.framework.http;

import android.content.Intent;
import android.net.Uri;
import android.os.Handler;
import android.os.Message;
import android.support.annotation.Nullable;
import android.util.Log;
import android.view.View;
import android.widget.Toast;

import com.squareup.okhttp.Request;
import com.squareup.okhttp.Response;

import org.json.JSONObject;

import java.io.IOException;
import java.net.ConnectException;
import java.util.HashMap;
import java.util.concurrent.TimeUnit;

import cc.livvy.framework.base.BaseApp;
import cc.livvy.framework.data.encrypt.EncryptHandle;
import cc.livvy.framework.data.model.DataResult;
import cc.livvy.framework.http.callback.Callback;
import cc.livvy.framework.log.LogLevel;
import cc.livvy.framework.manager.ActivityManager;
import cc.livvy.framework.manager.SplashManager;
import cc.livvy.framework.util.DeviceUtil;
import cc.livvy.framework.widget.AlertDialog;

/**
 * 用于网络访问唯一对外使用的方式
 * <p>
 * Created by livvy on 16-2-16.
 */
public class UnifyHttpClient {

    private static final int HTTP_STATUS_WITHOUT_PERMISSION = -4;//无权限访问
    private static final int HTTP_STATUS_UPGRADE  = -2; //强制升级
    private static final int HTTP_STATUS_NETWORK_ERROR = 10001; //网络问题，导致连接超时

    private static final int HTTP_CONNECT_ERROR_UI_REFRESH  = 3;

    public static LogLevel mLogLevel = LogLevel.FULL;  //网络访问的log是否打印
    private static UIHandler mHandler = new UIHandler();

    private static UnifyHttpClient instance;

    private static OkHttpUtils mHttpUtils;

    public UnifyHttpClient() {
        mHttpUtils = new OkHttpUtils();
    }

    public static UnifyHttpClient getInstance() {
        if (null == instance) {
            instance = new UnifyHttpClient();
        }
        return instance;
    }

    public UnifyHttpClient setHttpLogLevel(LogLevel level) {
        mLogLevel = level;
        if (null == instance) {
            throw new NullPointerException("UnifyHttpClient is null,UnifyHttpClient must init");
        }
        return instance;
    }

    public UnifyHttpClient setConnectTimeout(long time) {
        mHttpUtils.getOkHttpClient().setConnectTimeout(time, TimeUnit.MILLISECONDS);
        if (null == instance) {
            throw new NullPointerException("UnifyHttpClient is null,UnifyHttpClient must init");
        }
        return instance;
    }

    /**
     * 进行网络访问的唯一方式。因为目前项目需求统一为post方式访问
     *
     * @param url      访问路径url
     * @param params   访问的参数
     * @param callback 异步方式（不输入的话则为同步方式）
     */
    public static DataResult execute(String url, HashMap<String, String> params, @Nullable UnifyHttpCallback callback) {
        if (null == mHttpUtils) {
            throw new NullPointerException("UnifyHttpClient is null,UnifyHttpClient must init");
        }
        //判断是否需要打印网络访问的参数
        if (mLogLevel == LogLevel.FULL) {
            StringBuilder str = new StringBuilder();
            for (HashMap.Entry<String, String> entry : params.entrySet()) {
                if (str.length() > 1) {
                    str.append("&");
                }
                str.append(entry.getKey());
                str.append("=");
                str.append(entry.getValue());
            }
            Log.i("httpRequest", "========================params==================================");
            Log.i("httpRequest", "访问的地址: " + url);
            Log.i("httpRequest", "访问的参数: " + str.toString());
            Log.i("httpRequest", "========================params==================================");
        }
        //同步方式
        if (null == callback) {
            return execute(url, params, true);
        } else {
            //异步方式
            execute(url, params, callback, true);
            return null;
        }
    }

    private static DataResult execute(String url, HashMap<String, String> params, boolean isCipher) {
        DataResult result;
        try {
            Response response;
            response = OkHttpUtils.post()
                    .url(url)
                    .params(getEncryptParams(params, isCipher)) //判断是否需要加密
                    .build()
                    .execute();
            result = processResponse(response);
        } catch (IOException e) {
            result = new DataResult();
            if (e instanceof ConnectException) {
                Message message = new Message();
                message.what = HTTP_CONNECT_ERROR_UI_REFRESH;
                mHandler.sendMessage(message);
                result.message = "您的网络有些问题哦";
            }
        }
        return result;
    }

    private static void execute(String url, HashMap<String, String> params, UnifyHttpCallback callback, boolean isCipher) {
        OkHttpUtils.post()
                .url(url)
                .params(getEncryptParams(params, isCipher))//判断是否需要加密
                .build()
                .execute(callback);
    }

    private static DataResult processResponse(Response response) throws IOException {
        if (null == response) {
            throw new NullPointerException("Response is null");
        }
        DataResult result;
        switch (response.code()) {
            case 200://正常数据展示
                try {
                    //对返回的数据进行解密
                    String resultStrEncrypt = response.body().string();
                    String resultStr = EncryptHandle.decrypt(resultStrEncrypt, DeviceUtil.getUUID(), DeviceUtil.getUUID());
                    //是否打印返回的数据
                    if (mLogLevel == LogLevel.FULL) {
                        Log.i("===请求路径:===", response.request().urlString());
                        Log.i("===请求返回:===", resultStr);
                        Log.i("===请求code:===", response.code() + "");
                    }
                    JSONObject jsonObject = new JSONObject(resultStr);
                    result = new DataResult(jsonObject);
                    if (result.status == HTTP_STATUS_WITHOUT_PERMISSION) {
//                        //无权限访问，需要重新登陆
                        String mSplashAction = SplashManager.getSplashAction();
                        if (!mSplashAction.equals("")) {
                            result.hasError = true;
                            result.message = "呀，好像被拒绝了哦,点击重新加载试试 >_<";

                            Intent intent = new Intent(mSplashAction);
                            intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                            BaseApp.getAppContext().startActivity(intent);
                        } else {
                            throw new NullPointerException("SplashManager is null,SplashManager must init");
                        }
                    }else if(result.status == HTTP_STATUS_UPGRADE){
                        //强制升级
                        Message message = new Message();
                        message.what = HTTP_STATUS_UPGRADE;
                        mHandler.sendMessage(message);
                    }
                } catch (Throwable e) {
                    result = new DataResult();
                    result.hasError = true;
                    result.localError = true;
                    result.message = "囧，解析数据失败 >_<";
                }
                break;
            case 404:
                result = new DataResult();
                break;
            case 500:
                result = new DataResult();
                break;
            default://非正常数据
                result = new DataResult();
                result.hasError = true;
                result.localError = true;
                result.message = "囧，获取数据出现问题了 >_<";
                break;
        }
        return result;
    }


    /**
     * 对参数进行加密处理
     * 遍历获取业务需要传递的参数，拼接成url，然后统一进行加密处理。最后存放在key:body的map中。
     *
     * @param params 包含通用的参数的params
     */
    private static HashMap<String, String> getEncryptParams(HashMap<String, String> params, boolean isCipher) {
        if (!isCipher) {
            return params;
        }
        //因为遍历过程中进行remove操作会导致ConcurrentModificationException异常。
        //所以将通用的参数保存在新的map中，方便使用
        HashMap<String, String> mEncryptParams = new HashMap<>();
        //用于存放拼接加密的参数url
        StringBuilder str = new StringBuilder();
        //遍历params
        for (HashMap.Entry<String, String> entry : params.entrySet()) {
            //过滤通用的参数,然后将业务参数进行拼接
            if (!entry.getKey().contains("partner") && (!entry.getKey().contains("uuid"))
                    && (!entry.getKey().contains("guid")) && (!entry.getKey().contains("ver"))
                    && (!entry.getKey().contains("productname"))) {

                if (str.length() > 1) {
                    str.append("&");
                }
                str.append(entry.getKey());
                str.append("=");
                str.append(entry.getValue());
            } else {
                mEncryptParams.put(entry.getKey(), entry.getValue());
            }
        }
        //加密后放入body中。
        if (str.length() > 1) {
            mEncryptParams.put("eyBody", EncryptHandle.encrypt(str.toString(), DeviceUtil.getUUID(), DeviceUtil.getUUID()));
        }
        return mEncryptParams;
    }

    static class UIHandler extends Handler {

        @Override
        public void handleMessage(Message msg) {
            switch (msg.what) {
                //连接超时处理
                case HTTP_CONNECT_ERROR_UI_REFRESH:
                    Toast.makeText(BaseApp.getAppContext(), "请检查您的网络是否正常", Toast.LENGTH_LONG).show();
                    break;
                //强制升级处理
                case HTTP_STATUS_UPGRADE:
                    final AlertDialog dialog = new AlertDialog(ActivityManager.getCurrentActivity());
                    dialog.setTitle("版本过低，请下载最新版本");
                    dialog.setActionButton("确定", new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            /**
                             * 需要app升级时会跳转到下载app的页面（使用浏览器）
                             */
                            Uri uri = Uri.parse("http://m.ekuaizhi.com/appdown");
                            Intent intent = new  Intent(Intent.ACTION_VIEW, uri);
                            ActivityManager.getCurrentActivity().startActivity(intent);
                            dialog.dismiss();
                            ActivityManager.finishAllActivities();
                        }
                    }).setCancelButton("取消", new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            dialog.dismiss();
                            ActivityManager.finishAllActivities();
                        }
                    });
                    dialog.setCanceledOnTouchOutside(false);
                    dialog.show();
                    break;
            }
        }
    }

    public static abstract class UnifyHttpCallback extends Callback<DataResult> {
        @Override
        public DataResult parseNetworkResponse(Response response) throws IOException {
            DataResult result;
            result = processResponse(response);
            return result;
        }

        @Override
        public void onError(Request request, Exception e) {
            onErrors(request, e);
        }

        @Override
        public DataResult onErrors(Request request, Exception e) {
            if (mLogLevel == LogLevel.FULL) {
                Log.i("===HttpError===", "url:" + request.url().toString());
                Log.i("===HttpError===", "body:" + request.body().toString());
                Log.i("===HttpError===", "Exception:" + e.toString());
            }

            if (e instanceof ConnectException) {
                Message message = new Message();
                message.what = HTTP_CONNECT_ERROR_UI_REFRESH;
                mHandler.sendMessage(message);
            }

            DataResult result = new DataResult();
            result.hasError = true;
            result.status   = HTTP_STATUS_NETWORK_ERROR;
            result.message  = "呀，网络访问失败了啊 >_<";
            return result;
        }
    }

}
