/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal.audio;

import android.content.Context;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioRecord;
import android.media.AudioRecordingConfiguration;
import android.media.AudioTimestamp;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.RequiresPermission;
import androidx.camera.core.Logger;
import androidx.camera.video.internal.audio.AudioSettings;
import androidx.camera.video.internal.audio.AudioStream;
import androidx.camera.video.internal.audio.AudioUtils;
import androidx.camera.video.internal.compat.Api23Impl;
import androidx.camera.video.internal.compat.Api24Impl;
import androidx.camera.video.internal.compat.Api29Impl;
import androidx.camera.video.internal.compat.Api31Impl;
import androidx.camera.video.internal.compat.quirk.AudioTimestampFramePositionIncorrectQuirk;
import androidx.camera.video.internal.compat.quirk.DeviceQuirks;
import androidx.core.util.Preconditions;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

@RequiresApi(value=21)
public class AudioStreamImpl
implements AudioStream {
    private static final String TAG = "AudioStreamImpl";
    final AudioRecord mAudioRecord;
    private final AudioSettings mSettings;
    private final AtomicBoolean mIsReleased = new AtomicBoolean(false);
    private final AtomicBoolean mIsStarted = new AtomicBoolean(false);
    private final AtomicReference<Boolean> mNotifiedSilenceState = new AtomicReference<Object>(null);
    private final int mBufferSize;
    private final int mBytesPerFrame;
    @Nullable
    private AudioStream.AudioStreamCallback mAudioStreamCallback;
    @Nullable
    private Executor mCallbackExecutor;
    private long mTotalFramesRead;
    @Nullable
    private AudioManager.AudioRecordingCallback mAudioRecordingCallback;

    @RequiresPermission(value="android.permission.RECORD_AUDIO")
    public AudioStreamImpl(@NonNull AudioSettings settings, @Nullable Context attributionContext) throws IllegalArgumentException, AudioStream.AudioStreamException {
        if (!AudioStreamImpl.isSettingsSupported(settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat())) {
            throw new UnsupportedOperationException(String.format("The combination of sample rate %d, channel count %d and audio format %d is not supported.", settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat()));
        }
        this.mSettings = settings;
        this.mBytesPerFrame = settings.getBytesPerFrame();
        int minBufferSize = AudioStreamImpl.getMinBufferSize(settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat());
        Preconditions.checkState((minBufferSize > 0 ? 1 : 0) != 0);
        this.mBufferSize = minBufferSize * 2;
        if (Build.VERSION.SDK_INT >= 23) {
            AudioFormat audioFormatObj = new AudioFormat.Builder().setSampleRate(settings.getSampleRate()).setChannelMask(AudioUtils.channelCountToChannelMask(settings.getChannelCount())).setEncoding(settings.getAudioFormat()).build();
            AudioRecord.Builder audioRecordBuilder = Api23Impl.createAudioRecordBuilder();
            if (Build.VERSION.SDK_INT >= 31 && attributionContext != null) {
                Api31Impl.setContext(audioRecordBuilder, attributionContext);
            }
            Api23Impl.setAudioSource(audioRecordBuilder, settings.getAudioSource());
            Api23Impl.setAudioFormat(audioRecordBuilder, audioFormatObj);
            Api23Impl.setBufferSizeInBytes(audioRecordBuilder, this.mBufferSize);
            this.mAudioRecord = Api23Impl.build(audioRecordBuilder);
        } else {
            this.mAudioRecord = new AudioRecord(settings.getAudioSource(), settings.getSampleRate(), AudioUtils.channelCountToChannelConfig(settings.getChannelCount()), settings.getAudioFormat(), this.mBufferSize);
        }
        if (this.mAudioRecord.getState() != 1) {
            this.mAudioRecord.release();
            throw new AudioStream.AudioStreamException("Unable to initialize AudioRecord");
        }
    }

    @Override
    public void start() throws AudioStream.AudioStreamException {
        this.checkNotReleasedOrThrow();
        if (this.mIsStarted.getAndSet(true)) {
            return;
        }
        this.mAudioRecord.startRecording();
        if (this.mAudioRecord.getRecordingState() != 3) {
            this.mIsStarted.set(false);
            throw new AudioStream.AudioStreamException("Unable to start AudioRecord with state: " + this.mAudioRecord.getRecordingState());
        }
        this.mTotalFramesRead = 0L;
        this.mNotifiedSilenceState.set(null);
        boolean isSilenced = false;
        if (Build.VERSION.SDK_INT >= 29) {
            AudioRecordingConfiguration config = Api29Impl.getActiveRecordingConfiguration(this.mAudioRecord);
            isSilenced = config != null && Api29Impl.isClientSilenced(config);
        }
        this.notifySilenced(isSilenced);
    }

    @Override
    public void stop() {
        this.checkNotReleasedOrThrow();
        if (!this.mIsStarted.getAndSet(false)) {
            return;
        }
        this.mAudioRecord.stop();
        if (this.mAudioRecord.getRecordingState() != 1) {
            Logger.w((String)TAG, (String)("Failed to stop AudioRecord with state: " + this.mAudioRecord.getRecordingState()));
        }
    }

    @Override
    public void release() {
        if (this.mIsReleased.getAndSet(true)) {
            return;
        }
        if (Build.VERSION.SDK_INT >= 29 && this.mAudioRecordingCallback != null) {
            Api29Impl.unregisterAudioRecordingCallback(this.mAudioRecord, this.mAudioRecordingCallback);
        }
        this.mAudioRecord.release();
    }

    @Override
    @NonNull
    public AudioStream.PacketInfo read(@NonNull ByteBuffer byteBuffer) {
        this.checkNotReleasedOrThrow();
        this.checkStartedOrThrow();
        int sizeInBytes = this.mAudioRecord.read(byteBuffer, this.mBufferSize);
        long timestampNs = 0L;
        if (sizeInBytes > 0) {
            byteBuffer.limit(sizeInBytes);
            timestampNs = this.generatePresentationTimeNs();
            this.mTotalFramesRead += AudioUtils.sizeToFrameCount(sizeInBytes, this.mBytesPerFrame);
        }
        return AudioStream.PacketInfo.of(sizeInBytes, timestampNs);
    }

    @Override
    public void setCallback(@Nullable AudioStream.AudioStreamCallback callback, @Nullable Executor executor) {
        Preconditions.checkState((!this.mIsStarted.get() ? 1 : 0) != 0, (String)"AudioStream can not be started when setCallback.");
        this.checkNotReleasedOrThrow();
        Preconditions.checkArgument((callback == null || executor != null ? 1 : 0) != 0, (Object)"executor can't be null with non-null callback.");
        this.mAudioStreamCallback = callback;
        this.mCallbackExecutor = executor;
        if (Build.VERSION.SDK_INT >= 29) {
            if (this.mAudioRecordingCallback != null) {
                Api29Impl.unregisterAudioRecordingCallback(this.mAudioRecord, this.mAudioRecordingCallback);
            }
            if (callback == null) {
                return;
            }
            if (this.mAudioRecordingCallback == null) {
                this.mAudioRecordingCallback = new AudioRecordingApi29Callback();
            }
            Api29Impl.registerAudioRecordingCallback(this.mAudioRecord, executor, this.mAudioRecordingCallback);
        }
    }

    void notifySilenced(boolean isSilenced) {
        Executor executor = this.mCallbackExecutor;
        AudioStream.AudioStreamCallback callback = this.mAudioStreamCallback;
        if (executor != null && callback != null && !Objects.equals(this.mNotifiedSilenceState.getAndSet(isSilenced), isSilenced)) {
            executor.execute(() -> callback.onSilenceStateChanged(isSilenced));
        }
    }

    private long generatePresentationTimeNs() {
        long presentationTimeNs = -1L;
        if (Build.VERSION.SDK_INT >= 24 && !AudioStreamImpl.hasAudioTimestampQuirk()) {
            AudioTimestamp audioTimestamp = new AudioTimestamp();
            if (Api24Impl.getTimestamp(this.mAudioRecord, audioTimestamp, 0) == 0) {
                presentationTimeNs = AudioStreamImpl.computeInterpolatedTimeNs(this.mSettings.getSampleRate(), this.mTotalFramesRead, audioTimestamp);
            } else {
                Logger.w((String)TAG, (String)"Unable to get audio timestamp");
            }
        }
        if (presentationTimeNs == -1L) {
            presentationTimeNs = System.nanoTime();
        }
        return presentationTimeNs;
    }

    private void checkNotReleasedOrThrow() {
        Preconditions.checkState((!this.mIsReleased.get() ? 1 : 0) != 0, (String)"AudioStream has been released.");
    }

    private void checkStartedOrThrow() {
        Preconditions.checkState((boolean)this.mIsStarted.get(), (String)"AudioStream has not been started.");
    }

    public static boolean isSettingsSupported(int sampleRate, int channelCount, int audioFormat) {
        if (sampleRate <= 0 || channelCount <= 0) {
            return false;
        }
        return AudioStreamImpl.getMinBufferSize(sampleRate, channelCount, audioFormat) > 0;
    }

    private static boolean hasAudioTimestampQuirk() {
        return DeviceQuirks.get(AudioTimestampFramePositionIncorrectQuirk.class) != null;
    }

    private static long computeInterpolatedTimeNs(int sampleRate, long framePosition, @NonNull AudioTimestamp timestamp) {
        long frameDiff = framePosition - timestamp.framePosition;
        long compensateTimeInNanoSec = AudioUtils.frameCountToDurationNs(frameDiff, sampleRate);
        long resultInNanoSec = timestamp.nanoTime + compensateTimeInNanoSec;
        return resultInNanoSec < 0L ? 0L : resultInNanoSec;
    }

    private static int getMinBufferSize(int sampleRate, int channelCount, int audioFormat) {
        return AudioRecord.getMinBufferSize((int)sampleRate, (int)AudioUtils.channelCountToChannelConfig(channelCount), (int)audioFormat);
    }

    @RequiresApi(value=29)
    class AudioRecordingApi29Callback
    extends AudioManager.AudioRecordingCallback {
        AudioRecordingApi29Callback() {
        }

        public void onRecordingConfigChanged(List<AudioRecordingConfiguration> configs) {
            for (AudioRecordingConfiguration config : configs) {
                if (Api24Impl.getClientAudioSessionId(config) != AudioStreamImpl.this.mAudioRecord.getAudioSessionId()) continue;
                boolean isSilenced = Api29Impl.isClientSilenced(config);
                AudioStreamImpl.this.notifySilenced(isSilenced);
                break;
            }
        }
    }
}

