/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.core.internal.utils;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.BitmapRegionDecoder;
import android.graphics.Matrix;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.YuvImage;
import android.util.Rational;
import android.util.Size;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.core.ImageProxy;
import androidx.camera.core.Logger;
import androidx.core.util.Preconditions;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.Buffer;
import java.nio.ByteBuffer;

@RequiresApi(value=21)
public final class ImageUtil {
    private static final String TAG = "ImageUtil";
    public static final int DEFAULT_RGBA_PIXEL_STRIDE = 4;

    private ImageUtil() {
    }

    @NonNull
    public static Bitmap createBitmapFromPlane(@NonNull ImageProxy.PlaneProxy[] planes, int width, int height) {
        Preconditions.checkArgument((planes.length == 1 ? 1 : 0) != 0, (Object)"Expect a single plane");
        Preconditions.checkArgument((planes[0].getPixelStride() == 4 ? 1 : 0) != 0, (Object)"Expect pixelStride=4");
        Preconditions.checkArgument((planes[0].getRowStride() == 4 * width ? 1 : 0) != 0, (Object)"Expect rowStride=width*4");
        Bitmap bitmap = Bitmap.createBitmap((int)width, (int)height, (Bitmap.Config)Bitmap.Config.ARGB_8888);
        bitmap.copyPixelsFromBuffer((Buffer)planes[0].getBuffer());
        return bitmap;
    }

    @NonNull
    public static ByteBuffer createDirectByteBuffer(@NonNull Bitmap bitmap) {
        Preconditions.checkArgument((bitmap.getConfig() == Bitmap.Config.ARGB_8888 ? 1 : 0) != 0, (Object)"Only accept Bitmap with ARGB_8888 format for now.");
        ByteBuffer byteBuffer = ByteBuffer.allocateDirect(bitmap.getAllocationByteCount());
        bitmap.copyPixelsToBuffer((Buffer)byteBuffer);
        byteBuffer.rewind();
        return byteBuffer;
    }

    @NonNull
    public static float[] sizeToVertexes(@NonNull Size size) {
        return new float[]{0.0f, 0.0f, size.getWidth(), 0.0f, size.getWidth(), size.getHeight(), 0.0f, size.getHeight()};
    }

    public static float min(float value1, float value2, float value3, float value4) {
        return Math.min(Math.min(value1, value2), Math.min(value3, value4));
    }

    @NonNull
    public static Rational getRotatedAspectRatio(@IntRange(from=0L, to=359L) int rotationDegrees, @NonNull Rational aspectRatio) {
        if (rotationDegrees == 90 || rotationDegrees == 270) {
            return ImageUtil.inverseRational(aspectRatio);
        }
        return new Rational(aspectRatio.getNumerator(), aspectRatio.getDenominator());
    }

    @NonNull
    public static byte[] jpegImageToJpegByteArray(@NonNull ImageProxy image) {
        if (image.getFormat() != 256) {
            throw new IllegalArgumentException("Incorrect image format of the input image proxy: " + image.getFormat());
        }
        ImageProxy.PlaneProxy[] planes = image.getPlanes();
        ByteBuffer buffer = planes[0].getBuffer();
        byte[] data = new byte[buffer.capacity()];
        buffer.rewind();
        buffer.get(data);
        return data;
    }

    @NonNull
    public static byte[] jpegImageToJpegByteArray(@NonNull ImageProxy image, @NonNull Rect cropRect, @IntRange(from=1L, to=100L) int jpegQuality) throws CodecFailedException {
        if (image.getFormat() != 256) {
            throw new IllegalArgumentException("Incorrect image format of the input image proxy: " + image.getFormat());
        }
        byte[] data = ImageUtil.jpegImageToJpegByteArray(image);
        data = ImageUtil.cropJpegByteArray(data, cropRect, jpegQuality);
        return data;
    }

    @NonNull
    public static byte[] yuvImageToJpegByteArray(@NonNull ImageProxy image, @Nullable Rect cropRect, @IntRange(from=1L, to=100L) int jpegQuality) throws CodecFailedException {
        if (image.getFormat() != 35) {
            throw new IllegalArgumentException("Incorrect image format of the input image proxy: " + image.getFormat());
        }
        return ImageUtil.nv21ToJpeg(ImageUtil.yuv_420_888toNv21(image), image.getWidth(), image.getHeight(), cropRect, jpegQuality);
    }

    @NonNull
    public static byte[] yuv_420_888toNv21(@NonNull ImageProxy image) {
        ImageProxy.PlaneProxy yPlane = image.getPlanes()[0];
        ImageProxy.PlaneProxy uPlane = image.getPlanes()[1];
        ImageProxy.PlaneProxy vPlane = image.getPlanes()[2];
        ByteBuffer yBuffer = yPlane.getBuffer();
        ByteBuffer uBuffer = uPlane.getBuffer();
        ByteBuffer vBuffer = vPlane.getBuffer();
        yBuffer.rewind();
        uBuffer.rewind();
        vBuffer.rewind();
        int ySize = yBuffer.remaining();
        int position = 0;
        byte[] nv21 = new byte[ySize + image.getWidth() * image.getHeight() / 2];
        for (int row = 0; row < image.getHeight(); ++row) {
            yBuffer.get(nv21, position, image.getWidth());
            position += image.getWidth();
            yBuffer.position(Math.min(ySize, yBuffer.position() - image.getWidth() + yPlane.getRowStride()));
        }
        int chromaHeight = image.getHeight() / 2;
        int chromaWidth = image.getWidth() / 2;
        int vRowStride = vPlane.getRowStride();
        int uRowStride = uPlane.getRowStride();
        int vPixelStride = vPlane.getPixelStride();
        int uPixelStride = uPlane.getPixelStride();
        byte[] vLineBuffer = new byte[vRowStride];
        byte[] uLineBuffer = new byte[uRowStride];
        for (int row = 0; row < chromaHeight; ++row) {
            vBuffer.get(vLineBuffer, 0, Math.min(vRowStride, vBuffer.remaining()));
            uBuffer.get(uLineBuffer, 0, Math.min(uRowStride, uBuffer.remaining()));
            int vLineBufferPosition = 0;
            int uLineBufferPosition = 0;
            for (int col = 0; col < chromaWidth; ++col) {
                nv21[position++] = vLineBuffer[vLineBufferPosition];
                nv21[position++] = uLineBuffer[uLineBufferPosition];
                vLineBufferPosition += vPixelStride;
                uLineBufferPosition += uPixelStride;
            }
        }
        return nv21;
    }

    @NonNull
    private static byte[] cropJpegByteArray(@NonNull byte[] data, @NonNull Rect cropRect, @IntRange(from=1L, to=100L) int jpegQuality) throws CodecFailedException {
        Bitmap bitmap;
        try {
            BitmapRegionDecoder decoder = BitmapRegionDecoder.newInstance((byte[])data, (int)0, (int)data.length, (boolean)false);
            bitmap = decoder.decodeRegion(cropRect, new BitmapFactory.Options());
            decoder.recycle();
        }
        catch (IllegalArgumentException e) {
            throw new CodecFailedException("Decode byte array failed with illegal argument." + e, CodecFailedException.FailureType.DECODE_FAILED);
        }
        catch (IOException e) {
            throw new CodecFailedException("Decode byte array failed.", CodecFailedException.FailureType.DECODE_FAILED);
        }
        if (bitmap == null) {
            throw new CodecFailedException("Decode byte array failed.", CodecFailedException.FailureType.DECODE_FAILED);
        }
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        boolean success = bitmap.compress(Bitmap.CompressFormat.JPEG, jpegQuality, (OutputStream)out);
        if (!success) {
            throw new CodecFailedException("Encode bitmap failed.", CodecFailedException.FailureType.ENCODE_FAILED);
        }
        bitmap.recycle();
        return out.toByteArray();
    }

    public static boolean isAspectRatioValid(@Nullable Rational aspectRatio) {
        return aspectRatio != null && aspectRatio.floatValue() > 0.0f && !aspectRatio.isNaN();
    }

    public static boolean isAspectRatioValid(@NonNull Size sourceSize, @Nullable Rational aspectRatio) {
        return aspectRatio != null && aspectRatio.floatValue() > 0.0f && ImageUtil.isCropAspectRatioHasEffect(sourceSize, aspectRatio) && !aspectRatio.isNaN();
    }

    @Nullable
    public static Rect computeCropRectFromAspectRatio(@NonNull Size sourceSize, @NonNull Rational aspectRatio) {
        if (!ImageUtil.isAspectRatioValid(aspectRatio)) {
            Logger.w(TAG, "Invalid view ratio.");
            return null;
        }
        int sourceWidth = sourceSize.getWidth();
        int sourceHeight = sourceSize.getHeight();
        float srcRatio = (float)sourceWidth / (float)sourceHeight;
        int cropLeft = 0;
        int cropTop = 0;
        int outputWidth = sourceWidth;
        int outputHeight = sourceHeight;
        int numerator = aspectRatio.getNumerator();
        int denominator = aspectRatio.getDenominator();
        if (aspectRatio.floatValue() > srcRatio) {
            outputHeight = Math.round((float)sourceWidth / (float)numerator * (float)denominator);
            cropTop = (sourceHeight - outputHeight) / 2;
        } else {
            outputWidth = Math.round((float)sourceHeight / (float)denominator * (float)numerator);
            cropLeft = (sourceWidth - outputWidth) / 2;
        }
        return new Rect(cropLeft, cropTop, cropLeft + outputWidth, cropTop + outputHeight);
    }

    @NonNull
    public static Rect computeCropRectFromDispatchInfo(@NonNull Rect surfaceCropRect, int surfaceToOutputDegrees, @NonNull Size dispatchResolution, int dispatchToOutputDegrees) {
        Matrix matrix = new Matrix();
        matrix.setRotate((float)(dispatchToOutputDegrees - surfaceToOutputDegrees));
        float[] vertexes = ImageUtil.sizeToVertexes(dispatchResolution);
        matrix.mapPoints(vertexes);
        float left = ImageUtil.min(vertexes[0], vertexes[2], vertexes[4], vertexes[6]);
        float top = ImageUtil.min(vertexes[1], vertexes[3], vertexes[5], vertexes[7]);
        matrix.postTranslate(-left, -top);
        matrix.invert(matrix);
        RectF dispatchCropRectF = new RectF();
        matrix.mapRect(dispatchCropRectF, new RectF(surfaceCropRect));
        dispatchCropRectF.sort();
        Rect dispatchCropRect = new Rect();
        dispatchCropRectF.round(dispatchCropRect);
        return dispatchCropRect;
    }

    private static byte[] nv21ToJpeg(@NonNull byte[] nv21, int width, int height, @Nullable Rect cropRect, @IntRange(from=1L, to=100L) int jpegQuality) throws CodecFailedException {
        ByteArrayOutputStream out;
        YuvImage yuv = new YuvImage(nv21, 17, width, height, null);
        boolean success = yuv.compressToJpeg(cropRect == null ? new Rect(0, 0, width, height) : cropRect, jpegQuality, (OutputStream)(out = new ByteArrayOutputStream()));
        if (!success) {
            throw new CodecFailedException("YuvImage failed to encode jpeg.", CodecFailedException.FailureType.ENCODE_FAILED);
        }
        return out.toByteArray();
    }

    private static boolean isCropAspectRatioHasEffect(@NonNull Size sourceSize, @NonNull Rational aspectRatio) {
        int denominator;
        int numerator;
        int sourceWidth = sourceSize.getWidth();
        int sourceHeight = sourceSize.getHeight();
        return sourceHeight != Math.round((float)sourceWidth / (float)(numerator = aspectRatio.getNumerator()) * (float)(denominator = aspectRatio.getDenominator())) || sourceWidth != Math.round((float)sourceHeight / (float)denominator * (float)numerator);
    }

    private static Rational inverseRational(@Nullable Rational rational) {
        if (rational == null) {
            return rational;
        }
        return new Rational(rational.getDenominator(), rational.getNumerator());
    }

    public static boolean shouldCropImage(@NonNull ImageProxy image) {
        return ImageUtil.shouldCropImage(image.getWidth(), image.getHeight(), image.getCropRect().width(), image.getCropRect().height());
    }

    public static boolean shouldCropImage(int sourceWidth, int sourceHeight, int cropRectWidth, int cropRectHeight) {
        return sourceWidth != cropRectWidth || sourceHeight != cropRectHeight;
    }

    public static final class CodecFailedException
    extends Exception {
        private FailureType mFailureType;

        CodecFailedException(@NonNull String message) {
            super(message);
            this.mFailureType = FailureType.UNKNOWN;
        }

        CodecFailedException(@NonNull String message, @NonNull FailureType failureType) {
            super(message);
            this.mFailureType = failureType;
        }

        @NonNull
        public FailureType getFailureType() {
            return this.mFailureType;
        }

        public static enum FailureType {
            ENCODE_FAILED,
            DECODE_FAILED,
            UNKNOWN;

        }
    }
}

