/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.commerce.account.internal.search;

import com.liferay.commerce.account.model.CommerceAccountGroup;
import com.liferay.commerce.account.service.CommerceAccountGroupLocalService;
import com.liferay.portal.kernel.dao.orm.IndexableActionableDynamicQuery;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.search.BaseIndexer;
import com.liferay.portal.kernel.search.BooleanQuery;
import com.liferay.portal.kernel.search.Document;
import com.liferay.portal.kernel.search.Field;
import com.liferay.portal.kernel.search.IndexWriterHelper;
import com.liferay.portal.kernel.search.Indexer;
import com.liferay.portal.kernel.search.SearchContext;
import com.liferay.portal.kernel.search.Summary;
import com.liferay.portal.kernel.search.filter.BooleanFilter;
import com.liferay.portal.kernel.util.GetterUtil;

import java.util.Locale;

import javax.portlet.PortletRequest;
import javax.portlet.PortletResponse;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Alessio Antonio Rendina
 */
@Component(enabled = false, immediate = true, service = Indexer.class)
public class CommerceAccountGroupIndexer
	extends BaseIndexer<CommerceAccountGroup> {

	public static final String CLASS_NAME =
		CommerceAccountGroup.class.getName();

	@Override
	public String getClassName() {
		return CLASS_NAME;
	}

	@Override
	public void postProcessSearchQuery(
			BooleanQuery searchQuery, BooleanFilter fullQueryBooleanFilter,
			SearchContext searchContext)
		throws Exception {

		addSearchTerm(searchQuery, searchContext, Field.ENTRY_CLASS_PK, false);
		addSearchTerm(searchQuery, searchContext, Field.NAME, false);
	}

	@Override
	protected void doDelete(CommerceAccountGroup commerceAccountGroup)
		throws Exception {

		deleteDocument(
			commerceAccountGroup.getCompanyId(),
			commerceAccountGroup.getCommerceAccountGroupId());
	}

	@Override
	protected Document doGetDocument(CommerceAccountGroup commerceAccountGroup)
		throws Exception {

		if (_log.isDebugEnabled()) {
			_log.debug(
				"Indexing commerce account group " + commerceAccountGroup);
		}

		Document document = getBaseModelDocument(
			CLASS_NAME, commerceAccountGroup);

		document.addText(Field.NAME, commerceAccountGroup.getName());
		document.addNumber(Field.TYPE, commerceAccountGroup.getType());

		if (_log.isDebugEnabled()) {
			_log.debug(
				"Document " + commerceAccountGroup + " indexed successfully");
		}

		return document;
	}

	@Override
	protected Summary doGetSummary(
		Document document, Locale locale, String snippet,
		PortletRequest portletRequest, PortletResponse portletResponse) {

		Summary summary = createSummary(
			document, Field.ENTRY_CLASS_PK, Field.NAME);

		summary.setMaxContentLength(200);

		return summary;
	}

	@Override
	protected void doReindex(CommerceAccountGroup commerceAccountGroup)
		throws Exception {

		_indexWriterHelper.updateDocument(
			getSearchEngineId(), commerceAccountGroup.getCompanyId(),
			getDocument(commerceAccountGroup), isCommitImmediately());
	}

	@Override
	protected void doReindex(String className, long classPK) throws Exception {
		doReindex(
			_commerceAccountGroupLocalService.getCommerceAccountGroup(classPK));
	}

	@Override
	protected void doReindex(String[] ids) throws Exception {
		long companyId = GetterUtil.getLong(ids[0]);

		reindexCommerceAccountGroups(companyId);
	}

	protected void reindexCommerceAccountGroups(long companyId)
		throws PortalException {

		final IndexableActionableDynamicQuery indexableActionableDynamicQuery =
			_commerceAccountGroupLocalService.
				getIndexableActionableDynamicQuery();

		indexableActionableDynamicQuery.setCompanyId(companyId);
		indexableActionableDynamicQuery.setPerformActionMethod(
			(CommerceAccountGroup commerceAccountGroup) -> {
				try {
					indexableActionableDynamicQuery.addDocuments(
						getDocument(commerceAccountGroup));
				}
				catch (PortalException portalException) {
					if (_log.isWarnEnabled()) {
						long commerceAccountGroupId =
							commerceAccountGroup.getCommerceAccountGroupId();

						_log.warn(
							"Unable to index commerce account group " +
								commerceAccountGroupId,
							portalException);
					}
				}
			});
		indexableActionableDynamicQuery.setSearchEngineId(getSearchEngineId());

		indexableActionableDynamicQuery.performActions();
	}

	private static final Log _log = LogFactoryUtil.getLog(
		CommerceAccountGroupIndexer.class);

	@Reference
	private CommerceAccountGroupLocalService _commerceAccountGroupLocalService;

	@Reference
	private IndexWriterHelper _indexWriterHelper;

}