/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.commerce.account.internal.permission;

import com.liferay.commerce.account.constants.CommerceAccountActionKeys;
import com.liferay.commerce.account.model.CommerceAccount;
import com.liferay.commerce.account.model.CommerceAccountOrganizationRel;
import com.liferay.commerce.account.permission.CommerceAccountPermission;
import com.liferay.commerce.account.service.CommerceAccountLocalService;
import com.liferay.commerce.account.service.CommerceAccountOrganizationRelLocalService;
import com.liferay.commerce.account.service.persistence.CommerceAccountOrganizationRelPK;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.model.Organization;
import com.liferay.portal.kernel.model.User;
import com.liferay.portal.kernel.security.auth.PrincipalException;
import com.liferay.portal.kernel.security.permission.ActionKeys;
import com.liferay.portal.kernel.security.permission.PermissionChecker;
import com.liferay.portal.kernel.service.OrganizationLocalService;
import com.liferay.portal.kernel.service.UserLocalService;
import com.liferay.portal.kernel.service.permission.PortalPermissionUtil;
import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.ListUtil;

import java.util.List;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Marco Leo
 * @author Alessio Antonio Rendina
 */
@Component(
	enabled = false, immediate = true, service = CommerceAccountPermission.class
)
public class CommerceAccountPermissionImpl
	implements CommerceAccountPermission {

	@Override
	public void check(
			PermissionChecker permissionChecker,
			CommerceAccount commerceAccount, String actionId)
		throws PortalException {

		if (!contains(permissionChecker, commerceAccount, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, CommerceAccount.class.getName(),
				commerceAccount.getCommerceAccountId(), actionId);
		}
	}

	@Override
	public void check(
			PermissionChecker permissionChecker, long commerceAccountId,
			String actionId)
		throws PortalException {

		if (!contains(permissionChecker, commerceAccountId, actionId)) {
			throw new PrincipalException.MustHavePermission(
				permissionChecker, CommerceAccount.class.getName(),
				commerceAccountId, actionId);
		}
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker,
			CommerceAccount commerceAccount, String actionId)
		throws PortalException {

		if (contains(
				permissionChecker, commerceAccount.getCommerceAccountId(),
				actionId)) {

			return true;
		}

		return false;
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker, long commerceAccountId,
			String actionId)
		throws PortalException {

		if (PortalPermissionUtil.contains(
				permissionChecker,
				CommerceAccountActionKeys.MANAGE_ALL_ACCOUNTS)) {

			return true;
		}

		CommerceAccount commerceAccount =
			_commerceAccountLocalService.getCommerceAccount(commerceAccountId);

		if (commerceAccount == null) {
			return false;
		}

		return _contains(permissionChecker, commerceAccount, actionId);
	}

	@Override
	public boolean contains(
			PermissionChecker permissionChecker, long[] commerceAccountIds,
			String actionId)
		throws PortalException {

		if (ArrayUtil.isEmpty(commerceAccountIds)) {
			return false;
		}

		for (long commerceAccountId : commerceAccountIds) {
			if (!contains(permissionChecker, commerceAccountId, actionId)) {
				return false;
			}
		}

		return true;
	}

	private boolean _contains(
			PermissionChecker permissionChecker,
			CommerceAccount commerceAccount, String actionId)
		throws PortalException {

		if (permissionChecker.isCompanyAdmin(commerceAccount.getCompanyId()) ||
			permissionChecker.isOmniadmin()) {

			return true;
		}

		if (actionId.equals(ActionKeys.UPDATE)) {
			return _containsUpdatePermission(
				commerceAccount, permissionChecker);
		}
		else if (actionId.equals(ActionKeys.VIEW)) {
			return _containsViewPermission(commerceAccount, permissionChecker);
		}
		else if (actionId.equals(
					CommerceAccountActionKeys.MANAGE_ORGANIZATIONS)) {

			return _containsManageOrganizationPermission(
				commerceAccount, permissionChecker);
		}

		if (PortalPermissionUtil.contains(
				permissionChecker,
				CommerceAccountActionKeys.MANAGE_AVAILABLE_ACCOUNTS)) {

			return true;
		}

		return permissionChecker.hasPermission(
			commerceAccount.getCommerceAccountGroupId(),
			CommerceAccount.class.getName(),
			commerceAccount.getCommerceAccountId(), actionId);
	}

	private boolean _containsManageOrganizationPermission(
			CommerceAccount commerceAccount,
			PermissionChecker permissionChecker)
		throws PortalException {

		if (commerceAccount.isPersonalAccount()) {
			return false;
		}

		return permissionChecker.hasPermission(
			commerceAccount.getCommerceAccountGroupId(),
			CommerceAccount.class.getName(),
			commerceAccount.getCommerceAccountId(),
			CommerceAccountActionKeys.MANAGE_ORGANIZATIONS);
	}

	private boolean _containsUpdatePermission(
			CommerceAccount commerceAccount,
			PermissionChecker permissionChecker)
		throws PortalException {

		if (commerceAccount.isPersonalAccount()) {
			return _hasOwnerPermission(commerceAccount, permissionChecker);
		}

		if (PortalPermissionUtil.contains(
				permissionChecker,
				CommerceAccountActionKeys.MANAGE_AVAILABLE_ACCOUNTS)) {

			return true;
		}

		return permissionChecker.hasPermission(
			commerceAccount.getCommerceAccountGroupId(),
			CommerceAccount.class.getName(),
			commerceAccount.getCommerceAccountId(), ActionKeys.UPDATE);
	}

	private boolean _containsViewPermission(
			CommerceAccount commerceAccount,
			PermissionChecker permissionChecker)
		throws PortalException {

		if (commerceAccount.isPersonalAccount()) {
			return _hasOwnerPermission(commerceAccount, permissionChecker);
		}

		if (PortalPermissionUtil.contains(
				permissionChecker,
				CommerceAccountActionKeys.MANAGE_AVAILABLE_ACCOUNTS)) {

			return true;
		}

		for (Organization organization :
				_getUserOrganizations(permissionChecker.getUserId())) {

			CommerceAccountOrganizationRel commerceAccountOrganizationRel =
				_commerceAccountOrganizationRelLocalService.
					fetchCommerceAccountOrganizationRel(
						new CommerceAccountOrganizationRelPK(
							commerceAccount.getCommerceAccountId(),
							organization.getOrganizationId()));

			if (commerceAccountOrganizationRel != null) {
				return true;
			}
		}

		return permissionChecker.hasPermission(
			commerceAccount.getCommerceAccountGroupId(),
			CommerceAccount.class.getName(),
			commerceAccount.getCommerceAccountId(), ActionKeys.VIEW);
	}

	private List<Organization> _getUserOrganizations(long userId)
		throws PortalException {

		List<Organization> organizations =
			_organizationLocalService.getUserOrganizations(userId);

		List<Organization> userOrganizations = ListUtil.copy(organizations);

		User user = _userLocalService.getUser(userId);

		for (Organization organization : organizations) {
			for (Organization curOrganization :
					_organizationLocalService.getOrganizations(
						user.getCompanyId(),
						organization.getTreePath() + "%")) {

				userOrganizations.add(curOrganization);
			}
		}

		return userOrganizations;
	}

	private boolean _hasOwnerPermission(
		CommerceAccount commerceAccount, PermissionChecker permissionChecker) {

		if (permissionChecker.getUserId() == commerceAccount.getUserId()) {
			return true;
		}

		return false;
	}

	@Reference
	private CommerceAccountLocalService _commerceAccountLocalService;

	@Reference
	private CommerceAccountOrganizationRelLocalService
		_commerceAccountOrganizationRelLocalService;

	@Reference
	private OrganizationLocalService _organizationLocalService;

	@Reference
	private UserLocalService _userLocalService;

}