/**
 * SPDX-FileCopyrightText: (c) 2025 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import ClayAlert from '@clayui/alert';
import {LanguagePicker, Provider} from '@clayui/core';
import ClayForm, {ClayInput} from '@clayui/form';
import ClayIcon from '@clayui/icon';
import {sub} from 'frontend-js-web';
import React, {useState} from 'react';

interface Props {
	category: TaxonomyCategory;
	defaultLanguageId: string;
	locales: any[];
	nameInputError: string;
	setCategory: Function;
	setNameInputError: Function;
	spritemap: string;
}

const EditCategoryGeneralInfoTab = ({
	category,
	defaultLanguageId,
	locales,
	nameInputError,
	setCategory,
	setNameInputError,
	spritemap,
}: Props) => {
	const [languageId, setLanguageId] = useState<string>(defaultLanguageId);

	const getLanguageLabel = (languageId: string) => {
		return languageId.replace('_', '-');
	};

	const onChangeName = (newName: string) => {
		if (newName) {
			setNameInputError('');
		}
		else {
			setNameInputError(
				sub(
					Liferay.Language.get('the-x-field-is-required'),
					Liferay.Language.get('name')
				)
			);
		}

		setCategory(() => ({
			...category,
			...(languageId === defaultLanguageId && {name: newName}),
			name_i18n: {
				...category.name_i18n,
				[getLanguageLabel(languageId)]: newName,
			},
		}));
	};

	const onChangeDescription = (newDescription: string) => {
		setCategory(() => ({
			...category,
			...(languageId === defaultLanguageId && {
				description: newDescription,
			}),
			description_i18n: {
				...category.description_i18n,
				[getLanguageLabel(languageId)]: newDescription,
			},
		}));
	};

	return (
		<div className="vertical-nav-content-wrapper">
			<ClayForm.Group className="c-gap-4 d-flex flex-column p-4">
				<div className="d-flex">
					<div className="autofit-col autofit-col-expand form-title">
						{Liferay.Language.get('basic-info')}
					</div>

					<div className="autofit-col" style={{width: 'fit-content'}}>
						<Provider spritemap={spritemap}>
							<LanguagePicker
								defaultLocaleId={defaultLanguageId}
								locales={locales}
								onSelectedLocaleChange={(
									localId: React.Key
								) => {
									setLanguageId(localId as string);
								}}
								selectedLocaleId={languageId}
								small
							/>
						</Provider>
					</div>
				</div>

				<div className={nameInputError ? 'has-error' : ''}>
					<label>
						{Liferay.Language.get('name')}

						<ClayIcon
							className="c-ml-1 reference-mark"
							focusable="false"
							role="presentation"
							symbol="asterisk"
						/>
					</label>

					<ClayInput
						data-testid="name-input"
						id="name"
						onChange={({target: {value}}) => onChangeName(value)}
						required
						type="text"
						value={
							category.name_i18n[getLanguageLabel(languageId)] ||
							''
						}
					/>

					{nameInputError && (
						<ClayAlert displayType="danger" variant="feedback">
							{nameInputError}
						</ClayAlert>
					)}
				</div>

				<div>
					<label>{Liferay.Language.get('description')}</label>

					<ClayInput
						component="textarea"
						data-testid="description-input"
						id="description"
						onChange={({target: {value}}) =>
							onChangeDescription(value)
						}
						type="text"
						value={
							category.description_i18n
								? category.description_i18n[
										getLanguageLabel(languageId)
									] || ''
								: ''
						}
					/>
				</div>
			</ClayForm.Group>
		</div>
	);
};

export default EditCategoryGeneralInfoTab;
