/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.object.service.impl;

import com.liferay.object.constants.ObjectActionKeys;
import com.liferay.object.constants.ObjectConstants;
import com.liferay.object.constants.ObjectDefinitionConstants;
import com.liferay.object.model.ObjectDefinition;
import com.liferay.object.model.ObjectDefinitionSetting;
import com.liferay.object.model.ObjectField;
import com.liferay.object.model.ObjectFolder;
import com.liferay.object.service.ObjectFolderLocalService;
import com.liferay.object.service.base.ObjectDefinitionServiceBaseImpl;
import com.liferay.petra.function.transform.TransformUtil;
import com.liferay.portal.aop.AopService;
import com.liferay.portal.kernel.dao.orm.QueryUtil;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.model.WorkflowDefinitionLink;
import com.liferay.portal.kernel.security.permission.ActionKeys;
import com.liferay.portal.kernel.security.permission.resource.ModelResourcePermission;
import com.liferay.portal.kernel.security.permission.resource.PortletResourcePermission;
import com.liferay.portal.kernel.workflow.WorkflowConstants;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Marco Leo
 * @author Brian Wing Shun Chan
 */
@Component(
	property = {
		"json.web.service.context.name=object",
		"json.web.service.context.path=ObjectDefinition"
	},
	service = AopService.class
)
public class ObjectDefinitionServiceImpl
	extends ObjectDefinitionServiceBaseImpl {

	@Override
	public ObjectDefinition addCustomObjectDefinition(
			String externalReferenceCode, long objectFolderId, String className,
			boolean enableComments, boolean enableFormContainer,
			boolean enableFriendlyURLCustomization, boolean enableIndexSearch,
			boolean enableLocalization, boolean enableObjectEntryDraft,
			boolean enableObjectEntrySchedule,
			boolean enableObjectEntrySubscription,
			boolean enableObjectEntryVersioning, String friendlyURLSeparator,
			Map<Locale, String> labelMap, String name, String panelAppOrder,
			String panelCategoryKey, Map<Locale, String> pluralLabelMap,
			boolean portlet, String scope, String storageType,
			List<ObjectDefinitionSetting> objectDefinitionSettings,
			List<ObjectField> objectFields,
			List<WorkflowDefinitionLink> workflowDefinitionLinks)
		throws PortalException {

		_portletResourcePermission.check(
			getPermissionChecker(), null,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		_objectFolderModelResourcePermission.check(
			getPermissionChecker(), objectFolderId,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		return objectDefinitionLocalService.addCustomObjectDefinition(
			externalReferenceCode, getUserId(), objectFolderId, className,
			enableComments, enableFormContainer, enableFriendlyURLCustomization,
			enableIndexSearch, enableLocalization, enableObjectEntryDraft,
			enableObjectEntrySchedule, enableObjectEntrySubscription,
			enableObjectEntryVersioning, friendlyURLSeparator, labelMap, name,
			panelAppOrder, panelCategoryKey, pluralLabelMap, portlet, scope,
			storageType, objectDefinitionSettings, objectFields,
			workflowDefinitionLinks);
	}

	@Override
	public ObjectDefinition addObjectDefinition(
			String externalReferenceCode, long objectFolderId,
			boolean modifiable, String scope, boolean system)
		throws PortalException {

		_portletResourcePermission.check(
			getPermissionChecker(), null,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		_objectFolderModelResourcePermission.check(
			getPermissionChecker(), objectFolderId,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		return objectDefinitionLocalService.addObjectDefinition(
			externalReferenceCode, getUserId(), objectFolderId, modifiable,
			scope, system);
	}

	@Override
	public ObjectDefinition addSystemObjectDefinition(
			String externalReferenceCode, long userId, long objectFolderId,
			String className, boolean enableComments,
			boolean enableFormContainer, boolean enableFriendlyURLCustomization,
			boolean enableIndexSearch, boolean enableLocalization,
			boolean enableObjectEntryDraft, boolean enableObjectEntrySchedule,
			boolean enableObjectEntrySubscription,
			boolean enableObjectEntryVersioning, String friendlyURLSeparator,
			Map<Locale, String> labelMap, String name, String panelAppOrder,
			String panelCategoryKey, Map<Locale, String> pluralLabelMap,
			boolean portlet, String scope,
			List<ObjectDefinitionSetting> objectDefinitionSettings,
			List<ObjectField> objectFields,
			List<WorkflowDefinitionLink> workflowDefinitionLinks)
		throws PortalException {

		_portletResourcePermission.check(
			getPermissionChecker(), null,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		_objectFolderModelResourcePermission.check(
			getPermissionChecker(), objectFolderId,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		return objectDefinitionLocalService.addSystemObjectDefinition(
			externalReferenceCode, userId, objectFolderId, className, null,
			enableComments, enableFormContainer, enableFriendlyURLCustomization,
			enableIndexSearch, enableLocalization, enableObjectEntryDraft,
			enableObjectEntrySchedule, enableObjectEntrySubscription,
			enableObjectEntryVersioning, friendlyURLSeparator, labelMap, true,
			name, panelAppOrder, panelCategoryKey, null, null, pluralLabelMap,
			portlet, scope, null, 1, WorkflowConstants.STATUS_DRAFT,
			objectDefinitionSettings, objectFields, workflowDefinitionLinks);
	}

	@Override
	public ObjectDefinition deleteObjectDefinition(long objectDefinitionId)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.DELETE);

		return objectDefinitionLocalService.deleteObjectDefinition(
			objectDefinitionId);
	}

	@Override
	public ObjectDefinition fetchObjectDefinitionByExternalReferenceCode(
			String externalReferenceCode, long companyId)
		throws PortalException {

		ObjectDefinition objectDefinition =
			objectDefinitionLocalService.
				fetchObjectDefinitionByExternalReferenceCode(
					externalReferenceCode, companyId);

		if (objectDefinition != null) {
			_objectDefinitionModelResourcePermission.check(
				getPermissionChecker(), objectDefinition, ActionKeys.VIEW);
		}

		return objectDefinition;
	}

	@Override
	public List<ObjectDefinition> getCMSObjectDefinitions(
		long companyId, String[] objectFolderExternalReferenceCodes) {

		long[] objectFolderIds = TransformUtil.transformToLongArray(
			Arrays.asList(objectFolderExternalReferenceCodes),
			objectFolderExternalReferenceCode -> {
				ObjectFolder objectFolder =
					_objectFolderLocalService.
						fetchObjectFolderByExternalReferenceCode(
							objectFolderExternalReferenceCode, companyId);

				if (objectFolder != null) {
					return objectFolder.getObjectFolderId();
				}

				return null;
			});

		return objectDefinitionPersistence.filterFindByC_OFI_A_E_S_S(
			companyId, objectFolderIds, true, true,
			ObjectDefinitionConstants.SCOPE_DEPOT,
			WorkflowConstants.STATUS_APPROVED, QueryUtil.ALL_POS,
			QueryUtil.ALL_POS);
	}

	@Override
	public ObjectDefinition getObjectDefinition(long objectDefinitionId)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.VIEW);

		return objectDefinitionLocalService.getObjectDefinition(
			objectDefinitionId);
	}

	@Override
	public ObjectDefinition getObjectDefinitionByExternalReferenceCode(
			String externalReferenceCode, long companyId)
		throws PortalException {

		ObjectDefinition objectDefinition =
			objectDefinitionLocalService.
				getObjectDefinitionByExternalReferenceCode(
					externalReferenceCode, companyId);

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinition, ActionKeys.VIEW);

		return objectDefinition;
	}

	@Override
	public List<ObjectDefinition> getObjectDefinitions(int start, int end) {
		return objectDefinitionLocalService.getObjectDefinitions(start, end);
	}

	@Override
	public List<ObjectDefinition> getObjectDefinitions(
		long companyId, int start, int end) {

		return objectDefinitionPersistence.findByCompanyId(
			companyId, start, end);
	}

	@Override
	public int getObjectDefinitionsCount() throws PortalException {
		return objectDefinitionLocalService.getObjectDefinitionsCount();
	}

	@Override
	public int getObjectDefinitionsCount(long companyId)
		throws PortalException {

		return objectDefinitionLocalService.getObjectDefinitionsCount(
			companyId);
	}

	@Override
	public ObjectDefinition publishCustomObjectDefinition(
			long objectDefinitionId)
		throws PortalException {

		_portletResourcePermission.check(
			getPermissionChecker(), null,
			ObjectActionKeys.PUBLISH_OBJECT_DEFINITION);

		return objectDefinitionLocalService.publishCustomObjectDefinition(
			getUserId(), objectDefinitionId);
	}

	@Override
	public ObjectDefinition publishSystemObjectDefinition(
			long objectDefinitionId)
		throws PortalException {

		_portletResourcePermission.check(
			getPermissionChecker(), null,
			ObjectActionKeys.PUBLISH_OBJECT_DEFINITION);

		return objectDefinitionLocalService.publishSystemObjectDefinition(
			getUserId(), objectDefinitionId);
	}

	@Override
	public ObjectDefinition updateCustomObjectDefinition(
			String externalReferenceCode, long objectDefinitionId,
			long accountEntryRestrictedObjectFieldId,
			long descriptionObjectFieldId, long objectFolderId,
			long titleObjectFieldId, boolean accountEntryRestricted,
			boolean active, String className, boolean enableCategorization,
			boolean enableComments, boolean enableFormContainer,
			boolean enableFriendlyURLCustomization, boolean enableIndexSearch,
			boolean enableLocalization, boolean enableObjectEntryDraft,
			boolean enableObjectEntryHistory, boolean enableObjectEntrySchedule,
			boolean enableObjectEntrySubscription,
			boolean enableObjectEntryVersioning, String friendlyURLSeparator,
			Map<Locale, String> labelMap, String name, String panelAppOrder,
			String panelCategoryKey, boolean portlet,
			Map<Locale, String> pluralLabelMap, String scope, int status,
			List<ObjectDefinitionSetting> objectDefinitionSettings,
			List<ObjectField> objectFields,
			List<WorkflowDefinitionLink> workflowDefinitionLinks)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.UPDATE);

		_objectFolderModelResourcePermission.check(
			getPermissionChecker(), objectFolderId,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		return objectDefinitionLocalService.updateCustomObjectDefinition(
			externalReferenceCode, objectDefinitionId,
			accountEntryRestrictedObjectFieldId, descriptionObjectFieldId,
			objectFolderId, titleObjectFieldId, accountEntryRestricted, active,
			className, enableCategorization, enableComments,
			enableFormContainer, enableFriendlyURLCustomization,
			enableIndexSearch, enableLocalization, enableObjectEntryDraft,
			enableObjectEntryHistory, enableObjectEntrySchedule,
			enableObjectEntrySubscription, enableObjectEntryVersioning,
			friendlyURLSeparator, labelMap, name, panelAppOrder,
			panelCategoryKey, portlet, pluralLabelMap, scope, status,
			objectDefinitionSettings, objectFields, workflowDefinitionLinks);
	}

	@Override
	public ObjectDefinition updateExternalReferenceCode(
			long objectDefinitionId, String externalReferenceCode)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.UPDATE);

		return objectDefinitionLocalService.updateExternalReferenceCode(
			objectDefinitionId, externalReferenceCode);
	}

	@Override
	public ObjectDefinition updateSystemObjectDefinition(
			String externalReferenceCode, long objectDefinitionId,
			long objectFolderId, long titleObjectFieldId,
			List<ObjectDefinitionSetting> objectDefinitionSettings,
			List<ObjectField> objectFields,
			List<WorkflowDefinitionLink> workflowDefinitionLinks)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.UPDATE);

		_objectFolderModelResourcePermission.check(
			getPermissionChecker(), objectFolderId,
			ObjectActionKeys.ADD_OBJECT_DEFINITION);

		return objectDefinitionLocalService.updateSystemObjectDefinition(
			externalReferenceCode, objectDefinitionId, objectFolderId,
			titleObjectFieldId, objectDefinitionSettings, objectFields,
			workflowDefinitionLinks);
	}

	@Override
	public ObjectDefinition updateTitleObjectFieldId(
			long objectDefinitionId, long titleObjectFieldId)
		throws PortalException {

		_objectDefinitionModelResourcePermission.check(
			getPermissionChecker(), objectDefinitionId, ActionKeys.UPDATE);

		return objectDefinitionLocalService.updateTitleObjectFieldId(
			objectDefinitionId, titleObjectFieldId);
	}

	@Reference(
		target = "(model.class.name=com.liferay.object.model.ObjectDefinition)"
	)
	private ModelResourcePermission<ObjectDefinition>
		_objectDefinitionModelResourcePermission;

	@Reference
	private ObjectFolderLocalService _objectFolderLocalService;

	@Reference(
		target = "(model.class.name=com.liferay.object.model.ObjectFolder)"
	)
	private ModelResourcePermission<ObjectFolder>
		_objectFolderModelResourcePermission;

	@Reference(target = "(resource.name=" + ObjectConstants.RESOURCE_NAME + ")")
	private PortletResourcePermission _portletResourcePermission;

}