/*
 * Decompiled with CFR 0.152.
 */
package uk.sky.cqlmigrate;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import java.io.IOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CqlFileParser {
    private static final Logger LOGGER = LoggerFactory.getLogger(CqlFileParser.class);
    private static final Pattern EOL = Pattern.compile(".*\\R|.+\\z");

    private CqlFileParser() {
    }

    static List<String> getCqlStatementsFrom(Path cqlPath) {
        LineProcessor processor = new LineProcessor();
        try (Scanner in = new Scanner(cqlPath, "UTF-8");){
            String original;
            while ((original = in.findWithinHorizon(EOL, 0)) != null) {
                processor.process(original);
            }
        }
        catch (IOException e) {
            LOGGER.error("Failed to process cql script {}: {}", (Object)cqlPath.getFileName(), (Object)e.getMessage());
            throw Throwables.propagate((Throwable)e);
        }
        processor.check();
        return processor.getResult();
    }

    private static class LineProcessor {
        private static final char CQL_STATEMENT_STRING_DELIMITER = '\'';
        private static final String CQL_STATEMENT_TERMINATOR = ";";
        private static final String CQL_COMMENT_DOUBLE_HYPEN = "--";
        private static final String CQL_MULTI_LINE_COMMENT_OPEN = "/*";
        private static final String CQL_MULTI_LINE_COMMENT_CLOSE = "*/";
        private static final Pattern CQL_MULTI_LINE_COMMENT_PATTERN = Pattern.compile("/\\*.*?\\*/", 32);
        private static final String EMPTY_STR = "";
        List<String> statements;
        State curState = State.INIT;
        StringBuilder curStmt;

        private LineProcessor() {
        }

        void process(String original) throws IOException {
            switch (this.curState) {
                case INIT: {
                    this.init(original);
                    break;
                }
                case FIND_EOS: 
                case IS_OPEN_STMT: {
                    this.findStatement(original);
                    break;
                }
                case IS_OPEN_VALUE_EXP: {
                    this.findValueExpression(original);
                    break;
                }
                case IS_MULTI_LINE_COMMENT: {
                    this.findMultilineComment(original);
                    break;
                }
                case IS_CLOSE_STMT: {
                    this.closedStatement(original);
                }
            }
        }

        private void init(String original) throws IOException {
            if (this.statements == null) {
                this.statements = new ArrayList<String>();
            }
            this.curState = State.FIND_EOS;
            this.curStmt = new StringBuilder();
            this.process(original);
        }

        private void findStatement(String original) throws IOException {
            String line = CharMatcher.WHITESPACE.trimFrom((CharSequence)original);
            if (line.startsWith(CQL_COMMENT_DOUBLE_HYPEN) || line.isEmpty()) {
                return;
            }
            if (line.startsWith(CQL_MULTI_LINE_COMMENT_OPEN)) {
                this.curState = State.IS_MULTI_LINE_COMMENT;
                return;
            }
            if (line.endsWith(CQL_STATEMENT_TERMINATOR)) {
                this.curStmt.append(" ").append(line.substring(0, line.length() - 1));
                this.statements.add(CharMatcher.WHITESPACE.trimFrom((CharSequence)this.curStmt.toString()));
                this.curState = State.IS_CLOSE_STMT;
                this.process(original);
                return;
            }
            if (CharMatcher.is((char)'\'').countIn((CharSequence)line) % 2 != 0) {
                this.curState = State.IS_OPEN_VALUE_EXP;
                this.curStmt.append(" ").append(CharMatcher.WHITESPACE.trimLeadingFrom((CharSequence)original));
                return;
            }
            int pos = line.indexOf(CQL_COMMENT_DOUBLE_HYPEN);
            if (pos != -1) {
                this.curStmt.append(line.substring(0, pos));
                return;
            }
            Matcher matcher = CQL_MULTI_LINE_COMMENT_PATTERN.matcher(line);
            if (matcher.find()) {
                this.curStmt.append(matcher.replaceAll(EMPTY_STR));
                return;
            }
            if (State.IS_OPEN_STMT.equals((Object)this.curState)) {
                this.curStmt.append(" ").append(line);
            } else {
                this.curState = State.IS_OPEN_STMT;
                this.curStmt.append(line);
            }
        }

        private void findValueExpression(String original) {
            if (CharMatcher.is((char)'\'').countIn((CharSequence)original) % 2 != 0) {
                this.curStmt.append(original);
                this.curState = State.FIND_EOS;
                return;
            }
            this.curStmt.append(original);
        }

        private void findMultilineComment(String original) {
            if (CharMatcher.WHITESPACE.trimTrailingFrom((CharSequence)original).endsWith(CQL_MULTI_LINE_COMMENT_CLOSE)) {
                this.curState = State.FIND_EOS;
            }
        }

        private void closedStatement(String original) {
            LOGGER.trace("CQL parsed: {}", (Object)original);
            this.curState = State.INIT;
        }

        private void check() {
            Preconditions.checkState((State.IS_CLOSE_STMT.equals((Object)this.curState) || State.INIT.equals((Object)this.curState) ? 1 : 0) != 0, (Object)"File had a non-terminated cql line");
        }

        List<String> getResult() {
            return this.statements;
        }

        private static enum State {
            INIT,
            FIND_EOS,
            IS_MULTI_LINE_COMMENT,
            IS_OPEN_STMT,
            IS_OPEN_VALUE_EXP,
            IS_CLOSE_STMT;

        }
    }
}

