package top.mdoufei.logkit.logcat;

import android.support.annotation.NonNull;

import top.mdoufei.logkit.Formatter;
import top.mdoufei.logkit.LogKit;
import top.mdoufei.logkit.LogManager;
import top.mdoufei.logkit.Util;


/**
 * Created by leon on 2018/5/23.
 */

public class LogCatFormatter implements Formatter {

    private static final char TOP_LEFT_CORNER = '┌';
    private static final char BOTTOM_LEFT_CORNER = '└';
    private static final char MIDDLE_CORNER = '├';
    private static final char HORIZONTAL_LINE = '│';
    private static final String DOUBLE_DIVIDER = "────────────────────────────────────────────────────────";
    private static final String SINGLE_DIVIDER = "┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄┄";
    private static final String TOP_BORDER = TOP_LEFT_CORNER + DOUBLE_DIVIDER + DOUBLE_DIVIDER;
    private static final String BOTTOM_BORDER = BOTTOM_LEFT_CORNER + DOUBLE_DIVIDER + DOUBLE_DIVIDER;
    private static final String MIDDLE_BORDER = MIDDLE_CORNER + SINGLE_DIVIDER + SINGLE_DIVIDER;

    private static final int CHUNK_SIZE = 4000;
    private static final int MIN_STACK_OFFSET = 5;

    LogCatAppender logCatAppender = new LogCatAppender();

    @Override
    public void log(int priority, String tag, @NonNull String message) {
        logChunk(priority,tag,TOP_BORDER);
        logChunk(priority, tag, HORIZONTAL_LINE + " Thread: " + Thread.currentThread().getName());
        logChunk(priority,tag, MIDDLE_BORDER);
        logHeaderContent(priority,tag,1);
        logChunk(priority,tag, MIDDLE_BORDER);

        byte[] bytes = message.getBytes();
        int length = bytes.length;
        if (length <= CHUNK_SIZE) {
            logContent(priority, tag, message);
            logChunk(priority, tag, BOTTOM_BORDER);
            return;
        }else{
            logContent(priority, tag, "Content oversize... Over " + CHUNK_SIZE + " bytes.");
            logChunk(priority, tag, BOTTOM_BORDER);
        }

    }

    private void logChunk(int priority, String tag, @NonNull String message){
        logCatAppender.log(priority,tag,message);
    }

    private void logContent(int priority, String tag, @NonNull String message){
        String[] lines = message.split(System.getProperty("line.separator"));
        for (String line : lines) {
            logChunk(priority, tag, HORIZONTAL_LINE + " " + line);
        }
    }

    private int getStackOffset(@NonNull StackTraceElement[] trace) {

        for (int i = MIN_STACK_OFFSET; i < trace.length; i++) {
            StackTraceElement e = trace[i];
            String name = e.getClassName();
            if (!name.equals(LogManager.class.getName()) && !name.equals(LogKit.class.getName())) {
                return --i;
            }
        }
        return -1;
    }


    @SuppressWarnings("StringBufferReplaceableByString")
    private void logHeaderContent(int priority, String tag, int methodCount) {
        StackTraceElement[] trace = Thread.currentThread().getStackTrace();

        String level = "";

        int stackOffset = getStackOffset(trace) + 0;

        //corresponding method count with the current stack may exceeds the stack trace. Trims the count
        if (methodCount + stackOffset > trace.length) {
            methodCount = trace.length - stackOffset - 1;
        }

        for (int i = methodCount; i > 0; i--) {
            int stackIndex = i + stackOffset;
            if (stackIndex >= trace.length) {
                continue;
            }
            StringBuilder builder = new StringBuilder();
            builder.append(HORIZONTAL_LINE)
                    .append(' ')
                    .append(level)
                    .append(Util.getSimpleClassName(trace[stackIndex].getClassName()))
                    .append(".")
                    .append(trace[stackIndex].getMethodName())
                    .append(" ")
                    .append(" (")
                    .append(trace[stackIndex].getFileName())
                    .append(":")
                    .append(trace[stackIndex].getLineNumber())
                    .append(")");
            level += "   ";
            logChunk(priority, tag, builder.toString());
        }
    }
}
