/*
 * Copyright (c) 2016 NECTEC
 *   National Electronics and Computer Technology Center, Thailand
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package nectec.thai.widget.address;

import android.app.Dialog;
import android.content.Context;
import android.os.Bundle;
import android.view.View;
import android.view.Window;
import android.widget.AdapterView;
import android.widget.ListView;
import android.widget.SearchView;
import android.widget.TextView;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import nectec.thai.address.Address;
import nectec.thai.address.AddressEntity;
import nectec.thai.address.AddressRepository;
import nectec.thai.address.District;
import nectec.thai.address.Province;
import nectec.thai.address.SubDistrict;
import nectec.thai.widget.address.AddressView.OnAddressChangedListener;
import nectec.thai.widget.address.repository.AddressRepositoryImpl;
import nectec.thai.widget.address.repository.DistrictRepository;
import nectec.thai.widget.address.repository.ProvinceRepository;
import nectec.thai.widget.address.repository.SubDistrictRepository;
import th.or.nectec.thai.widget.R;

public class AddressPickerDialog extends Dialog implements AddressPopup, AdapterView.OnItemClickListener {

    private ArrayStack<AddressEntity> addressStack = new ArrayStack<>();
    private TextView header;
    private TextView breadcrumb;
    private ListView list;
    private SearchView search;
    private AddressRepositoryImpl addressRepository;
    private AddressRepository<Province> provinceRepository;
    private AddressRepository<District> districtRepository;
    private AddressRepository<SubDistrict> subDistrictRepository;
    private OnAddressChangedListener onAddressChangedListener;
    private AddressListAdapter addressListAdapter;


    public AddressPickerDialog(Context context) {
        this(context, null);
    }

    public AddressPickerDialog(Context context, OnAddressChangedListener onAddressChangedListener) {
        this(context, 0, onAddressChangedListener);
    }

    public AddressPickerDialog(Context context, int themeResId, OnAddressChangedListener onAddressChangedListener) {
        super(context, themeResId);
        requestWindowFeature(Window.FEATURE_NO_TITLE);

        addressRepository = AddressRepositoryImpl.getInstance(context);
        provinceRepository = ProvinceRepository.getInstance(context);
        districtRepository = DistrictRepository.getInstance(context);
        subDistrictRepository = SubDistrictRepository.getInstance(context);

        this.onAddressChangedListener = onAddressChangedListener;
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.dialog_address_picker);
        breadcrumb = (TextView) findViewById(R.id.breadcrumb);
        header = (TextView) findViewById(R.id.header);
        list = (ListView) findViewById(R.id.picker_list);
        search = (SearchView) findViewById(R.id.search);
        search.setOnCloseListener(new SearchView.OnCloseListener() {
            @Override public boolean onClose() {
                return false;
            }
        });
        search.setOnQueryTextListener(new SearchView.OnQueryTextListener() {
            @Override public boolean onQueryTextSubmit(String query) {
                return true;
            }

            @Override public boolean onQueryTextChange(String query) {
                addressListAdapter.getFilter().filter(query);
                return true;
            }
        });

        list.setOnItemClickListener(this);
        findViewById(R.id.back).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                onBackPressed();
            }
        });
        switchPage();
    }

    @Override
    public void onBackPressed() {
        if (addressStack.isEmpty()) {
            onAddressChangedListener.onAddressCanceled();
            dismiss();
            return;
        }
        addressStack.pop();
        switchPage();
    }

    public AddressPickerDialog setProvinceRepository(AddressRepository<Province> province) {
        this.provinceRepository = province;
        return this;
    }

    public AddressPickerDialog setDistrictRepository(AddressRepository<District> districtRepository) {
        this.districtRepository = districtRepository;
        return this;
    }

    public AddressPickerDialog setSubDistrictRepository(AddressRepository<SubDistrict> subDistrictRepository) {
        this.subDistrictRepository = subDistrictRepository;
        return this;
    }

    @Override
    public void onItemClick(AdapterView<?> adapterView, View view, int position, long id) {
        addressStack.push(addressListAdapter.getItem(position));
        switchPage();
    }

    private void switchPage() {
        updateBreadCrumb();
        if (addressStack.isEmpty()) {
            List<Province> provinceList = provinceRepository.find();
            Collections.sort(provinceList);
            updateDialog(R.string.choose_province, provinceList);
            return;
        }
        AddressEntity choosedEntity = addressStack.peek();
        if (choosedEntity instanceof Province) {
            List<District> districtList = districtRepository.findByParentCode(choosedEntity.getCode());
            Collections.sort(districtList);
            updateDialog(R.string.choose_district, districtList);
        } else if (choosedEntity instanceof District) {
            List<SubDistrict> subDistrictList = subDistrictRepository.findByParentCode(choosedEntity.getCode());
            Collections.sort(subDistrictList);
            updateDialog(R.string.choose_subdistrict, subDistrictList);
        } else if (choosedEntity instanceof SubDistrict) {
            notifyAddressChange(choosedEntity.getCode());
            dismiss();
        }
    }

    private void updateBreadCrumb() {
        if (addressStack.isEmpty()) {
            breadcrumb.setText("");
            breadcrumb.setVisibility(View.GONE);
            return;
        }
        StringBuilder builder = new StringBuilder();
        Iterator<AddressEntity> iterator = addressStack.getIterator();
        while (iterator.hasNext()) {
            builder.append(iterator.next().getName());
            if (iterator.hasNext()) {
                builder.append("  >  ");
            }
        }
        breadcrumb.setText(builder.toString());
        breadcrumb.setVisibility(View.VISIBLE);
    }

    public void updateDialog(int headerStringResId, List<? extends AddressEntity> addressEntityList) {
        header.setText(headerStringResId);
        setListAdapter(new AddressListAdapter<>(getContext(), addressEntityList));

        search.setQueryHint(getContext().getString(headerStringResId).replace("เลือก", "ค้นหา"));
        search.clearFocus();
        search.setIconified(true); //Clear search test if present
        search.setIconified(true);
    }

    private void setListAdapter(AddressListAdapter addressListAdapter) {
        this.addressListAdapter = addressListAdapter;
        list.setAdapter(addressListAdapter);
    }

    private void notifyAddressChange(String code) {
        Address address = addressRepository.findByCode(code);
        onAddressChangedListener.onAddressChanged(address);
    }

    @Override public void show(String addressCode) {
        show(addressRepository.findByCode(addressCode));
    }

    @Override
    public void show(Address area) {
        addressStack = new ArrayStack<>();
        if (area != null) {
            addressStack.push(area.getProvince());
            addressStack.push(area.getDistrict());
        }
        show();
    }

    @Override
    public void setOnAddressChangedListener(OnAddressChangedListener onAddressChangedListener) {
        this.onAddressChangedListener = onAddressChangedListener;
    }


}
