/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Operations engineers and IT professionals use Amazon Web Services Systems Manager OpsCenter to view, investigate, and
 * remediate operational work items (OpsItems) impacting the performance and health of their Amazon Web Services
 * resources. OpsCenter is integrated with Amazon EventBridge and Amazon CloudWatch. This means you can configure these
 * services to automatically create an OpsItem in OpsCenter when a CloudWatch alarm enters the ALARM state or when
 * EventBridge processes an event from any Amazon Web Services service that publishes events. Configuring Amazon
 * CloudWatch alarms and EventBridge events to automatically create OpsItems allows you to quickly diagnose and
 * remediate issues with Amazon Web Services resources from a single console.
 * </p>
 * <p>
 * To help you diagnose issues, each OpsItem includes contextually relevant information such as the name and ID of the
 * Amazon Web Services resource that generated the OpsItem, alarm or event details, alarm history, and an alarm timeline
 * graph. For the Amazon Web Services resource, OpsCenter aggregates information from Config, CloudTrail logs, and
 * EventBridge, so you don't have to navigate across multiple console pages during your investigation. For more
 * information, see <a href="https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter.html">Amazon Web
 * Services Systems Manager OpsCenter</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpsItem implements SdkPojo, Serializable, ToCopyableBuilder<OpsItem.Builder, OpsItem> {
    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatedBy").getter(getter(OpsItem::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<String> OPS_ITEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OpsItemType").getter(getter(OpsItem::opsItemType)).setter(setter(Builder::opsItemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpsItemType").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(OpsItem::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(OpsItem::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> LAST_MODIFIED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModifiedBy").getter(getter(OpsItem::lastModifiedBy)).setter(setter(Builder::lastModifiedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedBy").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(OpsItem::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<List<OpsItemNotification>> NOTIFICATIONS_FIELD = SdkField
            .<List<OpsItemNotification>> builder(MarshallingType.LIST)
            .memberName("Notifications")
            .getter(getter(OpsItem::notifications))
            .setter(setter(Builder::notifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Notifications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpsItemNotification> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemNotification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> PRIORITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Priority").getter(getter(OpsItem::priority)).setter(setter(Builder::priority))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Priority").build()).build();

    private static final SdkField<List<RelatedOpsItem>> RELATED_OPS_ITEMS_FIELD = SdkField
            .<List<RelatedOpsItem>> builder(MarshallingType.LIST)
            .memberName("RelatedOpsItems")
            .getter(getter(OpsItem::relatedOpsItems))
            .setter(setter(Builder::relatedOpsItems))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RelatedOpsItems").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RelatedOpsItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(RelatedOpsItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(OpsItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> OPS_ITEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OpsItemId").getter(getter(OpsItem::opsItemId)).setter(setter(Builder::opsItemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpsItemId").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(OpsItem::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(OpsItem::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Source")
            .getter(getter(OpsItem::source)).setter(setter(Builder::source))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<Map<String, OpsItemDataValue>> OPERATIONAL_DATA_FIELD = SdkField
            .<Map<String, OpsItemDataValue>> builder(MarshallingType.MAP)
            .memberName("OperationalData")
            .getter(getter(OpsItem::operationalData))
            .setter(setter(Builder::operationalData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperationalData").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<OpsItemDataValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpsItemDataValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Category").getter(getter(OpsItem::category)).setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category").build()).build();

    private static final SdkField<String> SEVERITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Severity").getter(getter(OpsItem::severity)).setter(setter(Builder::severity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Severity").build()).build();

    private static final SdkField<Instant> ACTUAL_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ActualStartTime").getter(getter(OpsItem::actualStartTime)).setter(setter(Builder::actualStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActualStartTime").build()).build();

    private static final SdkField<Instant> ACTUAL_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ActualEndTime").getter(getter(OpsItem::actualEndTime)).setter(setter(Builder::actualEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActualEndTime").build()).build();

    private static final SdkField<Instant> PLANNED_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("PlannedStartTime").getter(getter(OpsItem::plannedStartTime)).setter(setter(Builder::plannedStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlannedStartTime").build()).build();

    private static final SdkField<Instant> PLANNED_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("PlannedEndTime").getter(getter(OpsItem::plannedEndTime)).setter(setter(Builder::plannedEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlannedEndTime").build()).build();

    private static final SdkField<String> OPS_ITEM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OpsItemArn").getter(getter(OpsItem::opsItemArn)).setter(setter(Builder::opsItemArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OpsItemArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CREATED_BY_FIELD,
            OPS_ITEM_TYPE_FIELD, CREATED_TIME_FIELD, DESCRIPTION_FIELD, LAST_MODIFIED_BY_FIELD, LAST_MODIFIED_TIME_FIELD,
            NOTIFICATIONS_FIELD, PRIORITY_FIELD, RELATED_OPS_ITEMS_FIELD, STATUS_FIELD, OPS_ITEM_ID_FIELD, VERSION_FIELD,
            TITLE_FIELD, SOURCE_FIELD, OPERATIONAL_DATA_FIELD, CATEGORY_FIELD, SEVERITY_FIELD, ACTUAL_START_TIME_FIELD,
            ACTUAL_END_TIME_FIELD, PLANNED_START_TIME_FIELD, PLANNED_END_TIME_FIELD, OPS_ITEM_ARN_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String createdBy;

    private final String opsItemType;

    private final Instant createdTime;

    private final String description;

    private final String lastModifiedBy;

    private final Instant lastModifiedTime;

    private final List<OpsItemNotification> notifications;

    private final Integer priority;

    private final List<RelatedOpsItem> relatedOpsItems;

    private final String status;

    private final String opsItemId;

    private final String version;

    private final String title;

    private final String source;

    private final Map<String, OpsItemDataValue> operationalData;

    private final String category;

    private final String severity;

    private final Instant actualStartTime;

    private final Instant actualEndTime;

    private final Instant plannedStartTime;

    private final Instant plannedEndTime;

    private final String opsItemArn;

    private OpsItem(BuilderImpl builder) {
        this.createdBy = builder.createdBy;
        this.opsItemType = builder.opsItemType;
        this.createdTime = builder.createdTime;
        this.description = builder.description;
        this.lastModifiedBy = builder.lastModifiedBy;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.notifications = builder.notifications;
        this.priority = builder.priority;
        this.relatedOpsItems = builder.relatedOpsItems;
        this.status = builder.status;
        this.opsItemId = builder.opsItemId;
        this.version = builder.version;
        this.title = builder.title;
        this.source = builder.source;
        this.operationalData = builder.operationalData;
        this.category = builder.category;
        this.severity = builder.severity;
        this.actualStartTime = builder.actualStartTime;
        this.actualEndTime = builder.actualEndTime;
        this.plannedStartTime = builder.plannedStartTime;
        this.plannedEndTime = builder.plannedEndTime;
        this.opsItemArn = builder.opsItemArn;
    }

    /**
     * <p>
     * The ARN of the Amazon Web Services account that created the OpsItem.
     * </p>
     * 
     * @return The ARN of the Amazon Web Services account that created the OpsItem.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The type of OpsItem. Systems Manager supports the following types of OpsItems:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>/aws/issue</code>
     * </p>
     * <p>
     * This type of OpsItem is used for default OpsItems created by OpsCenter.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>/aws/changerequest</code>
     * </p>
     * <p>
     * This type of OpsItem is used by Change Manager for reviewing and approving or rejecting change requests.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>/aws/insight</code>
     * </p>
     * <p>
     * This type of OpsItem is used by OpsCenter for aggregating and reporting on duplicate OpsItems.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The type of OpsItem. Systems Manager supports the following types of OpsItems:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>/aws/issue</code>
     *         </p>
     *         <p>
     *         This type of OpsItem is used for default OpsItems created by OpsCenter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>/aws/changerequest</code>
     *         </p>
     *         <p>
     *         This type of OpsItem is used by Change Manager for reviewing and approving or rejecting change requests.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>/aws/insight</code>
     *         </p>
     *         <p>
     *         This type of OpsItem is used by OpsCenter for aggregating and reporting on duplicate OpsItems.
     *         </p>
     *         </li>
     */
    public final String opsItemType() {
        return opsItemType;
    }

    /**
     * <p>
     * The date and time the OpsItem was created.
     * </p>
     * 
     * @return The date and time the OpsItem was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The OpsItem description.
     * </p>
     * 
     * @return The OpsItem description.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The ARN of the Amazon Web Services account that last updated the OpsItem.
     * </p>
     * 
     * @return The ARN of the Amazon Web Services account that last updated the OpsItem.
     */
    public final String lastModifiedBy() {
        return lastModifiedBy;
    }

    /**
     * <p>
     * The date and time the OpsItem was last updated.
     * </p>
     * 
     * @return The date and time the OpsItem was last updated.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Notifications property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotifications() {
        return notifications != null && !(notifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where notifications
     * are sent when this OpsItem is edited or changed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotifications} method.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
     *         notifications are sent when this OpsItem is edited or changed.
     */
    public final List<OpsItemNotification> notifications() {
        return notifications;
    }

    /**
     * <p>
     * The importance of this OpsItem in relation to other OpsItems in the system.
     * </p>
     * 
     * @return The importance of this OpsItem in relation to other OpsItems in the system.
     */
    public final Integer priority() {
        return priority;
    }

    /**
     * For responses, this returns true if the service returned a value for the RelatedOpsItems property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRelatedOpsItems() {
        return relatedOpsItems != null && !(relatedOpsItems instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems can
     * include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRelatedOpsItems} method.
     * </p>
     * 
     * @return One or more OpsItems that share something in common with the current OpsItem. For example, related
     *         OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
     *         impacted resource.
     */
    public final List<RelatedOpsItem> relatedOpsItems() {
        return relatedOpsItems;
    }

    /**
     * <p>
     * The OpsItem status. For more information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     * >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     *         >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public final OpsItemStatus status() {
        return OpsItemStatus.fromValue(status);
    }

    /**
     * <p>
     * The OpsItem status. For more information, see <a href=
     * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     * >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link OpsItemStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The OpsItem status. For more information, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
     *         >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * @see OpsItemStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the OpsItem.
     * </p>
     * 
     * @return The ID of the OpsItem.
     */
    public final String opsItemId() {
        return opsItemId;
    }

    /**
     * <p>
     * The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
     * </p>
     * 
     * @return The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * A short heading that describes the nature of the OpsItem and the impacted resource.
     * </p>
     * 
     * @return A short heading that describes the nature of the OpsItem and the impacted resource.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The origin of the OpsItem, such as Amazon EC2 or Systems Manager. The impacted resource is a subset of source.
     * </p>
     * 
     * @return The origin of the OpsItem, such as Amazon EC2 or Systems Manager. The impacted resource is a subset of
     *         source.
     */
    public final String source() {
        return source;
    }

    /**
     * For responses, this returns true if the service returned a value for the OperationalData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOperationalData() {
        return operationalData != null && !(operationalData instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Operational data is custom data that provides useful reference details about the OpsItem. For example, you can
     * specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
     * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum size
     * of 20 KB.
     * </p>
     * <important>
     * <p>
     * Operational data keys <i>can't</i> begin with the following: <code>amazon</code>, <code>aws</code>,
     * <code>amzn</code>, <code>ssm</code>, <code>/amazon</code>, <code>/aws</code>, <code>/amzn</code>,
     * <code>/ssm</code>.
     * </p>
     * </important>
     * <p>
     * You can choose to make the data searchable by other users in the account or you can restrict search access.
     * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     * <a>DescribeOpsItems</a> API operation) can view and search on the specified data. Operational data that isn't
     * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
     * operation).
     * </p>
     * <p>
     * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use the
     * <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem. To view
     * Amazon Web Services CLI example commands that use these keys, see <a
     * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-manually-create-OpsItems.html"
     * >Creating OpsItems manually</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperationalData} method.
     * </p>
     * 
     * @return Operational data is custom data that provides useful reference details about the OpsItem. For example,
     *         you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You
     *         enter operational data as key-value pairs. The key has a maximum length of 128 characters. The value has
     *         a maximum size of 20 KB.</p> <important>
     *         <p>
     *         Operational data keys <i>can't</i> begin with the following: <code>amazon</code>, <code>aws</code>,
     *         <code>amzn</code>, <code>ssm</code>, <code>/amazon</code>, <code>/aws</code>, <code>/amzn</code>,
     *         <code>/ssm</code>.
     *         </p>
     *         </important>
     *         <p>
     *         You can choose to make the data searchable by other users in the account or you can restrict search
     *         access. Searchable data means that all users with access to the OpsItem Overview page (as provided by the
     *         <a>DescribeOpsItems</a> API operation) can view and search on the specified data. Operational data that
     *         isn't searchable is only viewable by users who have access to the OpsItem (as provided by the
     *         <a>GetOpsItem</a> API operation).
     *         </p>
     *         <p>
     *         Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request.
     *         Use the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the
     *         OpsItem. To view Amazon Web Services CLI example commands that use these keys, see <a href=
     *         "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-manually-create-OpsItems.html"
     *         >Creating OpsItems manually</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
     */
    public final Map<String, OpsItemDataValue> operationalData() {
        return operationalData;
    }

    /**
     * <p>
     * An OpsItem category. Category options include: Availability, Cost, Performance, Recovery, Security.
     * </p>
     * 
     * @return An OpsItem category. Category options include: Availability, Cost, Performance, Recovery, Security.
     */
    public final String category() {
        return category;
    }

    /**
     * <p>
     * The severity of the OpsItem. Severity options range from 1 to 4.
     * </p>
     * 
     * @return The severity of the OpsItem. Severity options range from 1 to 4.
     */
    public final String severity() {
        return severity;
    }

    /**
     * <p>
     * The time a runbook workflow started. Currently reported only for the OpsItem type <code>/aws/changerequest</code>
     * .
     * </p>
     * 
     * @return The time a runbook workflow started. Currently reported only for the OpsItem type
     *         <code>/aws/changerequest</code>.
     */
    public final Instant actualStartTime() {
        return actualStartTime;
    }

    /**
     * <p>
     * The time a runbook workflow ended. Currently reported only for the OpsItem type <code>/aws/changerequest</code>.
     * </p>
     * 
     * @return The time a runbook workflow ended. Currently reported only for the OpsItem type
     *         <code>/aws/changerequest</code>.
     */
    public final Instant actualEndTime() {
        return actualEndTime;
    }

    /**
     * <p>
     * The time specified in a change request for a runbook workflow to start. Currently supported only for the OpsItem
     * type <code>/aws/changerequest</code>.
     * </p>
     * 
     * @return The time specified in a change request for a runbook workflow to start. Currently supported only for the
     *         OpsItem type <code>/aws/changerequest</code>.
     */
    public final Instant plannedStartTime() {
        return plannedStartTime;
    }

    /**
     * <p>
     * The time specified in a change request for a runbook workflow to end. Currently supported only for the OpsItem
     * type <code>/aws/changerequest</code>.
     * </p>
     * 
     * @return The time specified in a change request for a runbook workflow to end. Currently supported only for the
     *         OpsItem type <code>/aws/changerequest</code>.
     */
    public final Instant plannedEndTime() {
        return plannedEndTime;
    }

    /**
     * <p>
     * The OpsItem Amazon Resource Name (ARN).
     * </p>
     * 
     * @return The OpsItem Amazon Resource Name (ARN).
     */
    public final String opsItemArn() {
        return opsItemArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(opsItemType());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedBy());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasNotifications() ? notifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(priority());
        hashCode = 31 * hashCode + Objects.hashCode(hasRelatedOpsItems() ? relatedOpsItems() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(opsItemId());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(hasOperationalData() ? operationalData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(category());
        hashCode = 31 * hashCode + Objects.hashCode(severity());
        hashCode = 31 * hashCode + Objects.hashCode(actualStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(actualEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(plannedStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(plannedEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(opsItemArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpsItem)) {
            return false;
        }
        OpsItem other = (OpsItem) obj;
        return Objects.equals(createdBy(), other.createdBy()) && Objects.equals(opsItemType(), other.opsItemType())
                && Objects.equals(createdTime(), other.createdTime()) && Objects.equals(description(), other.description())
                && Objects.equals(lastModifiedBy(), other.lastModifiedBy())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime()) && hasNotifications() == other.hasNotifications()
                && Objects.equals(notifications(), other.notifications()) && Objects.equals(priority(), other.priority())
                && hasRelatedOpsItems() == other.hasRelatedOpsItems()
                && Objects.equals(relatedOpsItems(), other.relatedOpsItems())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(opsItemId(), other.opsItemId())
                && Objects.equals(version(), other.version()) && Objects.equals(title(), other.title())
                && Objects.equals(source(), other.source()) && hasOperationalData() == other.hasOperationalData()
                && Objects.equals(operationalData(), other.operationalData()) && Objects.equals(category(), other.category())
                && Objects.equals(severity(), other.severity()) && Objects.equals(actualStartTime(), other.actualStartTime())
                && Objects.equals(actualEndTime(), other.actualEndTime())
                && Objects.equals(plannedStartTime(), other.plannedStartTime())
                && Objects.equals(plannedEndTime(), other.plannedEndTime()) && Objects.equals(opsItemArn(), other.opsItemArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpsItem").add("CreatedBy", createdBy()).add("OpsItemType", opsItemType())
                .add("CreatedTime", createdTime()).add("Description", description()).add("LastModifiedBy", lastModifiedBy())
                .add("LastModifiedTime", lastModifiedTime()).add("Notifications", hasNotifications() ? notifications() : null)
                .add("Priority", priority()).add("RelatedOpsItems", hasRelatedOpsItems() ? relatedOpsItems() : null)
                .add("Status", statusAsString()).add("OpsItemId", opsItemId()).add("Version", version()).add("Title", title())
                .add("Source", source()).add("OperationalData", hasOperationalData() ? operationalData() : null)
                .add("Category", category()).add("Severity", severity()).add("ActualStartTime", actualStartTime())
                .add("ActualEndTime", actualEndTime()).add("PlannedStartTime", plannedStartTime())
                .add("PlannedEndTime", plannedEndTime()).add("OpsItemArn", opsItemArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "OpsItemType":
            return Optional.ofNullable(clazz.cast(opsItemType()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "LastModifiedBy":
            return Optional.ofNullable(clazz.cast(lastModifiedBy()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "Notifications":
            return Optional.ofNullable(clazz.cast(notifications()));
        case "Priority":
            return Optional.ofNullable(clazz.cast(priority()));
        case "RelatedOpsItems":
            return Optional.ofNullable(clazz.cast(relatedOpsItems()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "OpsItemId":
            return Optional.ofNullable(clazz.cast(opsItemId()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "OperationalData":
            return Optional.ofNullable(clazz.cast(operationalData()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "Severity":
            return Optional.ofNullable(clazz.cast(severity()));
        case "ActualStartTime":
            return Optional.ofNullable(clazz.cast(actualStartTime()));
        case "ActualEndTime":
            return Optional.ofNullable(clazz.cast(actualEndTime()));
        case "PlannedStartTime":
            return Optional.ofNullable(clazz.cast(plannedStartTime()));
        case "PlannedEndTime":
            return Optional.ofNullable(clazz.cast(plannedEndTime()));
        case "OpsItemArn":
            return Optional.ofNullable(clazz.cast(opsItemArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("CreatedBy", CREATED_BY_FIELD);
        map.put("OpsItemType", OPS_ITEM_TYPE_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("LastModifiedBy", LAST_MODIFIED_BY_FIELD);
        map.put("LastModifiedTime", LAST_MODIFIED_TIME_FIELD);
        map.put("Notifications", NOTIFICATIONS_FIELD);
        map.put("Priority", PRIORITY_FIELD);
        map.put("RelatedOpsItems", RELATED_OPS_ITEMS_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("OpsItemId", OPS_ITEM_ID_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("Title", TITLE_FIELD);
        map.put("Source", SOURCE_FIELD);
        map.put("OperationalData", OPERATIONAL_DATA_FIELD);
        map.put("Category", CATEGORY_FIELD);
        map.put("Severity", SEVERITY_FIELD);
        map.put("ActualStartTime", ACTUAL_START_TIME_FIELD);
        map.put("ActualEndTime", ACTUAL_END_TIME_FIELD);
        map.put("PlannedStartTime", PLANNED_START_TIME_FIELD);
        map.put("PlannedEndTime", PLANNED_END_TIME_FIELD);
        map.put("OpsItemArn", OPS_ITEM_ARN_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OpsItem, T> g) {
        return obj -> g.apply((OpsItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpsItem> {
        /**
         * <p>
         * The ARN of the Amazon Web Services account that created the OpsItem.
         * </p>
         * 
         * @param createdBy
         *        The ARN of the Amazon Web Services account that created the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The type of OpsItem. Systems Manager supports the following types of OpsItems:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>/aws/issue</code>
         * </p>
         * <p>
         * This type of OpsItem is used for default OpsItems created by OpsCenter.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>/aws/changerequest</code>
         * </p>
         * <p>
         * This type of OpsItem is used by Change Manager for reviewing and approving or rejecting change requests.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>/aws/insight</code>
         * </p>
         * <p>
         * This type of OpsItem is used by OpsCenter for aggregating and reporting on duplicate OpsItems.
         * </p>
         * </li>
         * </ul>
         * 
         * @param opsItemType
         *        The type of OpsItem. Systems Manager supports the following types of OpsItems:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>/aws/issue</code>
         *        </p>
         *        <p>
         *        This type of OpsItem is used for default OpsItems created by OpsCenter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>/aws/changerequest</code>
         *        </p>
         *        <p>
         *        This type of OpsItem is used by Change Manager for reviewing and approving or rejecting change
         *        requests.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>/aws/insight</code>
         *        </p>
         *        <p>
         *        This type of OpsItem is used by OpsCenter for aggregating and reporting on duplicate OpsItems.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder opsItemType(String opsItemType);

        /**
         * <p>
         * The date and time the OpsItem was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time the OpsItem was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The OpsItem description.
         * </p>
         * 
         * @param description
         *        The OpsItem description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The ARN of the Amazon Web Services account that last updated the OpsItem.
         * </p>
         * 
         * @param lastModifiedBy
         *        The ARN of the Amazon Web Services account that last updated the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedBy(String lastModifiedBy);

        /**
         * <p>
         * The date and time the OpsItem was last updated.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time the OpsItem was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
         * notifications are sent when this OpsItem is edited or changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
         *        notifications are sent when this OpsItem is edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(Collection<OpsItemNotification> notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
         * notifications are sent when this OpsItem is edited or changed.
         * </p>
         * 
         * @param notifications
         *        The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
         *        notifications are sent when this OpsItem is edited or changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notifications(OpsItemNotification... notifications);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an Amazon Simple Notification Service (Amazon SNS) topic where
         * notifications are sent when this OpsItem is edited or changed.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.OpsItemNotification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.OpsItemNotification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.OpsItemNotification.Builder#build()} is called immediately
         * and its result is passed to {@link #notifications(List<OpsItemNotification>)}.
         * 
         * @param notifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.OpsItemNotification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notifications(java.util.Collection<OpsItemNotification>)
         */
        Builder notifications(Consumer<OpsItemNotification.Builder>... notifications);

        /**
         * <p>
         * The importance of this OpsItem in relation to other OpsItems in the system.
         * </p>
         * 
         * @param priority
         *        The importance of this OpsItem in relation to other OpsItems in the system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder priority(Integer priority);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItem. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * 
         * @param relatedOpsItems
         *        One or more OpsItems that share something in common with the current OpsItem. For example, related
         *        OpsItems can include OpsItems with similar error messages, impacted resources, or statuses for the
         *        impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems);

        /**
         * <p>
         * One or more OpsItems that share something in common with the current OpsItem. For example, related OpsItems
         * can include OpsItems with similar error messages, impacted resources, or statuses for the impacted resource.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.RelatedOpsItem.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ssm.model.RelatedOpsItem#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.RelatedOpsItem.Builder#build()} is called immediately and
         * its result is passed to {@link #relatedOpsItems(List<RelatedOpsItem>)}.
         * 
         * @param relatedOpsItems
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.RelatedOpsItem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #relatedOpsItems(java.util.Collection<RelatedOpsItem>)
         */
        Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems);

        /**
         * <p>
         * The OpsItem status. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         * >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         *        >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The OpsItem status. For more information, see <a href=
         * "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         * >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param status
         *        The OpsItem status. For more information, see <a href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-working-with-OpsItems-editing-details.html"
         *        >Editing OpsItem details</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @see OpsItemStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpsItemStatus
         */
        Builder status(OpsItemStatus status);

        /**
         * <p>
         * The ID of the OpsItem.
         * </p>
         * 
         * @param opsItemId
         *        The ID of the OpsItem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder opsItemId(String opsItemId);

        /**
         * <p>
         * The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
         * </p>
         * 
         * @param version
         *        The version of this OpsItem. Each time the OpsItem is edited the version number increments by one.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A short heading that describes the nature of the OpsItem and the impacted resource.
         * </p>
         * 
         * @param title
         *        A short heading that describes the nature of the OpsItem and the impacted resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The origin of the OpsItem, such as Amazon EC2 or Systems Manager. The impacted resource is a subset of
         * source.
         * </p>
         * 
         * @param source
         *        The origin of the OpsItem, such as Amazon EC2 or Systems Manager. The impacted resource is a subset of
         *        source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(String source);

        /**
         * <p>
         * Operational data is custom data that provides useful reference details about the OpsItem. For example, you
         * can specify log files, error strings, license keys, troubleshooting tips, or other relevant data. You enter
         * operational data as key-value pairs. The key has a maximum length of 128 characters. The value has a maximum
         * size of 20 KB.
         * </p>
         * <important>
         * <p>
         * Operational data keys <i>can't</i> begin with the following: <code>amazon</code>, <code>aws</code>,
         * <code>amzn</code>, <code>ssm</code>, <code>/amazon</code>, <code>/aws</code>, <code>/amzn</code>,
         * <code>/ssm</code>.
         * </p>
         * </important>
         * <p>
         * You can choose to make the data searchable by other users in the account or you can restrict search access.
         * Searchable data means that all users with access to the OpsItem Overview page (as provided by the
         * <a>DescribeOpsItems</a> API operation) can view and search on the specified data. Operational data that isn't
         * searchable is only viewable by users who have access to the OpsItem (as provided by the <a>GetOpsItem</a> API
         * operation).
         * </p>
         * <p>
         * Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the request. Use
         * the <code>/aws/automations</code> key in OperationalData to associate an Automation runbook with the OpsItem.
         * To view Amazon Web Services CLI example commands that use these keys, see <a
         * href="https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-manually-create-OpsItems.html"
         * >Creating OpsItems manually</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * </p>
         * 
         * @param operationalData
         *        Operational data is custom data that provides useful reference details about the OpsItem. For example,
         *        you can specify log files, error strings, license keys, troubleshooting tips, or other relevant data.
         *        You enter operational data as key-value pairs. The key has a maximum length of 128 characters. The
         *        value has a maximum size of 20 KB.</p> <important>
         *        <p>
         *        Operational data keys <i>can't</i> begin with the following: <code>amazon</code>, <code>aws</code>,
         *        <code>amzn</code>, <code>ssm</code>, <code>/amazon</code>, <code>/aws</code>, <code>/amzn</code>,
         *        <code>/ssm</code>.
         *        </p>
         *        </important>
         *        <p>
         *        You can choose to make the data searchable by other users in the account or you can restrict search
         *        access. Searchable data means that all users with access to the OpsItem Overview page (as provided by
         *        the <a>DescribeOpsItems</a> API operation) can view and search on the specified data. Operational data
         *        that isn't searchable is only viewable by users who have access to the OpsItem (as provided by the
         *        <a>GetOpsItem</a> API operation).
         *        </p>
         *        <p>
         *        Use the <code>/aws/resources</code> key in OperationalData to specify a related resource in the
         *        request. Use the <code>/aws/automations</code> key in OperationalData to associate an Automation
         *        runbook with the OpsItem. To view Amazon Web Services CLI example commands that use these keys, see <a
         *        href=
         *        "https://docs.aws.amazon.com/systems-manager/latest/userguide/OpsCenter-manually-create-OpsItems.html"
         *        >Creating OpsItems manually</a> in the <i>Amazon Web Services Systems Manager User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operationalData(Map<String, OpsItemDataValue> operationalData);

        /**
         * <p>
         * An OpsItem category. Category options include: Availability, Cost, Performance, Recovery, Security.
         * </p>
         * 
         * @param category
         *        An OpsItem category. Category options include: Availability, Cost, Performance, Recovery, Security.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String category);

        /**
         * <p>
         * The severity of the OpsItem. Severity options range from 1 to 4.
         * </p>
         * 
         * @param severity
         *        The severity of the OpsItem. Severity options range from 1 to 4.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder severity(String severity);

        /**
         * <p>
         * The time a runbook workflow started. Currently reported only for the OpsItem type
         * <code>/aws/changerequest</code>.
         * </p>
         * 
         * @param actualStartTime
         *        The time a runbook workflow started. Currently reported only for the OpsItem type
         *        <code>/aws/changerequest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actualStartTime(Instant actualStartTime);

        /**
         * <p>
         * The time a runbook workflow ended. Currently reported only for the OpsItem type
         * <code>/aws/changerequest</code>.
         * </p>
         * 
         * @param actualEndTime
         *        The time a runbook workflow ended. Currently reported only for the OpsItem type
         *        <code>/aws/changerequest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder actualEndTime(Instant actualEndTime);

        /**
         * <p>
         * The time specified in a change request for a runbook workflow to start. Currently supported only for the
         * OpsItem type <code>/aws/changerequest</code>.
         * </p>
         * 
         * @param plannedStartTime
         *        The time specified in a change request for a runbook workflow to start. Currently supported only for
         *        the OpsItem type <code>/aws/changerequest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder plannedStartTime(Instant plannedStartTime);

        /**
         * <p>
         * The time specified in a change request for a runbook workflow to end. Currently supported only for the
         * OpsItem type <code>/aws/changerequest</code>.
         * </p>
         * 
         * @param plannedEndTime
         *        The time specified in a change request for a runbook workflow to end. Currently supported only for the
         *        OpsItem type <code>/aws/changerequest</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder plannedEndTime(Instant plannedEndTime);

        /**
         * <p>
         * The OpsItem Amazon Resource Name (ARN).
         * </p>
         * 
         * @param opsItemArn
         *        The OpsItem Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder opsItemArn(String opsItemArn);
    }

    static final class BuilderImpl implements Builder {
        private String createdBy;

        private String opsItemType;

        private Instant createdTime;

        private String description;

        private String lastModifiedBy;

        private Instant lastModifiedTime;

        private List<OpsItemNotification> notifications = DefaultSdkAutoConstructList.getInstance();

        private Integer priority;

        private List<RelatedOpsItem> relatedOpsItems = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private String opsItemId;

        private String version;

        private String title;

        private String source;

        private Map<String, OpsItemDataValue> operationalData = DefaultSdkAutoConstructMap.getInstance();

        private String category;

        private String severity;

        private Instant actualStartTime;

        private Instant actualEndTime;

        private Instant plannedStartTime;

        private Instant plannedEndTime;

        private String opsItemArn;

        private BuilderImpl() {
        }

        private BuilderImpl(OpsItem model) {
            createdBy(model.createdBy);
            opsItemType(model.opsItemType);
            createdTime(model.createdTime);
            description(model.description);
            lastModifiedBy(model.lastModifiedBy);
            lastModifiedTime(model.lastModifiedTime);
            notifications(model.notifications);
            priority(model.priority);
            relatedOpsItems(model.relatedOpsItems);
            status(model.status);
            opsItemId(model.opsItemId);
            version(model.version);
            title(model.title);
            source(model.source);
            operationalData(model.operationalData);
            category(model.category);
            severity(model.severity);
            actualStartTime(model.actualStartTime);
            actualEndTime(model.actualEndTime);
            plannedStartTime(model.plannedStartTime);
            plannedEndTime(model.plannedEndTime);
            opsItemArn(model.opsItemArn);
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getOpsItemType() {
            return opsItemType;
        }

        public final void setOpsItemType(String opsItemType) {
            this.opsItemType = opsItemType;
        }

        @Override
        public final Builder opsItemType(String opsItemType) {
            this.opsItemType = opsItemType;
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getLastModifiedBy() {
            return lastModifiedBy;
        }

        public final void setLastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
        }

        @Override
        public final Builder lastModifiedBy(String lastModifiedBy) {
            this.lastModifiedBy = lastModifiedBy;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final List<OpsItemNotification.Builder> getNotifications() {
            List<OpsItemNotification.Builder> result = OpsItemNotificationsCopier.copyToBuilder(this.notifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNotifications(Collection<OpsItemNotification.BuilderImpl> notifications) {
            this.notifications = OpsItemNotificationsCopier.copyFromBuilder(notifications);
        }

        @Override
        public final Builder notifications(Collection<OpsItemNotification> notifications) {
            this.notifications = OpsItemNotificationsCopier.copy(notifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(OpsItemNotification... notifications) {
            notifications(Arrays.asList(notifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notifications(Consumer<OpsItemNotification.Builder>... notifications) {
            notifications(Stream.of(notifications).map(c -> OpsItemNotification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Integer getPriority() {
            return priority;
        }

        public final void setPriority(Integer priority) {
            this.priority = priority;
        }

        @Override
        public final Builder priority(Integer priority) {
            this.priority = priority;
            return this;
        }

        public final List<RelatedOpsItem.Builder> getRelatedOpsItems() {
            List<RelatedOpsItem.Builder> result = RelatedOpsItemsCopier.copyToBuilder(this.relatedOpsItems);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRelatedOpsItems(Collection<RelatedOpsItem.BuilderImpl> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copyFromBuilder(relatedOpsItems);
        }

        @Override
        public final Builder relatedOpsItems(Collection<RelatedOpsItem> relatedOpsItems) {
            this.relatedOpsItems = RelatedOpsItemsCopier.copy(relatedOpsItems);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(RelatedOpsItem... relatedOpsItems) {
            relatedOpsItems(Arrays.asList(relatedOpsItems));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder relatedOpsItems(Consumer<RelatedOpsItem.Builder>... relatedOpsItems) {
            relatedOpsItems(Stream.of(relatedOpsItems).map(c -> RelatedOpsItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(OpsItemStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getOpsItemId() {
            return opsItemId;
        }

        public final void setOpsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
        }

        @Override
        public final Builder opsItemId(String opsItemId) {
            this.opsItemId = opsItemId;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getSource() {
            return source;
        }

        public final void setSource(String source) {
            this.source = source;
        }

        @Override
        public final Builder source(String source) {
            this.source = source;
            return this;
        }

        public final Map<String, OpsItemDataValue.Builder> getOperationalData() {
            Map<String, OpsItemDataValue.Builder> result = OpsItemOperationalDataCopier.copyToBuilder(this.operationalData);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setOperationalData(Map<String, OpsItemDataValue.BuilderImpl> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copyFromBuilder(operationalData);
        }

        @Override
        public final Builder operationalData(Map<String, OpsItemDataValue> operationalData) {
            this.operationalData = OpsItemOperationalDataCopier.copy(operationalData);
            return this;
        }

        public final String getCategory() {
            return category;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        public final String getSeverity() {
            return severity;
        }

        public final void setSeverity(String severity) {
            this.severity = severity;
        }

        @Override
        public final Builder severity(String severity) {
            this.severity = severity;
            return this;
        }

        public final Instant getActualStartTime() {
            return actualStartTime;
        }

        public final void setActualStartTime(Instant actualStartTime) {
            this.actualStartTime = actualStartTime;
        }

        @Override
        public final Builder actualStartTime(Instant actualStartTime) {
            this.actualStartTime = actualStartTime;
            return this;
        }

        public final Instant getActualEndTime() {
            return actualEndTime;
        }

        public final void setActualEndTime(Instant actualEndTime) {
            this.actualEndTime = actualEndTime;
        }

        @Override
        public final Builder actualEndTime(Instant actualEndTime) {
            this.actualEndTime = actualEndTime;
            return this;
        }

        public final Instant getPlannedStartTime() {
            return plannedStartTime;
        }

        public final void setPlannedStartTime(Instant plannedStartTime) {
            this.plannedStartTime = plannedStartTime;
        }

        @Override
        public final Builder plannedStartTime(Instant plannedStartTime) {
            this.plannedStartTime = plannedStartTime;
            return this;
        }

        public final Instant getPlannedEndTime() {
            return plannedEndTime;
        }

        public final void setPlannedEndTime(Instant plannedEndTime) {
            this.plannedEndTime = plannedEndTime;
        }

        @Override
        public final Builder plannedEndTime(Instant plannedEndTime) {
            this.plannedEndTime = plannedEndTime;
            return this;
        }

        public final String getOpsItemArn() {
            return opsItemArn;
        }

        public final void setOpsItemArn(String opsItemArn) {
            this.opsItemArn = opsItemArn;
        }

        @Override
        public final Builder opsItemArn(String opsItemArn) {
            this.opsItemArn = opsItemArn;
            return this;
        }

        @Override
        public OpsItem build() {
            return new OpsItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
