/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ssm.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a task execution performed as part of a maintenance window execution.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MaintenanceWindowExecutionTaskIdentity implements SdkPojo, Serializable,
        ToCopyableBuilder<MaintenanceWindowExecutionTaskIdentity.Builder, MaintenanceWindowExecutionTaskIdentity> {
    private static final SdkField<String> WINDOW_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WindowExecutionId").getter(getter(MaintenanceWindowExecutionTaskIdentity::windowExecutionId))
            .setter(setter(Builder::windowExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowExecutionId").build()).build();

    private static final SdkField<String> TASK_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskExecutionId").getter(getter(MaintenanceWindowExecutionTaskIdentity::taskExecutionId))
            .setter(setter(Builder::taskExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskExecutionId").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(MaintenanceWindowExecutionTaskIdentity::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusDetails").getter(getter(MaintenanceWindowExecutionTaskIdentity::statusDetails))
            .setter(setter(Builder::statusDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusDetails").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(MaintenanceWindowExecutionTaskIdentity::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(MaintenanceWindowExecutionTaskIdentity::endTime))
            .setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> TASK_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskArn").getter(getter(MaintenanceWindowExecutionTaskIdentity::taskArn))
            .setter(setter(Builder::taskArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskArn").build()).build();

    private static final SdkField<String> TASK_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TaskType").getter(getter(MaintenanceWindowExecutionTaskIdentity::taskTypeAsString))
            .setter(setter(Builder::taskType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskType").build()).build();

    private static final SdkField<AlarmConfiguration> ALARM_CONFIGURATION_FIELD = SdkField
            .<AlarmConfiguration> builder(MarshallingType.SDK_POJO).memberName("AlarmConfiguration")
            .getter(getter(MaintenanceWindowExecutionTaskIdentity::alarmConfiguration))
            .setter(setter(Builder::alarmConfiguration)).constructor(AlarmConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AlarmConfiguration").build())
            .build();

    private static final SdkField<List<AlarmStateInformation>> TRIGGERED_ALARMS_FIELD = SdkField
            .<List<AlarmStateInformation>> builder(MarshallingType.LIST)
            .memberName("TriggeredAlarms")
            .getter(getter(MaintenanceWindowExecutionTaskIdentity::triggeredAlarms))
            .setter(setter(Builder::triggeredAlarms))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TriggeredAlarms").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AlarmStateInformation> builder(MarshallingType.SDK_POJO)
                                            .constructor(AlarmStateInformation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(WINDOW_EXECUTION_ID_FIELD,
            TASK_EXECUTION_ID_FIELD, STATUS_FIELD, STATUS_DETAILS_FIELD, START_TIME_FIELD, END_TIME_FIELD, TASK_ARN_FIELD,
            TASK_TYPE_FIELD, ALARM_CONFIGURATION_FIELD, TRIGGERED_ALARMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String windowExecutionId;

    private final String taskExecutionId;

    private final String status;

    private final String statusDetails;

    private final Instant startTime;

    private final Instant endTime;

    private final String taskArn;

    private final String taskType;

    private final AlarmConfiguration alarmConfiguration;

    private final List<AlarmStateInformation> triggeredAlarms;

    private MaintenanceWindowExecutionTaskIdentity(BuilderImpl builder) {
        this.windowExecutionId = builder.windowExecutionId;
        this.taskExecutionId = builder.taskExecutionId;
        this.status = builder.status;
        this.statusDetails = builder.statusDetails;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.taskArn = builder.taskArn;
        this.taskType = builder.taskType;
        this.alarmConfiguration = builder.alarmConfiguration;
        this.triggeredAlarms = builder.triggeredAlarms;
    }

    /**
     * <p>
     * The ID of the maintenance window execution that ran the task.
     * </p>
     * 
     * @return The ID of the maintenance window execution that ran the task.
     */
    public final String windowExecutionId() {
        return windowExecutionId;
    }

    /**
     * <p>
     * The ID of the specific task execution in the maintenance window execution.
     * </p>
     * 
     * @return The ID of the specific task execution in the maintenance window execution.
     */
    public final String taskExecutionId() {
        return taskExecutionId;
    }

    /**
     * <p>
     * The status of the task execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task execution.
     * @see MaintenanceWindowExecutionStatus
     */
    public final MaintenanceWindowExecutionStatus status() {
        return MaintenanceWindowExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the task execution.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link MaintenanceWindowExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the task execution.
     * @see MaintenanceWindowExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The details explaining the status of the task execution. Not available for all status values.
     * </p>
     * 
     * @return The details explaining the status of the task execution. Not available for all status values.
     */
    public final String statusDetails() {
        return statusDetails;
    }

    /**
     * <p>
     * The time the task execution started.
     * </p>
     * 
     * @return The time the task execution started.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time the task execution finished.
     * </p>
     * 
     * @return The time the task execution finished.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the task that ran.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the task that ran.
     */
    public final String taskArn() {
        return taskArn;
    }

    /**
     * <p>
     * The type of task that ran.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task that ran.
     * @see MaintenanceWindowTaskType
     */
    public final MaintenanceWindowTaskType taskType() {
        return MaintenanceWindowTaskType.fromValue(taskType);
    }

    /**
     * <p>
     * The type of task that ran.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MaintenanceWindowTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task that ran.
     * @see MaintenanceWindowTaskType
     */
    public final String taskTypeAsString() {
        return taskType;
    }

    /**
     * <p>
     * The details for the CloudWatch alarm applied to your maintenance window task.
     * </p>
     * 
     * @return The details for the CloudWatch alarm applied to your maintenance window task.
     */
    public final AlarmConfiguration alarmConfiguration() {
        return alarmConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the TriggeredAlarms property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTriggeredAlarms() {
        return triggeredAlarms != null && !(triggeredAlarms instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The CloudWatch alarm that was invoked by the maintenance window task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTriggeredAlarms} method.
     * </p>
     * 
     * @return The CloudWatch alarm that was invoked by the maintenance window task.
     */
    public final List<AlarmStateInformation> triggeredAlarms() {
        return triggeredAlarms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(windowExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(taskExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusDetails());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(taskArn());
        hashCode = 31 * hashCode + Objects.hashCode(taskTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(alarmConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTriggeredAlarms() ? triggeredAlarms() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MaintenanceWindowExecutionTaskIdentity)) {
            return false;
        }
        MaintenanceWindowExecutionTaskIdentity other = (MaintenanceWindowExecutionTaskIdentity) obj;
        return Objects.equals(windowExecutionId(), other.windowExecutionId())
                && Objects.equals(taskExecutionId(), other.taskExecutionId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusDetails(), other.statusDetails()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime()) && Objects.equals(taskArn(), other.taskArn())
                && Objects.equals(taskTypeAsString(), other.taskTypeAsString())
                && Objects.equals(alarmConfiguration(), other.alarmConfiguration())
                && hasTriggeredAlarms() == other.hasTriggeredAlarms()
                && Objects.equals(triggeredAlarms(), other.triggeredAlarms());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MaintenanceWindowExecutionTaskIdentity").add("WindowExecutionId", windowExecutionId())
                .add("TaskExecutionId", taskExecutionId()).add("Status", statusAsString()).add("StatusDetails", statusDetails())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("TaskArn", taskArn())
                .add("TaskType", taskTypeAsString()).add("AlarmConfiguration", alarmConfiguration())
                .add("TriggeredAlarms", hasTriggeredAlarms() ? triggeredAlarms() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "WindowExecutionId":
            return Optional.ofNullable(clazz.cast(windowExecutionId()));
        case "TaskExecutionId":
            return Optional.ofNullable(clazz.cast(taskExecutionId()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusDetails":
            return Optional.ofNullable(clazz.cast(statusDetails()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "TaskArn":
            return Optional.ofNullable(clazz.cast(taskArn()));
        case "TaskType":
            return Optional.ofNullable(clazz.cast(taskTypeAsString()));
        case "AlarmConfiguration":
            return Optional.ofNullable(clazz.cast(alarmConfiguration()));
        case "TriggeredAlarms":
            return Optional.ofNullable(clazz.cast(triggeredAlarms()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MaintenanceWindowExecutionTaskIdentity, T> g) {
        return obj -> g.apply((MaintenanceWindowExecutionTaskIdentity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MaintenanceWindowExecutionTaskIdentity> {
        /**
         * <p>
         * The ID of the maintenance window execution that ran the task.
         * </p>
         * 
         * @param windowExecutionId
         *        The ID of the maintenance window execution that ran the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowExecutionId(String windowExecutionId);

        /**
         * <p>
         * The ID of the specific task execution in the maintenance window execution.
         * </p>
         * 
         * @param taskExecutionId
         *        The ID of the specific task execution in the maintenance window execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskExecutionId(String taskExecutionId);

        /**
         * <p>
         * The status of the task execution.
         * </p>
         * 
         * @param status
         *        The status of the task execution.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the task execution.
         * </p>
         * 
         * @param status
         *        The status of the task execution.
         * @see MaintenanceWindowExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowExecutionStatus
         */
        Builder status(MaintenanceWindowExecutionStatus status);

        /**
         * <p>
         * The details explaining the status of the task execution. Not available for all status values.
         * </p>
         * 
         * @param statusDetails
         *        The details explaining the status of the task execution. Not available for all status values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusDetails(String statusDetails);

        /**
         * <p>
         * The time the task execution started.
         * </p>
         * 
         * @param startTime
         *        The time the task execution started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time the task execution finished.
         * </p>
         * 
         * @param endTime
         *        The time the task execution finished.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the task that ran.
         * </p>
         * 
         * @param taskArn
         *        The Amazon Resource Name (ARN) of the task that ran.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskArn(String taskArn);

        /**
         * <p>
         * The type of task that ran.
         * </p>
         * 
         * @param taskType
         *        The type of task that ran.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(String taskType);

        /**
         * <p>
         * The type of task that ran.
         * </p>
         * 
         * @param taskType
         *        The type of task that ran.
         * @see MaintenanceWindowTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaintenanceWindowTaskType
         */
        Builder taskType(MaintenanceWindowTaskType taskType);

        /**
         * <p>
         * The details for the CloudWatch alarm applied to your maintenance window task.
         * </p>
         * 
         * @param alarmConfiguration
         *        The details for the CloudWatch alarm applied to your maintenance window task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarmConfiguration(AlarmConfiguration alarmConfiguration);

        /**
         * <p>
         * The details for the CloudWatch alarm applied to your maintenance window task.
         * </p>
         * This is a convenience method that creates an instance of the {@link AlarmConfiguration.Builder} avoiding the
         * need to create one manually via {@link AlarmConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AlarmConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #alarmConfiguration(AlarmConfiguration)}.
         * 
         * @param alarmConfiguration
         *        a consumer that will call methods on {@link AlarmConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarmConfiguration(AlarmConfiguration)
         */
        default Builder alarmConfiguration(Consumer<AlarmConfiguration.Builder> alarmConfiguration) {
            return alarmConfiguration(AlarmConfiguration.builder().applyMutation(alarmConfiguration).build());
        }

        /**
         * <p>
         * The CloudWatch alarm that was invoked by the maintenance window task.
         * </p>
         * 
         * @param triggeredAlarms
         *        The CloudWatch alarm that was invoked by the maintenance window task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggeredAlarms(Collection<AlarmStateInformation> triggeredAlarms);

        /**
         * <p>
         * The CloudWatch alarm that was invoked by the maintenance window task.
         * </p>
         * 
         * @param triggeredAlarms
         *        The CloudWatch alarm that was invoked by the maintenance window task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggeredAlarms(AlarmStateInformation... triggeredAlarms);

        /**
         * <p>
         * The CloudWatch alarm that was invoked by the maintenance window task.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ssm.model.AlarmStateInformation.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ssm.model.AlarmStateInformation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ssm.model.AlarmStateInformation.Builder#build()} is called immediately
         * and its result is passed to {@link #triggeredAlarms(List<AlarmStateInformation>)}.
         * 
         * @param triggeredAlarms
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ssm.model.AlarmStateInformation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #triggeredAlarms(java.util.Collection<AlarmStateInformation>)
         */
        Builder triggeredAlarms(Consumer<AlarmStateInformation.Builder>... triggeredAlarms);
    }

    static final class BuilderImpl implements Builder {
        private String windowExecutionId;

        private String taskExecutionId;

        private String status;

        private String statusDetails;

        private Instant startTime;

        private Instant endTime;

        private String taskArn;

        private String taskType;

        private AlarmConfiguration alarmConfiguration;

        private List<AlarmStateInformation> triggeredAlarms = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MaintenanceWindowExecutionTaskIdentity model) {
            windowExecutionId(model.windowExecutionId);
            taskExecutionId(model.taskExecutionId);
            status(model.status);
            statusDetails(model.statusDetails);
            startTime(model.startTime);
            endTime(model.endTime);
            taskArn(model.taskArn);
            taskType(model.taskType);
            alarmConfiguration(model.alarmConfiguration);
            triggeredAlarms(model.triggeredAlarms);
        }

        public final String getWindowExecutionId() {
            return windowExecutionId;
        }

        public final void setWindowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
        }

        @Override
        public final Builder windowExecutionId(String windowExecutionId) {
            this.windowExecutionId = windowExecutionId;
            return this;
        }

        public final String getTaskExecutionId() {
            return taskExecutionId;
        }

        public final void setTaskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
        }

        @Override
        public final Builder taskExecutionId(String taskExecutionId) {
            this.taskExecutionId = taskExecutionId;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(MaintenanceWindowExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getStatusDetails() {
            return statusDetails;
        }

        public final void setStatusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
        }

        @Override
        public final Builder statusDetails(String statusDetails) {
            this.statusDetails = statusDetails;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getTaskArn() {
            return taskArn;
        }

        public final void setTaskArn(String taskArn) {
            this.taskArn = taskArn;
        }

        @Override
        public final Builder taskArn(String taskArn) {
            this.taskArn = taskArn;
            return this;
        }

        public final String getTaskType() {
            return taskType;
        }

        public final void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        @Override
        public final Builder taskType(String taskType) {
            this.taskType = taskType;
            return this;
        }

        @Override
        public final Builder taskType(MaintenanceWindowTaskType taskType) {
            this.taskType(taskType == null ? null : taskType.toString());
            return this;
        }

        public final AlarmConfiguration.Builder getAlarmConfiguration() {
            return alarmConfiguration != null ? alarmConfiguration.toBuilder() : null;
        }

        public final void setAlarmConfiguration(AlarmConfiguration.BuilderImpl alarmConfiguration) {
            this.alarmConfiguration = alarmConfiguration != null ? alarmConfiguration.build() : null;
        }

        @Override
        public final Builder alarmConfiguration(AlarmConfiguration alarmConfiguration) {
            this.alarmConfiguration = alarmConfiguration;
            return this;
        }

        public final List<AlarmStateInformation.Builder> getTriggeredAlarms() {
            List<AlarmStateInformation.Builder> result = AlarmStateInformationListCopier.copyToBuilder(this.triggeredAlarms);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTriggeredAlarms(Collection<AlarmStateInformation.BuilderImpl> triggeredAlarms) {
            this.triggeredAlarms = AlarmStateInformationListCopier.copyFromBuilder(triggeredAlarms);
        }

        @Override
        public final Builder triggeredAlarms(Collection<AlarmStateInformation> triggeredAlarms) {
            this.triggeredAlarms = AlarmStateInformationListCopier.copy(triggeredAlarms);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggeredAlarms(AlarmStateInformation... triggeredAlarms) {
            triggeredAlarms(Arrays.asList(triggeredAlarms));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder triggeredAlarms(Consumer<AlarmStateInformation.Builder>... triggeredAlarms) {
            triggeredAlarms(Stream.of(triggeredAlarms).map(c -> AlarmStateInformation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public MaintenanceWindowExecutionTaskIdentity build() {
            return new MaintenanceWindowExecutionTaskIdentity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
