/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The serializer that you want Firehose to use to convert data to the target format before writing it to Amazon S3.
 * Firehose supports two types of serializers: the <a
 * href="https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html">ORC SerDe</a> and
 * the <a
 * href="https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html"
 * >Parquet SerDe</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Serializer implements SdkPojo, Serializable, ToCopyableBuilder<Serializer.Builder, Serializer> {
    private static final SdkField<ParquetSerDe> PARQUET_SER_DE_FIELD = SdkField.<ParquetSerDe> builder(MarshallingType.SDK_POJO)
            .memberName("ParquetSerDe").getter(getter(Serializer::parquetSerDe)).setter(setter(Builder::parquetSerDe))
            .constructor(ParquetSerDe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParquetSerDe").build()).build();

    private static final SdkField<OrcSerDe> ORC_SER_DE_FIELD = SdkField.<OrcSerDe> builder(MarshallingType.SDK_POJO)
            .memberName("OrcSerDe").getter(getter(Serializer::orcSerDe)).setter(setter(Builder::orcSerDe))
            .constructor(OrcSerDe::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrcSerDe").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PARQUET_SER_DE_FIELD,
            ORC_SER_DE_FIELD));

    private static final long serialVersionUID = 1L;

    private final ParquetSerDe parquetSerDe;

    private final OrcSerDe orcSerDe;

    private Serializer(BuilderImpl builder) {
        this.parquetSerDe = builder.parquetSerDe;
        this.orcSerDe = builder.orcSerDe;
    }

    /**
     * <p>
     * A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more
     * information, see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
     * </p>
     * 
     * @return A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more
     *         information, see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
     */
    public final ParquetSerDe parquetSerDe() {
        return parquetSerDe;
    }

    /**
     * <p>
     * A serializer to use for converting data to the ORC format before storing it in Amazon S3. For more information,
     * see <a href="https://orc.apache.org/docs/">Apache ORC</a>.
     * </p>
     * 
     * @return A serializer to use for converting data to the ORC format before storing it in Amazon S3. For more
     *         information, see <a href="https://orc.apache.org/docs/">Apache ORC</a>.
     */
    public final OrcSerDe orcSerDe() {
        return orcSerDe;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(parquetSerDe());
        hashCode = 31 * hashCode + Objects.hashCode(orcSerDe());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Serializer)) {
            return false;
        }
        Serializer other = (Serializer) obj;
        return Objects.equals(parquetSerDe(), other.parquetSerDe()) && Objects.equals(orcSerDe(), other.orcSerDe());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Serializer").add("ParquetSerDe", parquetSerDe()).add("OrcSerDe", orcSerDe()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParquetSerDe":
            return Optional.ofNullable(clazz.cast(parquetSerDe()));
        case "OrcSerDe":
            return Optional.ofNullable(clazz.cast(orcSerDe()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Serializer, T> g) {
        return obj -> g.apply((Serializer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Serializer> {
        /**
         * <p>
         * A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more
         * information, see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
         * </p>
         * 
         * @param parquetSerDe
         *        A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more
         *        information, see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parquetSerDe(ParquetSerDe parquetSerDe);

        /**
         * <p>
         * A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more
         * information, see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link ParquetSerDe.Builder} avoiding the need
         * to create one manually via {@link ParquetSerDe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ParquetSerDe.Builder#build()} is called immediately and its
         * result is passed to {@link #parquetSerDe(ParquetSerDe)}.
         * 
         * @param parquetSerDe
         *        a consumer that will call methods on {@link ParquetSerDe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parquetSerDe(ParquetSerDe)
         */
        default Builder parquetSerDe(Consumer<ParquetSerDe.Builder> parquetSerDe) {
            return parquetSerDe(ParquetSerDe.builder().applyMutation(parquetSerDe).build());
        }

        /**
         * <p>
         * A serializer to use for converting data to the ORC format before storing it in Amazon S3. For more
         * information, see <a href="https://orc.apache.org/docs/">Apache ORC</a>.
         * </p>
         * 
         * @param orcSerDe
         *        A serializer to use for converting data to the ORC format before storing it in Amazon S3. For more
         *        information, see <a href="https://orc.apache.org/docs/">Apache ORC</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder orcSerDe(OrcSerDe orcSerDe);

        /**
         * <p>
         * A serializer to use for converting data to the ORC format before storing it in Amazon S3. For more
         * information, see <a href="https://orc.apache.org/docs/">Apache ORC</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link OrcSerDe.Builder} avoiding the need to
         * create one manually via {@link OrcSerDe#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OrcSerDe.Builder#build()} is called immediately and its result is
         * passed to {@link #orcSerDe(OrcSerDe)}.
         * 
         * @param orcSerDe
         *        a consumer that will call methods on {@link OrcSerDe.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #orcSerDe(OrcSerDe)
         */
        default Builder orcSerDe(Consumer<OrcSerDe.Builder> orcSerDe) {
            return orcSerDe(OrcSerDe.builder().applyMutation(orcSerDe).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private ParquetSerDe parquetSerDe;

        private OrcSerDe orcSerDe;

        private BuilderImpl() {
        }

        private BuilderImpl(Serializer model) {
            parquetSerDe(model.parquetSerDe);
            orcSerDe(model.orcSerDe);
        }

        public final ParquetSerDe.Builder getParquetSerDe() {
            return parquetSerDe != null ? parquetSerDe.toBuilder() : null;
        }

        public final void setParquetSerDe(ParquetSerDe.BuilderImpl parquetSerDe) {
            this.parquetSerDe = parquetSerDe != null ? parquetSerDe.build() : null;
        }

        @Override
        public final Builder parquetSerDe(ParquetSerDe parquetSerDe) {
            this.parquetSerDe = parquetSerDe;
            return this;
        }

        public final OrcSerDe.Builder getOrcSerDe() {
            return orcSerDe != null ? orcSerDe.toBuilder() : null;
        }

        public final void setOrcSerDe(OrcSerDe.BuilderImpl orcSerDe) {
            this.orcSerDe = orcSerDe != null ? orcSerDe.build() : null;
        }

        @Override
        public final Builder orcSerDe(OrcSerDe orcSerDe) {
            this.orcSerDe = orcSerDe;
            return this;
        }

        @Override
        public Serializer build() {
            return new Serializer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
