/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.firehose.internal.FirehoseServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.firehose.model.ConcurrentModificationException;
import software.amazon.awssdk.services.firehose.model.CreateDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.CreateDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.DeleteDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.DeleteDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.DescribeDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.DescribeDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.FirehoseException;
import software.amazon.awssdk.services.firehose.model.GetKinesisStreamRequest;
import software.amazon.awssdk.services.firehose.model.GetKinesisStreamResponse;
import software.amazon.awssdk.services.firehose.model.InvalidArgumentException;
import software.amazon.awssdk.services.firehose.model.InvalidKmsResourceException;
import software.amazon.awssdk.services.firehose.model.InvalidSourceException;
import software.amazon.awssdk.services.firehose.model.InvalidStreamTypeException;
import software.amazon.awssdk.services.firehose.model.LimitExceededException;
import software.amazon.awssdk.services.firehose.model.ListDeliveryStreamsRequest;
import software.amazon.awssdk.services.firehose.model.ListDeliveryStreamsResponse;
import software.amazon.awssdk.services.firehose.model.ListTagsForDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.ListTagsForDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.PutRecordBatchRequest;
import software.amazon.awssdk.services.firehose.model.PutRecordBatchResponse;
import software.amazon.awssdk.services.firehose.model.PutRecordRequest;
import software.amazon.awssdk.services.firehose.model.PutRecordResponse;
import software.amazon.awssdk.services.firehose.model.ResourceInUseException;
import software.amazon.awssdk.services.firehose.model.ResourceNotFoundException;
import software.amazon.awssdk.services.firehose.model.ServiceUnavailableException;
import software.amazon.awssdk.services.firehose.model.StartDeliveryStreamEncryptionRequest;
import software.amazon.awssdk.services.firehose.model.StartDeliveryStreamEncryptionResponse;
import software.amazon.awssdk.services.firehose.model.StopDeliveryStreamEncryptionRequest;
import software.amazon.awssdk.services.firehose.model.StopDeliveryStreamEncryptionResponse;
import software.amazon.awssdk.services.firehose.model.TagDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.TagDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.TagrisAccessDeniedException;
import software.amazon.awssdk.services.firehose.model.TagrisInternalServiceException;
import software.amazon.awssdk.services.firehose.model.TagrisInvalidArnException;
import software.amazon.awssdk.services.firehose.model.TagrisInvalidParameterException;
import software.amazon.awssdk.services.firehose.model.TagrisPartialResourcesExistResultsException;
import software.amazon.awssdk.services.firehose.model.TagrisThrottledException;
import software.amazon.awssdk.services.firehose.model.UntagDeliveryStreamRequest;
import software.amazon.awssdk.services.firehose.model.UntagDeliveryStreamResponse;
import software.amazon.awssdk.services.firehose.model.UpdateDestinationRequest;
import software.amazon.awssdk.services.firehose.model.UpdateDestinationResponse;
import software.amazon.awssdk.services.firehose.model.VerifyResourcesExistForTagrisRequest;
import software.amazon.awssdk.services.firehose.model.VerifyResourcesExistForTagrisResponse;
import software.amazon.awssdk.services.firehose.transform.CreateDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.DeleteDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.DescribeDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.GetKinesisStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.ListDeliveryStreamsRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.ListTagsForDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.PutRecordBatchRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.PutRecordRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.StartDeliveryStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.StopDeliveryStreamEncryptionRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.TagDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.UntagDeliveryStreamRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.UpdateDestinationRequestMarshaller;
import software.amazon.awssdk.services.firehose.transform.VerifyResourcesExistForTagrisRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FirehoseClient}.
 *
 * @see FirehoseClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFirehoseClient implements FirehoseClient {
    private static final Logger log = Logger.loggerFor(DefaultFirehoseClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFirehoseClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a Firehose delivery stream.
     * </p>
     * <p>
     * By default, you can create up to 50 delivery streams per Amazon Web Services Region.
     * </p>
     * <p>
     * This is an asynchronous operation that immediately returns. The initial status of the delivery stream is
     * <code>CREATING</code>. After the delivery stream is created, its status is <code>ACTIVE</code> and it now accepts
     * data. If the delivery stream creation fails, the status transitions to <code>CREATING_FAILED</code>. Attempts to
     * send data to a delivery stream that is not in the <code>ACTIVE</code> state cause an exception. To check the
     * state of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * If the status of a delivery stream is <code>CREATING_FAILED</code>, this status doesn't change, and you can't
     * invoke <code>CreateDeliveryStream</code> again on it. However, you can invoke the <a>DeleteDeliveryStream</a>
     * operation to delete it.
     * </p>
     * <p>
     * A Firehose delivery stream can be configured to receive records directly from providers using <a>PutRecord</a> or
     * <a>PutRecordBatch</a>, or it can be configured to use an existing Kinesis stream as its source. To specify a
     * Kinesis data stream as input, set the <code>DeliveryStreamType</code> parameter to
     * <code>KinesisStreamAsSource</code>, and provide the Kinesis stream Amazon Resource Name (ARN) and role ARN in the
     * <code>KinesisStreamSourceConfiguration</code> parameter.
     * </p>
     * <p>
     * To create a delivery stream with server-side encryption (SSE) enabled, include
     * <a>DeliveryStreamEncryptionConfigurationInput</a> in your request. This is optional. You can also invoke
     * <a>StartDeliveryStreamEncryption</a> to turn on SSE for an existing delivery stream that doesn't have SSE
     * enabled.
     * </p>
     * <p>
     * A delivery stream is configured with a single destination, such as Amazon Simple Storage Service (Amazon S3),
     * Amazon Redshift, Amazon OpenSearch Service, Amazon OpenSearch Serverless, Splunk, and any custom HTTP endpoint or
     * HTTP endpoints owned by or supported by third-party service providers, including Datadog, Dynatrace,
     * LogicMonitor, MongoDB, New Relic, and Sumo Logic. You must specify only one of the following destination
     * configuration parameters: <code>ExtendedS3DestinationConfiguration</code>,
     * <code>S3DestinationConfiguration</code>, <code>ElasticsearchDestinationConfiguration</code>,
     * <code>RedshiftDestinationConfiguration</code>, or <code>SplunkDestinationConfiguration</code>.
     * </p>
     * <p>
     * When you specify <code>S3DestinationConfiguration</code>, you can also provide the following optional values:
     * BufferingHints, <code>EncryptionConfiguration</code>, and <code>CompressionFormat</code>. By default, if no
     * <code>BufferingHints</code> value is provided, Firehose buffers data up to 5 MB or for 5 minutes, whichever
     * condition is satisfied first. <code>BufferingHints</code> is a hint, so there are some cases where the service
     * cannot adhere to these conditions strictly. For example, record boundaries might be such that the size is a
     * little over or under the configured buffering size. By default, no encryption is performed. We strongly recommend
     * that you enable encryption to ensure secure data storage in Amazon S3.
     * </p>
     * <p>
     * A few notes about Amazon Redshift as a destination:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Redshift destination requires an S3 bucket as intermediate location. Firehose first delivers data to
     * Amazon S3 and then uses <code>COPY</code> syntax to load data into an Amazon Redshift table. This is specified in
     * the <code>RedshiftDestinationConfiguration.S3Configuration</code> parameter.
     * </p>
     * </li>
     * <li>
     * <p>
     * The compression formats <code>SNAPPY</code> or <code>ZIP</code> cannot be specified in
     * <code>RedshiftDestinationConfiguration.S3Configuration</code> because the Amazon Redshift <code>COPY</code>
     * operation that reads from the S3 bucket doesn't support these compression formats.
     * </p>
     * </li>
     * <li>
     * <p>
     * We strongly recommend that you use the user name and password you provide exclusively with Firehose, and that the
     * permissions for the account are restricted for Amazon Redshift <code>INSERT</code> permissions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Firehose assumes the IAM role that is configured as part of the destination. The role should allow the Firehose
     * principal to assume the role, and the role should have permissions that allow the service to deliver the data.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/controlling-access.html#using-iam-s3">Grant Firehose Access
     * to an Amazon S3 Destination</a> in the <i>Amazon Firehose Developer Guide</i>.
     * </p>
     *
     * @param createDeliveryStreamRequest
     * @return Result of the CreateDeliveryStream operation returned by the service.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws InvalidKmsResourceException
     *         Firehose throws this exception when an attempt to put records or to start or stop delivery stream
     *         encryption fails. This happens when the KMS service throws one of the following exception types:
     *         <code>AccessDeniedException</code>, <code>InvalidStateException</code>, <code>DisabledException</code>,
     *         or <code>NotFoundException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.CreateDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/CreateDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDeliveryStreamResponse createDeliveryStream(CreateDeliveryStreamRequest createDeliveryStreamRequest)
            throws InvalidArgumentException, LimitExceededException, ResourceInUseException, InvalidKmsResourceException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeliveryStream");

            return clientHandler.execute(new ClientExecutionParams<CreateDeliveryStreamRequest, CreateDeliveryStreamResponse>()
                    .withOperationName("CreateDeliveryStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createDeliveryStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a delivery stream and its data.
     * </p>
     * <p>
     * You can delete a delivery stream only if it is in one of the following states: <code>ACTIVE</code>,
     * <code>DELETING</code>, <code>CREATING_FAILED</code>, or <code>DELETING_FAILED</code>. You can't delete a delivery
     * stream that is in the <code>CREATING</code> state. To check the state of a delivery stream, use
     * <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * DeleteDeliveryStream is an asynchronous API. When an API request to DeleteDeliveryStream succeeds, the delivery
     * stream is marked for deletion, and it goes into the <code>DELETING</code> state.While the delivery stream is in
     * the <code>DELETING</code> state, the service might continue to accept records, but it doesn't make any guarantees
     * with respect to delivering the data. Therefore, as a best practice, first stop any applications that are sending
     * records before you delete a delivery stream.
     * </p>
     * <p>
     * Removal of a delivery stream that is in the <code>DELETING</code> state is a low priority operation for the
     * service. A stream may remain in the <code>DELETING</code> state for several minutes. Therefore, as a best
     * practice, applications should not wait for streams in the <code>DELETING</code> state to be removed.
     * </p>
     *
     * @param deleteDeliveryStreamRequest
     * @return Result of the DeleteDeliveryStream operation returned by the service.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.DeleteDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/DeleteDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDeliveryStreamResponse deleteDeliveryStream(DeleteDeliveryStreamRequest deleteDeliveryStreamRequest)
            throws ResourceInUseException, ResourceNotFoundException, AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeliveryStream");

            return clientHandler.execute(new ClientExecutionParams<DeleteDeliveryStreamRequest, DeleteDeliveryStreamResponse>()
                    .withOperationName("DeleteDeliveryStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteDeliveryStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes the specified delivery stream and its status. For example, after your delivery stream is created, call
     * <code>DescribeDeliveryStream</code> to see whether the delivery stream is <code>ACTIVE</code> and therefore ready
     * for data to be sent to it.
     * </p>
     * <p>
     * If the status of a delivery stream is <code>CREATING_FAILED</code>, this status doesn't change, and you can't
     * invoke <a>CreateDeliveryStream</a> again on it. However, you can invoke the <a>DeleteDeliveryStream</a> operation
     * to delete it. If the status is <code>DELETING_FAILED</code>, you can force deletion by invoking
     * <a>DeleteDeliveryStream</a> again but with <a>DeleteDeliveryStreamInput&#36AllowForceDelete</a> set to true.
     * </p>
     *
     * @param describeDeliveryStreamRequest
     * @return Result of the DescribeDeliveryStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.DescribeDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/DescribeDeliveryStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDeliveryStreamResponse describeDeliveryStream(DescribeDeliveryStreamRequest describeDeliveryStreamRequest)
            throws ResourceNotFoundException, AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDeliveryStream");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDeliveryStreamRequest, DescribeDeliveryStreamResponse>()
                            .withOperationName("DescribeDeliveryStream").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeDeliveryStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Invokes the GetKinesisStream operation.
     *
     * @param getKinesisStreamRequest
     * @return Result of the GetKinesisStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws InvalidStreamTypeException
     * @throws InvalidKmsResourceException
     *         Firehose throws this exception when an attempt to put records or to start or stop delivery stream
     *         encryption fails. This happens when the KMS service throws one of the following exception types:
     *         <code>AccessDeniedException</code>, <code>InvalidStateException</code>, <code>DisabledException</code>,
     *         or <code>NotFoundException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.GetKinesisStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/GetKinesisStream" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetKinesisStreamResponse getKinesisStream(GetKinesisStreamRequest getKinesisStreamRequest)
            throws ResourceNotFoundException, InvalidArgumentException, InvalidStreamTypeException, InvalidKmsResourceException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetKinesisStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetKinesisStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKinesisStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKinesisStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKinesisStream");

            return clientHandler.execute(new ClientExecutionParams<GetKinesisStreamRequest, GetKinesisStreamResponse>()
                    .withOperationName("GetKinesisStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getKinesisStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetKinesisStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists your delivery streams in alphabetical order of their names.
     * </p>
     * <p>
     * The number of delivery streams might be too large to return using a single call to
     * <code>ListDeliveryStreams</code>. You can limit the number of delivery streams returned, using the
     * <code>Limit</code> parameter. To determine whether there are more delivery streams to list, check the value of
     * <code>HasMoreDeliveryStreams</code> in the output. If there are more delivery streams to list, you can request
     * them by calling this operation again and setting the <code>ExclusiveStartDeliveryStreamName</code> parameter to
     * the name of the last delivery stream returned in the last call.
     * </p>
     *
     * @param listDeliveryStreamsRequest
     * @return Result of the ListDeliveryStreams operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.ListDeliveryStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/ListDeliveryStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListDeliveryStreamsResponse listDeliveryStreams(ListDeliveryStreamsRequest listDeliveryStreamsRequest)
            throws AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeliveryStreamsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeliveryStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDeliveryStreamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeliveryStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeliveryStreams");

            return clientHandler.execute(new ClientExecutionParams<ListDeliveryStreamsRequest, ListDeliveryStreamsResponse>()
                    .withOperationName("ListDeliveryStreams").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listDeliveryStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeliveryStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the tags for the specified delivery stream. This operation has a limit of five transactions per second per
     * account.
     * </p>
     *
     * @param listTagsForDeliveryStreamRequest
     * @return Result of the ListTagsForDeliveryStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.ListTagsForDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/ListTagsForDeliveryStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForDeliveryStreamResponse listTagsForDeliveryStream(
            ListTagsForDeliveryStreamRequest listTagsForDeliveryStreamRequest) throws ResourceNotFoundException,
            InvalidArgumentException, LimitExceededException, AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForDeliveryStream");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTagsForDeliveryStreamRequest, ListTagsForDeliveryStreamResponse>()
                            .withOperationName("ListTagsForDeliveryStream").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listTagsForDeliveryStreamRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTagsForDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Writes a single data record into an Amazon Firehose delivery stream. To write multiple data records into a
     * delivery stream, use <a>PutRecordBatch</a>. Applications using these operations are referred to as producers.
     * </p>
     * <p>
     * By default, each delivery stream can take in up to 2,000 transactions per second, 5,000 records per second, or 5
     * MB per second. If you use <a>PutRecord</a> and <a>PutRecordBatch</a>, the limits are an aggregate across these
     * two operations for each delivery stream. For more information about limits and how to request an increase, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Firehose Limits</a>.
     * </p>
     * <p>
     * Firehose accumulates and publishes a particular metric for a customer account in one minute intervals. It is
     * possible that the bursts of incoming bytes/records ingested to a delivery stream last only for a few seconds. Due
     * to this, the actual spikes in the traffic might not be fully visible in the customer's 1 minute CloudWatch
     * metrics.
     * </p>
     * <p>
     * You must specify the name of the delivery stream and the data record when using <a>PutRecord</a>. The data record
     * consists of a data blob that can be up to 1,000 KiB in size, and any kind of data. For example, it can be a
     * segment from a log file, geographic location data, website clickstream data, and so on.
     * </p>
     * <p>
     * Firehose buffers records before delivering them to the destination. To disambiguate the data blobs at the
     * destination, a common solution is to use delimiters in the data, such as a newline (<code>\n</code>) or some
     * other character unique within the data. This allows the consumer application to parse individual data items when
     * reading the data from the destination.
     * </p>
     * <p>
     * The <code>PutRecord</code> operation returns a <code>RecordId</code>, which is a unique string assigned to each
     * record. Producer applications can use this ID for purposes such as auditability and investigation.
     * </p>
     * <p>
     * If the <code>PutRecord</code> operation throws a <code>ServiceUnavailableException</code>, the API is
     * automatically reinvoked (retried) 3 times. If the exception persists, it is possible that the throughput limits
     * have been exceeded for the delivery stream.
     * </p>
     * <p>
     * Re-invoking the Put API operations (for example, PutRecord and PutRecordBatch) can result in data duplicates. For
     * larger data assets, allow for a longer time out before retrying Put API operations.
     * </p>
     * <p>
     * Data records sent to Firehose are stored for 24 hours from the time they are added to a delivery stream as it
     * tries to send the records to the destination. If the destination is unreachable for more than 24 hours, the data
     * is no longer available.
     * </p>
     * <important>
     * <p>
     * Don't concatenate two or more base64 strings to form the data fields of your records. Instead, concatenate the
     * raw data, then perform base64 encoding.
     * </p>
     * </important>
     *
     * @param putRecordRequest
     * @return Result of the PutRecord operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws InvalidKmsResourceException
     *         Firehose throws this exception when an attempt to put records or to start or stop delivery stream
     *         encryption fails. This happens when the KMS service throws one of the following exception types:
     *         <code>AccessDeniedException</code>, <code>InvalidStateException</code>, <code>DisabledException</code>,
     *         or <code>NotFoundException</code>.
     * @throws InvalidSourceException
     *         Only requests from CloudWatch Logs are supported when CloudWatch Logs decompression is enabled.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation. If you continue to see the exception,
     *         throughput limits for the delivery stream may have been exceeded. For more information about limits and
     *         how to request an increase, see <a
     *         href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Firehose Limits</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.PutRecord
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/PutRecord" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRecordResponse putRecord(PutRecordRequest putRecordRequest) throws ResourceNotFoundException,
            InvalidArgumentException, InvalidKmsResourceException, InvalidSourceException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRecordResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRecordResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecordRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecord");

            return clientHandler.execute(new ClientExecutionParams<PutRecordRequest, PutRecordResponse>()
                    .withOperationName("PutRecord").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(putRecordRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRecordRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per
     * producer than when writing single records. To write single data records into a delivery stream, use
     * <a>PutRecord</a>. Applications using these operations are referred to as producers.
     * </p>
     * <p>
     * Firehose accumulates and publishes a particular metric for a customer account in one minute intervals. It is
     * possible that the bursts of incoming bytes/records ingested to a delivery stream last only for a few seconds. Due
     * to this, the actual spikes in the traffic might not be fully visible in the customer's 1 minute CloudWatch
     * metrics.
     * </p>
     * <p>
     * For information about service quota, see <a
     * href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Firehose Quota</a>.
     * </p>
     * <p>
     * Each <a>PutRecordBatch</a> request supports up to 500 records. Each record in the request can be as large as
     * 1,000 KB (before base64 encoding), up to a limit of 4 MB for the entire request. These limits cannot be changed.
     * </p>
     * <p>
     * You must specify the name of the delivery stream and the data record when using <a>PutRecord</a>. The data record
     * consists of a data blob that can be up to 1,000 KB in size, and any kind of data. For example, it could be a
     * segment from a log file, geographic location data, website clickstream data, and so on.
     * </p>
     * <p>
     * Firehose buffers records before delivering them to the destination. To disambiguate the data blobs at the
     * destination, a common solution is to use delimiters in the data, such as a newline (<code>\n</code>) or some
     * other character unique within the data. This allows the consumer application to parse individual data items when
     * reading the data from the destination.
     * </p>
     * <p>
     * The <a>PutRecordBatch</a> response includes a count of failed records, <code>FailedPutCount</code>, and an array
     * of responses, <code>RequestResponses</code>. Even if the <a>PutRecordBatch</a> call succeeds, the value of
     * <code>FailedPutCount</code> may be greater than 0, indicating that there are records for which the operation
     * didn't succeed. Each entry in the <code>RequestResponses</code> array provides additional information about the
     * processed record. It directly correlates with a record in the request array using the same ordering, from the top
     * to the bottom. The response array always includes the same number of records as the request array.
     * <code>RequestResponses</code> includes both successfully and unsuccessfully processed records. Firehose tries to
     * process all records in each <a>PutRecordBatch</a> request. A single record failure does not stop the processing
     * of subsequent records.
     * </p>
     * <p>
     * A successfully processed record includes a <code>RecordId</code> value, which is unique for the record. An
     * unsuccessfully processed record includes <code>ErrorCode</code> and <code>ErrorMessage</code> values.
     * <code>ErrorCode</code> reflects the type of error, and is one of the following values:
     * <code>ServiceUnavailableException</code> or <code>InternalFailure</code>. <code>ErrorMessage</code> provides more
     * detailed information about the error.
     * </p>
     * <p>
     * If there is an internal server error or a timeout, the write might have completed or it might have failed. If
     * <code>FailedPutCount</code> is greater than 0, retry the request, resending only those records that might have
     * failed processing. This minimizes the possible duplicate records and also reduces the total bytes sent (and
     * corresponding charges). We recommend that you handle any duplicates at the destination.
     * </p>
     * <p>
     * If <a>PutRecordBatch</a> throws <code>ServiceUnavailableException</code>, the API is automatically reinvoked
     * (retried) 3 times. If the exception persists, it is possible that the throughput limits have been exceeded for
     * the delivery stream.
     * </p>
     * <p>
     * Re-invoking the Put API operations (for example, PutRecord and PutRecordBatch) can result in data duplicates. For
     * larger data assets, allow for a longer time out before retrying Put API operations.
     * </p>
     * <p>
     * Data records sent to Firehose are stored for 24 hours from the time they are added to a delivery stream as it
     * attempts to send the records to the destination. If the destination is unreachable for more than 24 hours, the
     * data is no longer available.
     * </p>
     * <important>
     * <p>
     * Don't concatenate two or more base64 strings to form the data fields of your records. Instead, concatenate the
     * raw data, then perform base64 encoding.
     * </p>
     * </important>
     *
     * @param putRecordBatchRequest
     * @return Result of the PutRecordBatch operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws InvalidKmsResourceException
     *         Firehose throws this exception when an attempt to put records or to start or stop delivery stream
     *         encryption fails. This happens when the KMS service throws one of the following exception types:
     *         <code>AccessDeniedException</code>, <code>InvalidStateException</code>, <code>DisabledException</code>,
     *         or <code>NotFoundException</code>.
     * @throws InvalidSourceException
     *         Only requests from CloudWatch Logs are supported when CloudWatch Logs decompression is enabled.
     * @throws ServiceUnavailableException
     *         The service is unavailable. Back off and retry the operation. If you continue to see the exception,
     *         throughput limits for the delivery stream may have been exceeded. For more information about limits and
     *         how to request an increase, see <a
     *         href="https://docs.aws.amazon.com/firehose/latest/dev/limits.html">Amazon Firehose Limits</a>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.PutRecordBatch
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/PutRecordBatch" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutRecordBatchResponse putRecordBatch(PutRecordBatchRequest putRecordBatchRequest) throws ResourceNotFoundException,
            InvalidArgumentException, InvalidKmsResourceException, InvalidSourceException, ServiceUnavailableException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRecordBatchResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutRecordBatchResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecordBatchRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecordBatchRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecordBatch");

            return clientHandler.execute(new ClientExecutionParams<PutRecordBatchRequest, PutRecordBatchResponse>()
                    .withOperationName("PutRecordBatch").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putRecordBatchRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRecordBatchRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables server-side encryption (SSE) for the delivery stream.
     * </p>
     * <p>
     * This operation is asynchronous. It returns immediately. When you invoke it, Firehose first sets the encryption
     * status of the stream to <code>ENABLING</code>, and then to <code>ENABLED</code>. The encryption status of a
     * delivery stream is the <code>Status</code> property in <a>DeliveryStreamEncryptionConfiguration</a>. If the
     * operation fails, the encryption status changes to <code>ENABLING_FAILED</code>. You can continue to read and
     * write data to your delivery stream while the encryption status is <code>ENABLING</code>, but the data is not
     * encrypted. It can take up to 5 seconds after the encryption status changes to <code>ENABLED</code> before all
     * records written to the delivery stream are encrypted. To find out whether a record or a batch of records was
     * encrypted, check the response elements <a>PutRecordOutput&#36Encrypted</a> and
     * <a>PutRecordBatchOutput&#36Encrypted</a>, respectively.
     * </p>
     * <p>
     * To check the encryption status of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * Even if encryption is currently enabled for a delivery stream, you can still invoke this operation on it to
     * change the ARN of the CMK or both its type and ARN. If you invoke this method to change the CMK, and the old CMK
     * is of type <code>CUSTOMER_MANAGED_CMK</code>, Firehose schedules the grant it had on the old CMK for retirement.
     * If the new CMK is of type <code>CUSTOMER_MANAGED_CMK</code>, Firehose creates a grant that enables it to use the
     * new CMK to encrypt and decrypt data and to manage the grant.
     * </p>
     * <p>
     * For the KMS grant creation to be successful, Firehose APIs <code>StartDeliveryStreamEncryption</code> and
     * <code>CreateDeliveryStream</code> should not be called with session credentials that are more than 6 hours old.
     * </p>
     * <p>
     * If a delivery stream already has encryption enabled and then you invoke this operation to change the ARN of the
     * CMK or both its type and ARN and you get <code>ENABLING_FAILED</code>, this only means that the attempt to change
     * the CMK failed. In this case, encryption remains enabled with the old CMK.
     * </p>
     * <p>
     * If the encryption status of your delivery stream is <code>ENABLING_FAILED</code>, you can invoke this operation
     * again with a valid CMK. The CMK must be enabled and the key policy mustn't explicitly deny the permission for
     * Firehose to invoke KMS encrypt and decrypt operations.
     * </p>
     * <p>
     * You can enable SSE for a delivery stream only if it's a delivery stream that uses <code>DirectPut</code> as its
     * source.
     * </p>
     * <p>
     * The <code>StartDeliveryStreamEncryption</code> and <code>StopDeliveryStreamEncryption</code> operations have a
     * combined limit of 25 calls per delivery stream per 24 hours. For example, you reach the limit if you call
     * <code>StartDeliveryStreamEncryption</code> 13 times and <code>StopDeliveryStreamEncryption</code> 12 times for
     * the same delivery stream in a 24-hour period.
     * </p>
     *
     * @param startDeliveryStreamEncryptionRequest
     * @return Result of the StartDeliveryStreamEncryption operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws InvalidKmsResourceException
     *         Firehose throws this exception when an attempt to put records or to start or stop delivery stream
     *         encryption fails. This happens when the KMS service throws one of the following exception types:
     *         <code>AccessDeniedException</code>, <code>InvalidStateException</code>, <code>DisabledException</code>,
     *         or <code>NotFoundException</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.StartDeliveryStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/StartDeliveryStreamEncryption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartDeliveryStreamEncryptionResponse startDeliveryStreamEncryption(
            StartDeliveryStreamEncryptionRequest startDeliveryStreamEncryptionRequest) throws ResourceNotFoundException,
            ResourceInUseException, InvalidArgumentException, LimitExceededException, InvalidKmsResourceException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeliveryStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartDeliveryStreamEncryptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startDeliveryStreamEncryptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startDeliveryStreamEncryptionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeliveryStreamEncryption");

            return clientHandler
                    .execute(new ClientExecutionParams<StartDeliveryStreamEncryptionRequest, StartDeliveryStreamEncryptionResponse>()
                            .withOperationName("StartDeliveryStreamEncryption").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startDeliveryStreamEncryptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartDeliveryStreamEncryptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables server-side encryption (SSE) for the delivery stream.
     * </p>
     * <p>
     * This operation is asynchronous. It returns immediately. When you invoke it, Firehose first sets the encryption
     * status of the stream to <code>DISABLING</code>, and then to <code>DISABLED</code>. You can continue to read and
     * write data to your stream while its status is <code>DISABLING</code>. It can take up to 5 seconds after the
     * encryption status changes to <code>DISABLED</code> before all records written to the delivery stream are no
     * longer subject to encryption. To find out whether a record or a batch of records was encrypted, check the
     * response elements <a>PutRecordOutput&#36Encrypted</a> and <a>PutRecordBatchOutput&#36Encrypted</a>, respectively.
     * </p>
     * <p>
     * To check the encryption state of a delivery stream, use <a>DescribeDeliveryStream</a>.
     * </p>
     * <p>
     * If SSE is enabled using a customer managed CMK and then you invoke <code>StopDeliveryStreamEncryption</code>,
     * Firehose schedules the related KMS grant for retirement and then retires it after it ensures that it is finished
     * delivering records to the destination.
     * </p>
     * <p>
     * The <code>StartDeliveryStreamEncryption</code> and <code>StopDeliveryStreamEncryption</code> operations have a
     * combined limit of 25 calls per delivery stream per 24 hours. For example, you reach the limit if you call
     * <code>StartDeliveryStreamEncryption</code> 13 times and <code>StopDeliveryStreamEncryption</code> 12 times for
     * the same delivery stream in a 24-hour period.
     * </p>
     *
     * @param stopDeliveryStreamEncryptionRequest
     * @return Result of the StopDeliveryStreamEncryption operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.StopDeliveryStreamEncryption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/StopDeliveryStreamEncryption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopDeliveryStreamEncryptionResponse stopDeliveryStreamEncryption(
            StopDeliveryStreamEncryptionRequest stopDeliveryStreamEncryptionRequest) throws ResourceNotFoundException,
            ResourceInUseException, InvalidArgumentException, LimitExceededException, AwsServiceException, SdkClientException,
            FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopDeliveryStreamEncryptionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopDeliveryStreamEncryptionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopDeliveryStreamEncryptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDeliveryStreamEncryptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDeliveryStreamEncryption");

            return clientHandler
                    .execute(new ClientExecutionParams<StopDeliveryStreamEncryptionRequest, StopDeliveryStreamEncryptionResponse>()
                            .withOperationName("StopDeliveryStreamEncryption").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(stopDeliveryStreamEncryptionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopDeliveryStreamEncryptionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or updates tags for the specified delivery stream. A tag is a key-value pair that you can define and assign
     * to Amazon Web Services resources. If you specify a tag that already exists, the tag value is replaced with the
     * value that you specify in the request. Tags are metadata. For example, you can add friendly names and
     * descriptions or other types of information that can help you distinguish the delivery stream. For more
     * information about tags, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/cost-alloc-tags.html">Using Cost Allocation
     * Tags</a> in the <i>Amazon Web Services Billing and Cost Management User Guide</i>.
     * </p>
     * <p>
     * Each delivery stream can have up to 50 tags.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per account.
     * </p>
     *
     * @param tagDeliveryStreamRequest
     * @return Result of the TagDeliveryStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.TagDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/TagDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TagDeliveryStreamResponse tagDeliveryStream(TagDeliveryStreamRequest tagDeliveryStreamRequest)
            throws ResourceNotFoundException, ResourceInUseException, InvalidArgumentException, LimitExceededException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagDeliveryStream");

            return clientHandler.execute(new ClientExecutionParams<TagDeliveryStreamRequest, TagDeliveryStreamResponse>()
                    .withOperationName("TagDeliveryStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(tagDeliveryStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes tags from the specified delivery stream. Removed tags are deleted, and you can't recover them after this
     * operation successfully completes.
     * </p>
     * <p>
     * If you specify a tag that doesn't exist, the operation ignores it.
     * </p>
     * <p>
     * This operation has a limit of five transactions per second per account.
     * </p>
     *
     * @param untagDeliveryStreamRequest
     * @return Result of the UntagDeliveryStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws LimitExceededException
     *         You have already reached the limit for a requested resource.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.UntagDeliveryStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/UntagDeliveryStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UntagDeliveryStreamResponse untagDeliveryStream(UntagDeliveryStreamRequest untagDeliveryStreamRequest)
            throws ResourceNotFoundException, ResourceInUseException, InvalidArgumentException, LimitExceededException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagDeliveryStreamResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UntagDeliveryStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagDeliveryStreamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagDeliveryStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagDeliveryStream");

            return clientHandler.execute(new ClientExecutionParams<UntagDeliveryStreamRequest, UntagDeliveryStreamResponse>()
                    .withOperationName("UntagDeliveryStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagDeliveryStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagDeliveryStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified destination of the specified delivery stream.
     * </p>
     * <p>
     * Use this operation to change the destination type (for example, to replace the Amazon S3 destination with Amazon
     * Redshift) or change the parameters associated with a destination (for example, to change the bucket name of the
     * Amazon S3 destination). The update might not occur immediately. The target delivery stream remains active while
     * the configurations are updated, so data writes to the delivery stream can continue during this process. The
     * updated configurations are usually effective within a few minutes.
     * </p>
     * <p>
     * Switching between Amazon OpenSearch Service and other services is not supported. For an Amazon OpenSearch Service
     * destination, you can only update to another Amazon OpenSearch Service destination.
     * </p>
     * <p>
     * If the destination type is the same, Firehose merges the configuration parameters specified with the destination
     * configuration that already exists on the delivery stream. If any of the parameters are not specified in the call,
     * the existing values are retained. For example, in the Amazon S3 destination, if <a>EncryptionConfiguration</a> is
     * not specified, then the existing <code>EncryptionConfiguration</code> is maintained on the destination.
     * </p>
     * <p>
     * If the destination type is not the same, for example, changing the destination from Amazon S3 to Amazon Redshift,
     * Firehose does not merge any parameters. In this case, all parameters must be specified.
     * </p>
     * <p>
     * Firehose uses <code>CurrentDeliveryStreamVersionId</code> to avoid race conditions and conflicting merges. This
     * is a required field, and the service updates the configuration only if the existing configuration has a version
     * ID that matches. After the update is applied successfully, the version ID is updated, and can be retrieved using
     * <a>DescribeDeliveryStream</a>. Use the new version ID to set <code>CurrentDeliveryStreamVersionId</code> in the
     * next call.
     * </p>
     *
     * @param updateDestinationRequest
     * @return Result of the UpdateDestination operation returned by the service.
     * @throws InvalidArgumentException
     *         The specified input parameter has a value that is not valid.
     * @throws ResourceInUseException
     *         The resource is already in use and not available for this operation.
     * @throws ResourceNotFoundException
     *         The specified resource could not be found.
     * @throws ConcurrentModificationException
     *         Another modification has already happened. Fetch <code>VersionId</code> again and use it to update the
     *         destination.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.UpdateDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/UpdateDestination" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateDestinationResponse updateDestination(UpdateDestinationRequest updateDestinationRequest)
            throws InvalidArgumentException, ResourceInUseException, ResourceNotFoundException, ConcurrentModificationException,
            AwsServiceException, SdkClientException, FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDestinationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDestinationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDestination");

            return clientHandler.execute(new ClientExecutionParams<UpdateDestinationRequest, UpdateDestinationResponse>()
                    .withOperationName("UpdateDestination").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateDestinationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Invokes the VerifyResourcesExistForTagris operation.
     *
     * @param verifyResourcesExistForTagrisRequest
     * @return Result of the VerifyResourcesExistForTagris operation returned by the service.
     * @throws TagrisAccessDeniedException
     * @throws TagrisInternalServiceException
     * @throws TagrisInvalidArnException
     * @throws TagrisInvalidParameterException
     * @throws TagrisPartialResourcesExistResultsException
     * @throws TagrisThrottledException
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FirehoseException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FirehoseClient.VerifyResourcesExistForTagris
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/firehose-2015-08-04/VerifyResourcesExistForTagris"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public VerifyResourcesExistForTagrisResponse verifyResourcesExistForTagris(
            VerifyResourcesExistForTagrisRequest verifyResourcesExistForTagrisRequest) throws TagrisAccessDeniedException,
            TagrisInternalServiceException, TagrisInvalidArnException, TagrisInvalidParameterException,
            TagrisPartialResourcesExistResultsException, TagrisThrottledException, AwsServiceException, SdkClientException,
            FirehoseException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<VerifyResourcesExistForTagrisResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, VerifyResourcesExistForTagrisResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(verifyResourcesExistForTagrisRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                verifyResourcesExistForTagrisRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Firehose");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "VerifyResourcesExistForTagris");

            return clientHandler
                    .execute(new ClientExecutionParams<VerifyResourcesExistForTagrisRequest, VerifyResourcesExistForTagrisResponse>()
                            .withOperationName("VerifyResourcesExistForTagris").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(verifyResourcesExistForTagrisRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new VerifyResourcesExistForTagrisRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        FirehoseServiceClientConfigurationBuilder serviceConfigBuilder = new FirehoseServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FirehoseException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisInvalidParameterException")
                                .exceptionBuilderSupplier(TagrisInvalidParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidSourceException")
                                .exceptionBuilderSupplier(InvalidSourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidKMSResourceException")
                                .exceptionBuilderSupplier(InvalidKmsResourceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStreamTypeException")
                                .exceptionBuilderSupplier(InvalidStreamTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisAccessDeniedException")
                                .exceptionBuilderSupplier(TagrisAccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisThrottledException")
                                .exceptionBuilderSupplier(TagrisThrottledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisInvalidArnException")
                                .exceptionBuilderSupplier(TagrisInvalidArnException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisPartialResourcesExistResultsException")
                                .exceptionBuilderSupplier(TagrisPartialResourcesExistResultsException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgumentException")
                                .exceptionBuilderSupplier(InvalidArgumentException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TagrisInternalServiceException")
                                .exceptionBuilderSupplier(TagrisInternalServiceException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build());
    }

    @Override
    public final FirehoseServiceClientConfiguration serviceClientConfiguration() {
        return new FirehoseServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
