/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.docdb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Detailed information about a cluster snapshot.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DBClusterSnapshot implements SdkPojo, Serializable,
        ToCopyableBuilder<DBClusterSnapshot.Builder, DBClusterSnapshot> {
    private static final SdkField<List<String>> AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AvailabilityZones")
            .getter(getter(DBClusterSnapshot::availabilityZones))
            .setter(setter(Builder::availabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final SdkField<String> DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DBClusterSnapshotIdentifier")
            .getter(getter(DBClusterSnapshot::dbClusterSnapshotIdentifier))
            .setter(setter(Builder::dbClusterSnapshotIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterSnapshotIdentifier")
                    .build()).build();

    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterIdentifier").getter(getter(DBClusterSnapshot::dbClusterIdentifier))
            .setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<Instant> SNAPSHOT_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SnapshotCreateTime").getter(getter(DBClusterSnapshot::snapshotCreateTime))
            .setter(setter(Builder::snapshotCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotCreateTime").build())
            .build();

    private static final SdkField<String> ENGINE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Engine")
            .getter(getter(DBClusterSnapshot::engine)).setter(setter(Builder::engine))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Engine").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DBClusterSnapshot::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(DBClusterSnapshot::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(DBClusterSnapshot::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId").build()).build();

    private static final SdkField<Instant> CLUSTER_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ClusterCreateTime").getter(getter(DBClusterSnapshot::clusterCreateTime))
            .setter(setter(Builder::clusterCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterCreateTime").build()).build();

    private static final SdkField<String> MASTER_USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterUsername").getter(getter(DBClusterSnapshot::masterUsername))
            .setter(setter(Builder::masterUsername))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUsername").build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(DBClusterSnapshot::engineVersion)).setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<String> SNAPSHOT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SnapshotType").getter(getter(DBClusterSnapshot::snapshotType)).setter(setter(Builder::snapshotType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotType").build()).build();

    private static final SdkField<Integer> PERCENT_PROGRESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("PercentProgress").getter(getter(DBClusterSnapshot::percentProgress))
            .setter(setter(Builder::percentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentProgress").build()).build();

    private static final SdkField<Boolean> STORAGE_ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("StorageEncrypted").getter(getter(DBClusterSnapshot::storageEncrypted))
            .setter(setter(Builder::storageEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageEncrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(DBClusterSnapshot::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> DB_CLUSTER_SNAPSHOT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterSnapshotArn").getter(getter(DBClusterSnapshot::dbClusterSnapshotArn))
            .setter(setter(Builder::dbClusterSnapshotArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterSnapshotArn").build())
            .build();

    private static final SdkField<String> SOURCE_DB_CLUSTER_SNAPSHOT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceDBClusterSnapshotArn")
            .getter(getter(DBClusterSnapshot::sourceDBClusterSnapshotArn))
            .setter(setter(Builder::sourceDBClusterSnapshotArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceDBClusterSnapshotArn").build())
            .build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(DBClusterSnapshot::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVAILABILITY_ZONES_FIELD,
            DB_CLUSTER_SNAPSHOT_IDENTIFIER_FIELD, DB_CLUSTER_IDENTIFIER_FIELD, SNAPSHOT_CREATE_TIME_FIELD, ENGINE_FIELD,
            STATUS_FIELD, PORT_FIELD, VPC_ID_FIELD, CLUSTER_CREATE_TIME_FIELD, MASTER_USERNAME_FIELD, ENGINE_VERSION_FIELD,
            SNAPSHOT_TYPE_FIELD, PERCENT_PROGRESS_FIELD, STORAGE_ENCRYPTED_FIELD, KMS_KEY_ID_FIELD,
            DB_CLUSTER_SNAPSHOT_ARN_FIELD, SOURCE_DB_CLUSTER_SNAPSHOT_ARN_FIELD, STORAGE_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> availabilityZones;

    private final String dbClusterSnapshotIdentifier;

    private final String dbClusterIdentifier;

    private final Instant snapshotCreateTime;

    private final String engine;

    private final String status;

    private final Integer port;

    private final String vpcId;

    private final Instant clusterCreateTime;

    private final String masterUsername;

    private final String engineVersion;

    private final String snapshotType;

    private final Integer percentProgress;

    private final Boolean storageEncrypted;

    private final String kmsKeyId;

    private final String dbClusterSnapshotArn;

    private final String sourceDBClusterSnapshotArn;

    private final String storageType;

    private DBClusterSnapshot(BuilderImpl builder) {
        this.availabilityZones = builder.availabilityZones;
        this.dbClusterSnapshotIdentifier = builder.dbClusterSnapshotIdentifier;
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.snapshotCreateTime = builder.snapshotCreateTime;
        this.engine = builder.engine;
        this.status = builder.status;
        this.port = builder.port;
        this.vpcId = builder.vpcId;
        this.clusterCreateTime = builder.clusterCreateTime;
        this.masterUsername = builder.masterUsername;
        this.engineVersion = builder.engineVersion;
        this.snapshotType = builder.snapshotType;
        this.percentProgress = builder.percentProgress;
        this.storageEncrypted = builder.storageEncrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.dbClusterSnapshotArn = builder.dbClusterSnapshotArn;
        this.sourceDBClusterSnapshotArn = builder.sourceDBClusterSnapshotArn;
        this.storageType = builder.storageType;
    }

    /**
     * For responses, this returns true if the service returned a value for the AvailabilityZones property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAvailabilityZones() {
        return availabilityZones != null && !(availabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be restored in.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAvailabilityZones} method.
     * </p>
     * 
     * @return Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be restored
     *         in.
     */
    public final List<String> availabilityZones() {
        return availabilityZones;
    }

    /**
     * <p>
     * Specifies the identifier for the cluster snapshot.
     * </p>
     * 
     * @return Specifies the identifier for the cluster snapshot.
     */
    public final String dbClusterSnapshotIdentifier() {
        return dbClusterSnapshotIdentifier;
    }

    /**
     * <p>
     * Specifies the cluster identifier of the cluster that this cluster snapshot was created from.
     * </p>
     * 
     * @return Specifies the cluster identifier of the cluster that this cluster snapshot was created from.
     */
    public final String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * <p>
     * Provides the time when the snapshot was taken, in UTC.
     * </p>
     * 
     * @return Provides the time when the snapshot was taken, in UTC.
     */
    public final Instant snapshotCreateTime() {
        return snapshotCreateTime;
    }

    /**
     * <p>
     * Specifies the name of the database engine.
     * </p>
     * 
     * @return Specifies the name of the database engine.
     */
    public final String engine() {
        return engine;
    }

    /**
     * <p>
     * Specifies the status of this cluster snapshot.
     * </p>
     * 
     * @return Specifies the status of this cluster snapshot.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * Specifies the port that the cluster was listening on at the time of the snapshot.
     * </p>
     * 
     * @return Specifies the port that the cluster was listening on at the time of the snapshot.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * Provides the virtual private cloud (VPC) ID that is associated with the cluster snapshot.
     * </p>
     * 
     * @return Provides the virtual private cloud (VPC) ID that is associated with the cluster snapshot.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * Specifies the time when the cluster was created, in Universal Coordinated Time (UTC).
     * </p>
     * 
     * @return Specifies the time when the cluster was created, in Universal Coordinated Time (UTC).
     */
    public final Instant clusterCreateTime() {
        return clusterCreateTime;
    }

    /**
     * <p>
     * Provides the master user name for the cluster snapshot.
     * </p>
     * 
     * @return Provides the master user name for the cluster snapshot.
     */
    public final String masterUsername() {
        return masterUsername;
    }

    /**
     * <p>
     * Provides the version of the database engine for this cluster snapshot.
     * </p>
     * 
     * @return Provides the version of the database engine for this cluster snapshot.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * Provides the type of the cluster snapshot.
     * </p>
     * 
     * @return Provides the type of the cluster snapshot.
     */
    public final String snapshotType() {
        return snapshotType;
    }

    /**
     * <p>
     * Specifies the percentage of the estimated data that has been transferred.
     * </p>
     * 
     * @return Specifies the percentage of the estimated data that has been transferred.
     */
    public final Integer percentProgress() {
        return percentProgress;
    }

    /**
     * <p>
     * Specifies whether the cluster snapshot is encrypted.
     * </p>
     * 
     * @return Specifies whether the cluster snapshot is encrypted.
     */
    public final Boolean storageEncrypted() {
        return storageEncrypted;
    }

    /**
     * <p>
     * If <code>StorageEncrypted</code> is <code>true</code>, the KMS key identifier for the encrypted cluster snapshot.
     * </p>
     * 
     * @return If <code>StorageEncrypted</code> is <code>true</code>, the KMS key identifier for the encrypted cluster
     *         snapshot.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the cluster snapshot.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the cluster snapshot.
     */
    public final String dbClusterSnapshotArn() {
        return dbClusterSnapshotArn;
    }

    /**
     * <p>
     * If the cluster snapshot was copied from a source cluster snapshot, the ARN for the source cluster snapshot;
     * otherwise, a null value.
     * </p>
     * 
     * @return If the cluster snapshot was copied from a source cluster snapshot, the ARN for the source cluster
     *         snapshot; otherwise, a null value.
     */
    public final String sourceDBClusterSnapshotArn() {
        return sourceDBClusterSnapshotArn;
    }

    /**
     * <p>
     * Storage type associated with your cluster snapshot
     * </p>
     * <p>
     * For information on storage types for Amazon DocumentDB clusters, see Cluster storage configurations in the
     * <i>Amazon DocumentDB Developer Guide</i>.
     * </p>
     * <p>
     * Valid values for storage type - <code>standard | iopt1</code>
     * </p>
     * <p>
     * Default value is <code>standard </code>
     * </p>
     * 
     * @return Storage type associated with your cluster snapshot </p>
     *         <p>
     *         For information on storage types for Amazon DocumentDB clusters, see Cluster storage configurations in
     *         the <i>Amazon DocumentDB Developer Guide</i>.
     *         </p>
     *         <p>
     *         Valid values for storage type - <code>standard | iopt1</code>
     *         </p>
     *         <p>
     *         Default value is <code>standard </code>
     */
    public final String storageType() {
        return storageType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAvailabilityZones() ? availabilityZones() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterSnapshotIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(engine());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(clusterCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(masterUsername());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotType());
        hashCode = 31 * hashCode + Objects.hashCode(percentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(storageEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterSnapshotArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceDBClusterSnapshotArn());
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DBClusterSnapshot)) {
            return false;
        }
        DBClusterSnapshot other = (DBClusterSnapshot) obj;
        return hasAvailabilityZones() == other.hasAvailabilityZones()
                && Objects.equals(availabilityZones(), other.availabilityZones())
                && Objects.equals(dbClusterSnapshotIdentifier(), other.dbClusterSnapshotIdentifier())
                && Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(snapshotCreateTime(), other.snapshotCreateTime()) && Objects.equals(engine(), other.engine())
                && Objects.equals(status(), other.status()) && Objects.equals(port(), other.port())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(clusterCreateTime(), other.clusterCreateTime())
                && Objects.equals(masterUsername(), other.masterUsername())
                && Objects.equals(engineVersion(), other.engineVersion()) && Objects.equals(snapshotType(), other.snapshotType())
                && Objects.equals(percentProgress(), other.percentProgress())
                && Objects.equals(storageEncrypted(), other.storageEncrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(dbClusterSnapshotArn(), other.dbClusterSnapshotArn())
                && Objects.equals(sourceDBClusterSnapshotArn(), other.sourceDBClusterSnapshotArn())
                && Objects.equals(storageType(), other.storageType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DBClusterSnapshot")
                .add("AvailabilityZones", hasAvailabilityZones() ? availabilityZones() : null)
                .add("DBClusterSnapshotIdentifier", dbClusterSnapshotIdentifier())
                .add("DBClusterIdentifier", dbClusterIdentifier()).add("SnapshotCreateTime", snapshotCreateTime())
                .add("Engine", engine()).add("Status", status()).add("Port", port()).add("VpcId", vpcId())
                .add("ClusterCreateTime", clusterCreateTime()).add("MasterUsername", masterUsername())
                .add("EngineVersion", engineVersion()).add("SnapshotType", snapshotType())
                .add("PercentProgress", percentProgress()).add("StorageEncrypted", storageEncrypted())
                .add("KmsKeyId", kmsKeyId()).add("DBClusterSnapshotArn", dbClusterSnapshotArn())
                .add("SourceDBClusterSnapshotArn", sourceDBClusterSnapshotArn()).add("StorageType", storageType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityZones":
            return Optional.ofNullable(clazz.cast(availabilityZones()));
        case "DBClusterSnapshotIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterSnapshotIdentifier()));
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "SnapshotCreateTime":
            return Optional.ofNullable(clazz.cast(snapshotCreateTime()));
        case "Engine":
            return Optional.ofNullable(clazz.cast(engine()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "ClusterCreateTime":
            return Optional.ofNullable(clazz.cast(clusterCreateTime()));
        case "MasterUsername":
            return Optional.ofNullable(clazz.cast(masterUsername()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "SnapshotType":
            return Optional.ofNullable(clazz.cast(snapshotType()));
        case "PercentProgress":
            return Optional.ofNullable(clazz.cast(percentProgress()));
        case "StorageEncrypted":
            return Optional.ofNullable(clazz.cast(storageEncrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "DBClusterSnapshotArn":
            return Optional.ofNullable(clazz.cast(dbClusterSnapshotArn()));
        case "SourceDBClusterSnapshotArn":
            return Optional.ofNullable(clazz.cast(sourceDBClusterSnapshotArn()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DBClusterSnapshot, T> g) {
        return obj -> g.apply((DBClusterSnapshot) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DBClusterSnapshot> {
        /**
         * <p>
         * Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be restored in.
         * </p>
         * 
         * @param availabilityZones
         *        Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be
         *        restored in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(Collection<String> availabilityZones);

        /**
         * <p>
         * Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be restored in.
         * </p>
         * 
         * @param availabilityZones
         *        Provides the list of Amazon EC2 Availability Zones that instances in the cluster snapshot can be
         *        restored in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZones(String... availabilityZones);

        /**
         * <p>
         * Specifies the identifier for the cluster snapshot.
         * </p>
         * 
         * @param dbClusterSnapshotIdentifier
         *        Specifies the identifier for the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier);

        /**
         * <p>
         * Specifies the cluster identifier of the cluster that this cluster snapshot was created from.
         * </p>
         * 
         * @param dbClusterIdentifier
         *        Specifies the cluster identifier of the cluster that this cluster snapshot was created from.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * <p>
         * Provides the time when the snapshot was taken, in UTC.
         * </p>
         * 
         * @param snapshotCreateTime
         *        Provides the time when the snapshot was taken, in UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotCreateTime(Instant snapshotCreateTime);

        /**
         * <p>
         * Specifies the name of the database engine.
         * </p>
         * 
         * @param engine
         *        Specifies the name of the database engine.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engine(String engine);

        /**
         * <p>
         * Specifies the status of this cluster snapshot.
         * </p>
         * 
         * @param status
         *        Specifies the status of this cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Specifies the port that the cluster was listening on at the time of the snapshot.
         * </p>
         * 
         * @param port
         *        Specifies the port that the cluster was listening on at the time of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Provides the virtual private cloud (VPC) ID that is associated with the cluster snapshot.
         * </p>
         * 
         * @param vpcId
         *        Provides the virtual private cloud (VPC) ID that is associated with the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * Specifies the time when the cluster was created, in Universal Coordinated Time (UTC).
         * </p>
         * 
         * @param clusterCreateTime
         *        Specifies the time when the cluster was created, in Universal Coordinated Time (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterCreateTime(Instant clusterCreateTime);

        /**
         * <p>
         * Provides the master user name for the cluster snapshot.
         * </p>
         * 
         * @param masterUsername
         *        Provides the master user name for the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUsername(String masterUsername);

        /**
         * <p>
         * Provides the version of the database engine for this cluster snapshot.
         * </p>
         * 
         * @param engineVersion
         *        Provides the version of the database engine for this cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * Provides the type of the cluster snapshot.
         * </p>
         * 
         * @param snapshotType
         *        Provides the type of the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotType(String snapshotType);

        /**
         * <p>
         * Specifies the percentage of the estimated data that has been transferred.
         * </p>
         * 
         * @param percentProgress
         *        Specifies the percentage of the estimated data that has been transferred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentProgress(Integer percentProgress);

        /**
         * <p>
         * Specifies whether the cluster snapshot is encrypted.
         * </p>
         * 
         * @param storageEncrypted
         *        Specifies whether the cluster snapshot is encrypted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageEncrypted(Boolean storageEncrypted);

        /**
         * <p>
         * If <code>StorageEncrypted</code> is <code>true</code>, the KMS key identifier for the encrypted cluster
         * snapshot.
         * </p>
         * 
         * @param kmsKeyId
         *        If <code>StorageEncrypted</code> is <code>true</code>, the KMS key identifier for the encrypted
         *        cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the cluster snapshot.
         * </p>
         * 
         * @param dbClusterSnapshotArn
         *        The Amazon Resource Name (ARN) for the cluster snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterSnapshotArn(String dbClusterSnapshotArn);

        /**
         * <p>
         * If the cluster snapshot was copied from a source cluster snapshot, the ARN for the source cluster snapshot;
         * otherwise, a null value.
         * </p>
         * 
         * @param sourceDBClusterSnapshotArn
         *        If the cluster snapshot was copied from a source cluster snapshot, the ARN for the source cluster
         *        snapshot; otherwise, a null value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn);

        /**
         * <p>
         * Storage type associated with your cluster snapshot
         * </p>
         * <p>
         * For information on storage types for Amazon DocumentDB clusters, see Cluster storage configurations in the
         * <i>Amazon DocumentDB Developer Guide</i>.
         * </p>
         * <p>
         * Valid values for storage type - <code>standard | iopt1</code>
         * </p>
         * <p>
         * Default value is <code>standard </code>
         * </p>
         * 
         * @param storageType
         *        Storage type associated with your cluster snapshot </p>
         *        <p>
         *        For information on storage types for Amazon DocumentDB clusters, see Cluster storage configurations in
         *        the <i>Amazon DocumentDB Developer Guide</i>.
         *        </p>
         *        <p>
         *        Valid values for storage type - <code>standard | iopt1</code>
         *        </p>
         *        <p>
         *        Default value is <code>standard </code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);
    }

    static final class BuilderImpl implements Builder {
        private List<String> availabilityZones = DefaultSdkAutoConstructList.getInstance();

        private String dbClusterSnapshotIdentifier;

        private String dbClusterIdentifier;

        private Instant snapshotCreateTime;

        private String engine;

        private String status;

        private Integer port;

        private String vpcId;

        private Instant clusterCreateTime;

        private String masterUsername;

        private String engineVersion;

        private String snapshotType;

        private Integer percentProgress;

        private Boolean storageEncrypted;

        private String kmsKeyId;

        private String dbClusterSnapshotArn;

        private String sourceDBClusterSnapshotArn;

        private String storageType;

        private BuilderImpl() {
        }

        private BuilderImpl(DBClusterSnapshot model) {
            availabilityZones(model.availabilityZones);
            dbClusterSnapshotIdentifier(model.dbClusterSnapshotIdentifier);
            dbClusterIdentifier(model.dbClusterIdentifier);
            snapshotCreateTime(model.snapshotCreateTime);
            engine(model.engine);
            status(model.status);
            port(model.port);
            vpcId(model.vpcId);
            clusterCreateTime(model.clusterCreateTime);
            masterUsername(model.masterUsername);
            engineVersion(model.engineVersion);
            snapshotType(model.snapshotType);
            percentProgress(model.percentProgress);
            storageEncrypted(model.storageEncrypted);
            kmsKeyId(model.kmsKeyId);
            dbClusterSnapshotArn(model.dbClusterSnapshotArn);
            sourceDBClusterSnapshotArn(model.sourceDBClusterSnapshotArn);
            storageType(model.storageType);
        }

        public final Collection<String> getAvailabilityZones() {
            if (availabilityZones instanceof SdkAutoConstructList) {
                return null;
            }
            return availabilityZones;
        }

        public final void setAvailabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
        }

        @Override
        public final Builder availabilityZones(Collection<String> availabilityZones) {
            this.availabilityZones = AvailabilityZonesCopier.copy(availabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder availabilityZones(String... availabilityZones) {
            availabilityZones(Arrays.asList(availabilityZones));
            return this;
        }

        public final String getDbClusterSnapshotIdentifier() {
            return dbClusterSnapshotIdentifier;
        }

        public final void setDbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier) {
            this.dbClusterSnapshotIdentifier = dbClusterSnapshotIdentifier;
        }

        @Override
        public final Builder dbClusterSnapshotIdentifier(String dbClusterSnapshotIdentifier) {
            this.dbClusterSnapshotIdentifier = dbClusterSnapshotIdentifier;
            return this;
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        @Override
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final Instant getSnapshotCreateTime() {
            return snapshotCreateTime;
        }

        public final void setSnapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
        }

        @Override
        public final Builder snapshotCreateTime(Instant snapshotCreateTime) {
            this.snapshotCreateTime = snapshotCreateTime;
            return this;
        }

        public final String getEngine() {
            return engine;
        }

        public final void setEngine(String engine) {
            this.engine = engine;
        }

        @Override
        public final Builder engine(String engine) {
            this.engine = engine;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Instant getClusterCreateTime() {
            return clusterCreateTime;
        }

        public final void setClusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
        }

        @Override
        public final Builder clusterCreateTime(Instant clusterCreateTime) {
            this.clusterCreateTime = clusterCreateTime;
            return this;
        }

        public final String getMasterUsername() {
            return masterUsername;
        }

        public final void setMasterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
        }

        @Override
        public final Builder masterUsername(String masterUsername) {
            this.masterUsername = masterUsername;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final String getSnapshotType() {
            return snapshotType;
        }

        public final void setSnapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
        }

        @Override
        public final Builder snapshotType(String snapshotType) {
            this.snapshotType = snapshotType;
            return this;
        }

        public final Integer getPercentProgress() {
            return percentProgress;
        }

        public final void setPercentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
        }

        @Override
        public final Builder percentProgress(Integer percentProgress) {
            this.percentProgress = percentProgress;
            return this;
        }

        public final Boolean getStorageEncrypted() {
            return storageEncrypted;
        }

        public final void setStorageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
        }

        @Override
        public final Builder storageEncrypted(Boolean storageEncrypted) {
            this.storageEncrypted = storageEncrypted;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getDbClusterSnapshotArn() {
            return dbClusterSnapshotArn;
        }

        public final void setDbClusterSnapshotArn(String dbClusterSnapshotArn) {
            this.dbClusterSnapshotArn = dbClusterSnapshotArn;
        }

        @Override
        public final Builder dbClusterSnapshotArn(String dbClusterSnapshotArn) {
            this.dbClusterSnapshotArn = dbClusterSnapshotArn;
            return this;
        }

        public final String getSourceDBClusterSnapshotArn() {
            return sourceDBClusterSnapshotArn;
        }

        public final void setSourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn) {
            this.sourceDBClusterSnapshotArn = sourceDBClusterSnapshotArn;
        }

        @Override
        public final Builder sourceDBClusterSnapshotArn(String sourceDBClusterSnapshotArn) {
            this.sourceDBClusterSnapshotArn = sourceDBClusterSnapshotArn;
            return this;
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public DBClusterSnapshot build() {
            return new DBClusterSnapshot(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
