/*
 * Copyright 2016 Hans Chen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package site.hanschen.easyloader.action;

import android.app.Notification;
import android.app.NotificationManager;
import android.appwidget.AppWidgetManager;
import android.graphics.Bitmap;
import android.widget.RemoteViews;

import site.hanschen.easyloader.EasyLoader;
import site.hanschen.easyloader.LoadedFrom;
import site.hanschen.easyloader.request.Request;
import site.hanschen.easyloader.util.Utils;

import static android.content.Context.NOTIFICATION_SERVICE;

public abstract class RemoteViewsAction extends Action<RemoteViewsAction.RemoteViewsTarget> {
    final RemoteViews remoteViews;
    final int         viewId;

    private RemoteViewsTarget target;

    public RemoteViewsAction(EasyLoader loader,
                             Request data,
                             RemoteViews remoteViews,
                             int viewId,
                             int errorResId,
                             int memoryPolicy,
                             int diskPolicy,
                             Object tag,
                             String key) {
        super(loader, null, data, memoryPolicy, diskPolicy, errorResId, null, key, tag, false);
        this.remoteViews = remoteViews;
        this.viewId = viewId;
    }

    @Override
    public void onComplete(Bitmap result, LoadedFrom from) {
        remoteViews.setImageViewBitmap(viewId, result);
        update();
    }

    @Override
    public void onError() {
        if (errorResId != 0) {
            setImageResource(errorResId);
        }
    }

    @Override
    public RemoteViewsTarget getTarget() {
        if (target == null) {
            target = new RemoteViewsTarget(remoteViews, viewId);
        }
        return target;
    }

    public void setImageResource(int resId) {
        remoteViews.setImageViewResource(viewId, resId);
        update();
    }

    abstract void update();

    static class RemoteViewsTarget {
        final RemoteViews remoteViews;
        final int         viewId;

        RemoteViewsTarget(RemoteViews remoteViews, int viewId) {
            this.remoteViews = remoteViews;
            this.viewId = viewId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o)
                return true;
            if (o == null || getClass() != o.getClass())
                return false;
            RemoteViewsTarget remoteViewsTarget = (RemoteViewsTarget) o;
            return viewId == remoteViewsTarget.viewId && remoteViews.equals(remoteViewsTarget.remoteViews);
        }

        @Override
        public int hashCode() {
            return 31 * remoteViews.hashCode() + viewId;
        }
    }

    public static class AppWidgetAction extends RemoteViewsAction {
        private final int[] appWidgetIds;

        public AppWidgetAction(EasyLoader loader,
                               Request data,
                               RemoteViews remoteViews,
                               int viewId,
                               int[] appWidgetIds,
                               int memoryPolicy,
                               int diskPolicy,
                               String key,
                               Object tag,
                               int errorResId) {
            super(loader, data, remoteViews, viewId, errorResId, memoryPolicy, diskPolicy, tag, key);
            this.appWidgetIds = appWidgetIds;
        }

        @Override
        void update() {
            AppWidgetManager manager = AppWidgetManager.getInstance(loader.getContext());
            manager.updateAppWidget(appWidgetIds, remoteViews);
        }
    }

    public static class NotificationAction extends RemoteViewsAction {
        private final int          notificationId;
        private final String       notificationTag;
        private final Notification notification;

        public NotificationAction(EasyLoader loader,
                                  Request data,
                                  RemoteViews remoteViews,
                                  int viewId,
                                  int notificationId,
                                  Notification notification,
                                  String notificationTag,
                                  int memoryPolicy,
                                  int diskPolicy,
                                  String key,
                                  Object tag,
                                  int errorResId) {
            super(loader, data, remoteViews, viewId, errorResId, memoryPolicy, diskPolicy, tag, key);
            this.notificationId = notificationId;
            this.notificationTag = notificationTag;
            this.notification = notification;
        }

        @Override
        void update() {
            NotificationManager manager = Utils.getService(loader.getContext(), NOTIFICATION_SERVICE);
            manager.notify(notificationTag, notificationId, notification);
        }
    }
}
