package sh.bourbon.gist_firebase

import com.google.firebase.inappmessaging.FirebaseInAppMessagingDisplay
import com.google.firebase.inappmessaging.FirebaseInAppMessagingDisplayCallbacks
import com.google.firebase.inappmessaging.model.*
import sh.bourbon.gist.data.model.Message
import sh.bourbon.gist.presentation.GistListener
import sh.bourbon.gist.presentation.GistSdk

class GistFirebaseInAppMessagingDisplay : FirebaseInAppMessagingDisplay, GistListener {

    private var messageId: String? = null
    private var currentCallbacks: FirebaseInAppMessagingDisplayCallbacks? = null

    init {
        GistSdk.addListener(this)
    }

    override fun displayMessage(
        message: InAppMessage,
        callbacks: FirebaseInAppMessagingDisplayCallbacks
    ) {
        when (message) {
            is BannerMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            is CardMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            is ModalMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            else -> callbacks.impressionDetected()
        }
    }

    override fun onMessageShown(message: Message) {
        if (message.messageId == messageId) {
            // Message was shown successfully, notify Firebase
            currentCallbacks?.impressionDetected()
        }
    }

    override fun onMessageDismissed(message: Message) {
        if (message.messageId == messageId) {
            // Message was dismissed, notify Firebase
            currentCallbacks?.messageDismissed(FirebaseInAppMessagingDisplayCallbacks.InAppMessagingDismissType.CLICK)
            cleanUp()
        }
    }

    override fun onAction(currentRoute: String, action: String) {
        // Action performed, notify Firebase
        val text = Text.builder().setText(action).setHexColor("#000000").build()
        val button = Button.builder().setText(text).setButtonHexColor("#FFFFFF").build()
        val inAppMessagingAction = Action.builder()
            .setButton(button)
            .build()

        currentCallbacks?.messageClicked(inAppMessagingAction)
    }

    override fun onError(message: Message) {
        cleanUp()
    }

    private fun tryShowMessageForNullableRoute(
        messageId: String?,
        callbacks: FirebaseInAppMessagingDisplayCallbacks
    ) {
        if (messageId != null) {
            this.messageId = messageId
            this.currentCallbacks = callbacks

            // Show message
            GistSdk.showMessage(Message(messageId))
        }
    }

    private fun cleanUp() {
        currentCallbacks = null
        messageId = null
    }
}