package sh.bourbon.gist_firebase

import com.google.firebase.inappmessaging.FirebaseInAppMessagingDisplay
import com.google.firebase.inappmessaging.FirebaseInAppMessagingDisplayCallbacks
import com.google.firebase.inappmessaging.model.*
import sh.bourbon.gist.presentation.GistListener
import sh.bourbon.gist.presentation.GistSdk

class GistFirebaseInAppMessagingDisplay : FirebaseInAppMessagingDisplay, GistListener {

    private var currentMessageRoute: String? = null
    private var currentCallbacks: FirebaseInAppMessagingDisplayCallbacks? = null

    init {
        GistSdk.addListener(this)
    }

    override fun displayMessage(
        message: InAppMessage,
        callbacks: FirebaseInAppMessagingDisplayCallbacks
    ) {
        when (message) {
            is BannerMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            is CardMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            is ModalMessage -> tryShowMessageForNullableRoute(message.title.text, callbacks)
            else -> callbacks.impressionDetected()
        }
    }

    override fun onMessageShown(messageId: String) {
        if (messageId == currentMessageRoute) {
            // Message was shown successfully, notify Firebase
            currentCallbacks?.impressionDetected()
        }
    }

    override fun onMessageDismissed(messageId: String) {
        if (messageId == currentMessageRoute) {
            // Message was dismissed, notify Firebase
            currentCallbacks?.messageDismissed(FirebaseInAppMessagingDisplayCallbacks.InAppMessagingDismissType.CLICK)
            cleanUp()
        }
    }

    override fun onAction(action: String) {
        // Action performed, notify Firebase
        val text = Text.builder().setText(action).build()
        val button = Button.builder().setText(text).build()
        val inAppMessagingAction = Action.builder()
            .setButton(button)
            .build()

        currentCallbacks?.messageClicked(inAppMessagingAction)
    }

    override fun onError(messageId: String) {
        cleanUp()
    }

    private fun tryShowMessageForNullableRoute(
        messageRoute: String?,
        callbacks: FirebaseInAppMessagingDisplayCallbacks
    ) {
        if (messageRoute != null) {
            this.currentMessageRoute = messageRoute
            this.currentCallbacks = callbacks

            // Show message
            GistSdk.showMessage(messageRoute)
        }
    }

    private fun cleanUp() {
        currentCallbacks = null
        currentMessageRoute = null
    }
}