package sh.bourbon.engine

import android.content.Context
import android.util.AttributeSet
import android.util.Log
import android.view.View
import android.widget.FrameLayout
import androidx.appcompat.app.AppCompatActivity
import androidx.fragment.app.Fragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.OnLifecycleEvent
import com.google.gson.Gson
import io.flutter.embedding.android.FlutterFragment
import io.flutter.embedding.android.FlutterView
import io.flutter.embedding.engine.FlutterEngine
import io.flutter.embedding.engine.FlutterEngineCache
import io.flutter.embedding.engine.dart.DartExecutor
import io.flutter.plugin.common.MethodChannel
import java.lang.IllegalStateException
import java.util.*


class BourbonEngineView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null
) : FrameLayout(context, attrs), LifecycleObserver {

    companion object {
        private const val CHANNEL = "bourbon.sh/engine"
        private const val ENGINE_ID = "flutterEngine"
    }

    private var listener: BourbonEngineListener? = null

    private val gson by lazy { Gson() }

    private val flutterFragment by lazy {
        FlutterFragment.withCachedEngine(ENGINE_ID)
            .renderMode(FlutterView.RenderMode.texture) // Required to animate the view
            .transparencyMode(FlutterView.TransparencyMode.transparent)
            .build<FlutterFragment>()
    }

    private val flutterEngine by lazy {
        val engine = FlutterEngine(context)
        engine.dartExecutor.executeDartEntrypoint(DartExecutor.DartEntrypoint.createDefault())

        return@lazy engine
    }

    private val methodChannel by lazy {
        MethodChannel(flutterEngine.dartExecutor, CHANNEL).apply {
            setMethodCallHandler { call, _ ->
                when (call.method) {
                    "bootstrapped" -> listener?.onBootstrapped()
                    "routeLoaded" -> {
                        getRouteFromArguments(call.arguments)?.let { listener?.onRouteLoaded(it) }
                    }
                    "routeError" -> {
                        getRouteFromArguments(call.arguments)?.let { listener?.onRouteError(it) }
                    }
                    "routeChanged" -> {
                        getRouteFromArguments(call.arguments)?.let { listener?.onRouteChanged(it) }
                    }
                    "tap" -> {
                        val action = call.arguments as? String
                        action?.let { listener?.onTap(it) }
                    }
                    else -> Log.d(this::class.java.simpleName, "Unhandled method: ${call.method}")

                }
            }
        }
    }

    init {
        View.inflate(context, R.layout.view_bourbon_engine, this)

        // Setup flutter engine
        FlutterEngineCache.getInstance().put(ENGINE_ID, flutterEngine)

        // Add fragment
        if (context !is AppCompatActivity) {
            throw IllegalStateException("BourbonEngineView must be used in an AppCompatActivity")
        }

        @Suppress("CAST_NEVER_SUCCEEDS")
        context.supportFragmentManager
            .beginTransaction()
            .add(R.id.fragmentContainer, flutterFragment as Fragment)
            .commit()
    }

    private fun getRouteFromArguments(arguments: Any?): String? {
        val argumentsMap: Map<String, String>? = arguments as? Map<String, String>
        return argumentsMap?.get("route")
    }

    fun setListener(listener: BourbonEngineListener) {
        this.listener = listener
    }


    fun setLifecycle(lifecycle: Lifecycle) {
        lifecycle.addObserver(this)
    }

    fun setup(configuration: EngineConfiguration) {
        val jsonConfig = gson.toJson(configuration)
        methodChannel.invokeMethod("setup", jsonConfig)
    }

    fun updateRoute(route: String, behaviour: RouteBehaviour) {
        val routeData = RouteData(route, behaviour.name.toLowerCase(Locale.ROOT))
        val jsonRouteData = gson.toJson(routeData)
        methodChannel.invokeMethod("updateRoute", jsonRouteData)
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_START)
    fun onStart() {
        flutterFragment.onStart()
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_RESUME)
    fun onResume() {
        flutterFragment.onResume()
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_PAUSE)
    fun onPause() {
        flutterFragment.onPause()
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_STOP)
    fun onStop() {
        flutterFragment.onStop()
    }

    @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
    fun onDestroy() {
        flutterFragment
    }

    private data class RouteData(val route: String, val behaviour: String)
}

interface BourbonEngineListener {

    fun onBootstrapped()

    fun onRouteLoaded(route: String)

    fun onRouteError(route: String)

    fun onRouteChanged(newRoute: String)

    fun onTap(action: String)
}