package pl.rjuszczyk.panorama.viewer;

import android.content.Context;
import android.content.res.Configuration;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.PointF;

import android.hardware.SensorManager;
import android.opengl.GLSurfaceView;
import android.util.AttributeSet;
import android.util.Log;
import android.view.GestureDetector;
import android.view.MotionEvent;
import android.view.OrientationEventListener;
import android.view.ScaleGestureDetector;

import java.util.Timer;
import java.util.TimerTask;

import pl.rjuszczyk.panorama.R;
import pl.rjuszczyk.panorama.gyroscope.GyroscopeHandlerOld;
import pl.rjuszczyk.panorama.multitouch.MoveGestureDetector;
import pl.rjuszczyk.panorama.multitouch.RotateGestureDetector;

import pl.rjuszczyk.panorama.gyroscope.GyroscopeHandler;

public class PanoramaGLSurfaceView extends GLSurfaceView
{
	public float MAX_X_ROT = Float.NaN;
	public float MIN_X_ROT = Float.NaN;
	public float MAX_Y_ROT = Float.NaN;
	public float MIN_Y_ROT = Float.NaN;
	public float MAX_SCALE = 0.3f;
	public float MIN_SCALE = 0.5f;

	private float DEFAULT_TOUCH_SCALE = 0.06f;
	private float TOUCH_SCALE = 0.06f;

	MoveGestureDetector mMoveDetector;
	RawImageDrawer mImageDrawer;
	RotateGestureDetector mRotateGestureDetector;
	GestureDetector mFlingDetector;

	GyroscopeHandler gyroscopeHandler;
	GyroscopeHandler gyroscopeHandler2;
	GyroscopeHandlerOld gyroscopeHandlerOld;
	private float mDefaultModelScale = 1f;
	private PanoramaRenderer mPanoramaRenderer;
	private float mScaleFactor = 1;
	int beginEvents = 0;

	private ScaleGestureDetector mScaleGestureDetector;
	private boolean isGyroAvailable;
	private boolean useVerticalGyro;
	private int currentGyroHandler = 1;
	private float[] currentRotationMatrix;
	private float[] currentRotationMatrix2;
	private float currentProgress = 0;
	private int targetProgress;
	boolean autoCorrection = false;
	private Timer timer;
	private int mDeviceOrientation;

	public PanoramaGLSurfaceView(Context context) {
		super(context);

		if(isInEditMode())
			return;

		mScaleGestureDetector = new ScaleGestureDetector(context, new ScaleListener());
		//mGestureDetector = new GestureDetector(context, new MyOnGestureListener());
		mMoveDetector = new MoveGestureDetector(context, new MoveListener());
		mRotateGestureDetector = new RotateGestureDetector(context, new MyRotateGestureDetector());

		mScaleFactor = mDefaultModelScale;

		mImageDrawer = new RawImageDrawer(getResources());



		mPanoramaRenderer = new PanoramaRenderer(context, mImageDrawer, R.raw.sphere);


		mPanoramaRenderer.MAX_X_ROT = MAX_X_ROT;
		mPanoramaRenderer.MIN_X_ROT = MIN_X_ROT;
		mPanoramaRenderer.MAX_Y_ROT = MAX_Y_ROT;
		mPanoramaRenderer.MIN_Y_ROT = MIN_Y_ROT;
		mPanoramaRenderer.MAX_SCALE = MAX_SCALE;
		mPanoramaRenderer.MIN_SCALE = MIN_SCALE;
		mPanoramaRenderer.isGyroAvailable = isGyroAvailable;
		mPanoramaRenderer.useVerticalGyro = useVerticalGyro;

		mPanoramaRenderer.setModelScale(mDefaultModelScale);

		setEGLContextClientVersion(2);
		super.setEGLConfigChooser(8 , 8, 8, 8, 16, 0);
		setRenderer(mPanoramaRenderer);
	}

	public PanoramaGLSurfaceView(Context context, AttributeSet attrs) {
		super(context, attrs);

		if(isInEditMode())
			return;

		TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.PanoramaGLSurfaceView, 0, 0);

		int imageResource = a.getResourceId(R.styleable.PanoramaGLSurfaceView_img, -1);
		boolean lockVertical = a.getBoolean(R.styleable.PanoramaGLSurfaceView_lockVertical, false);

		boolean useGyro = a.getBoolean(R.styleable.PanoramaGLSurfaceView_useGyro, true);
		float maxX = a.getFloat(R.styleable.PanoramaGLSurfaceView_maxX, Float.NaN);
		float minX = a.getFloat(R.styleable.PanoramaGLSurfaceView_minX, Float.NaN);
		float maxY = a.getFloat(R.styleable.PanoramaGLSurfaceView_maxY, Float.NaN);
		float minY = a.getFloat(R.styleable.PanoramaGLSurfaceView_minY, Float.NaN);
		float maxScale = a.getFloat(R.styleable.PanoramaGLSurfaceView_maxScale, Float.NaN);
		float minScale = a.getFloat(R.styleable.PanoramaGLSurfaceView_minScale, Float.NaN);
		int mappingType = a.getInt(R.styleable.PanoramaGLSurfaceView_mappingType, 0);

		if(!Float.isNaN(maxScale) && !Float.isNaN(minScale)) {
			MAX_SCALE = maxScale;
			MIN_SCALE = minScale;
		}

		if(useGyro && !lockVertical) {
			if(!(Float.isNaN(maxX) && Float.isNaN(minX) && Float.isNaN(maxY) && Float.isNaN(minY) )) {
				throw new RuntimeException("cant set max min if it is using gyro without lock vertical");
			}
		}

		isGyroAvailable = useGyro;
		useVerticalGyro = lockVertical;


		if(Float.isNaN(maxX) != Float.isNaN(minX) ) {
			throw new RuntimeException("You have to specify both maxX and minX");
		}

		if(Float.isNaN(maxY) != Float.isNaN(minY) ) {
			throw new RuntimeException("You have to specify both maxY and minY");
		}

		MAX_X_ROT = maxX;
		MIN_X_ROT = minX;
		MAX_Y_ROT = maxY;
		MIN_Y_ROT = minY;


		a.recycle();

		mScaleGestureDetector = new ScaleGestureDetector(context, new ScaleListener());
		//mGestureDetector = new GestureDetector(context, new MyOnGestureListener());
		mMoveDetector = new MoveGestureDetector(context, new MoveListener());
		mRotateGestureDetector = new RotateGestureDetector(context, new MyRotateGestureDetector());
		mFlingDetector = new GestureDetector(getContext(), new GestureDetector.OnGestureListener() {

			@Override
			public boolean onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {

				mPanoramaRenderer.speedX= -TOUCH_SCALE*velocityX;
				mPanoramaRenderer.speedY= -TOUCH_SCALE*velocityY;
				return false;
			}

			@Override
			public void onLongPress(MotionEvent e) {

			}

			@Override
			public boolean onDown(MotionEvent motionEvent) {
				return false;
			}

			@Override
			public void onShowPress(MotionEvent motionEvent) {

			}

			@Override
			public boolean onSingleTapUp(MotionEvent event) {

				return false;
			}

			@Override
			public boolean onScroll(MotionEvent motionEvent, MotionEvent motionEvent1, float v, float v1) {
				return false;
			}
		});
		mScaleFactor = mDefaultModelScale;

		mImageDrawer = new RawImageDrawer(getResources());

		int modelResource;
		if(mappingType == 0) {
			modelResource = R.raw.sphere;
		} else {
			modelResource = -1;
		}

		mPanoramaRenderer = new PanoramaRenderer(context, mImageDrawer, modelResource);
		mPanoramaRenderer.setModelScale(mDefaultModelScale);


		setEGLContextClientVersion(2);
		super.setEGLConfigChooser(8 , 8, 8, 8, 16, 0);
		setRenderer(mPanoramaRenderer);

		if(imageResource!=-1) {
			setTexDrawableResourceID(imageResource);
		}
	}



	public boolean onTouchEvent(MotionEvent event) {
		boolean retVal = mRotateGestureDetector.onTouchEvent(event);
		retVal = mScaleGestureDetector.onTouchEvent(event) || retVal;

		retVal = mMoveDetector.onTouchEvent(event) || retVal;
		retVal = mFlingDetector.onTouchEvent(event) || retVal;
		return retVal || super.onTouchEvent(event);
	}
	@Override
	protected void onAttachedToWindow() {
		super.onAttachedToWindow();

		if (isInEditMode())
			return;

		OrientationEventListener orientationEventListener = new OrientationEventListener(getContext(), SensorManager.SENSOR_DELAY_NORMAL) {
			@Override
			public void onOrientationChanged(int orientation) {
				mDeviceOrientation = orientation;
			}
		};
	}

	public void onPause() {
		gyroscopeHandler.stop();
		gyroscopeHandler2.stop();
		gyroscopeHandlerOld.stop();

		if(autoCorrection) {
			timer.cancel();
		}
	}

	public void onResume() {
		if(isInEditMode())
			return;

		gyroscopeHandler = new GyroscopeHandler();
		gyroscopeHandler2 = new GyroscopeHandler();
		gyroscopeHandlerOld = new GyroscopeHandlerOld();
		currentGyroHandler = 2;



		gyroscopeHandler.start(getContext(), new GyroscopeHandler.OnGyroscopeChanged() {


			@Override
			public void onGyroscopeChange(double x, double y, double z) {

			}


			@Override
			public void onGyroscopeChanged2(float[] currentRotationMatrix) {
				if (!useVerticalGyro && isGyroAvailable) {
					PanoramaGLSurfaceView.this.currentRotationMatrix = currentRotationMatrix;
					if (currentGyroHandler == 2) {
						currentProgress = lerp(0.01f, currentProgress, targetProgress);
						if (currentProgress < 0.01 && targetProgress == 0) currentProgress = 0;
						if (currentProgress > 0.99 && targetProgress == 1) currentProgress = 1;
						MyLog.d("gyro1", "onGyroscopeChanged2: currentProgress = " + currentProgress);
					} else {
						if (PanoramaGLSurfaceView.this.currentRotationMatrix == null ||
								PanoramaGLSurfaceView.this.currentRotationMatrix2 == null) {
							return;
						}

						float[] rotationMatrix = getCurrentRotationMatrix(
								PanoramaGLSurfaceView.this.currentRotationMatrix,
								PanoramaGLSurfaceView.this.currentRotationMatrix2,
								currentProgress
						);
						setModelRotationMatrix(rotationMatrix);
					}
				}
			}

			@Override
			public void onGyroscopeNotAvailable() {
				isGyroAvailable = false;
			}
		});

		gyroscopeHandler2.start(getContext(), new GyroscopeHandler.OnGyroscopeChanged() {


			@Override
			public void onGyroscopeChange(double x, double y, double z) {
			}

			@Override
			public void onGyroscopeChanged2(float[] currentRotationMatrix) {
				if (!useVerticalGyro && isGyroAvailable) {
					PanoramaGLSurfaceView.this.currentRotationMatrix2 = currentRotationMatrix;
					if (currentGyroHandler == 1) {
						currentProgress = lerp(0.01f, currentProgress, targetProgress);
						MyLog.d("gyro2", "onGyroscopeChanged2: currentProgress = " + currentProgress);
						if (currentProgress < 0.01 && targetProgress == 0) currentProgress = 0;
						if (currentProgress > 0.99 && targetProgress == 1) currentProgress = 1;

					} else {
						if (PanoramaGLSurfaceView.this.currentRotationMatrix == null ||
								PanoramaGLSurfaceView.this.currentRotationMatrix2 == null) {
							return;
						}

						float[] rotationMatrix = getCurrentRotationMatrix(
								PanoramaGLSurfaceView.this.currentRotationMatrix,
								PanoramaGLSurfaceView.this.currentRotationMatrix2,
								currentProgress
						);
						setModelRotationMatrix(rotationMatrix);
					}
				}
			}

			@Override
			public void onGyroscopeNotAvailable() {
				isGyroAvailable = false;
			}
		});

		gyroscopeHandlerOld.start(getContext(), new GyroscopeHandlerOld.OnGyroscopeChanged() {
			@Override
			public void onGyroscopeChange(double x, double y, double z) {
				if(useVerticalGyro && isGyroAvailable) {
					if (getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE) {
						int orientation = 90 * Math.round(mDeviceOrientation / 90);
						if (orientation == 90 || orientation == 0) {
							mPanoramaRenderer.setModelRotationX((float) -(z - 90));
							mPanoramaRenderer.setModelRotationY((float) -(x + 90));
						} else {
							mPanoramaRenderer.setModelRotationX((float) -(z + 90));
							mPanoramaRenderer.setModelRotationY((float) -x);
						}
					} else if (getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
						mPanoramaRenderer.setModelRotationX((float) (y));
						mPanoramaRenderer.setModelRotationY((float) -x);
					}
				}
			}

			@Override
			public void onGyroscopeNotAvailable() {

			}
		});
		if(autoCorrection) {
			timer = new Timer();
			timer.scheduleAtFixedRate(new TimerTask() {
				@Override
				public void run() {
					reset();
				}
			}, 0, 10000);
		}
	}

	private void setModelRotationMatrix(float[] rotationMatrix) {
		mPanoramaRenderer.setModelRotationMatrix(rotationMatrix);
	}

	public void setOnDrawListener(OnDrawListener onDrawListener) {
		mPanoramaRenderer.setOnDrawListener(onDrawListener);
	}


	float[] getCurrentRotationMatrix(
		float[] matrix1,
		float[] matrix2,
		float progress
	) {
		float[] matrixResult = new float[matrix1.length];
		for (int i = 0; i < matrix1.length; i++) {
			matrixResult[i] = lerp(progress, matrix1[i], matrix2[i]);
		}

		return matrixResult;
	}

	float lerp(float t, float a, float b) {
		return (1 - t) * a + t * b;
	}

	public void reset() {
		if(gyroscopeHandler == null || gyroscopeHandler2 == null) {
			return;
		}

		if(currentGyroHandler==1) {
			gyroscopeHandler.reset();
			gyroscopeHandler.restart();
			currentGyroHandler = 2;
			targetProgress = 0;
		} else {
			gyroscopeHandler2.reset();
			gyroscopeHandler2.restart();
			currentGyroHandler = 1;
			targetProgress = 1;
		}

//		mPanoramaRenderer.resetCamera();
	}




	private class ScaleListener extends ScaleGestureDetector.SimpleOnScaleGestureListener {
		@Override
		public boolean onScale(ScaleGestureDetector detector) {
			mScaleFactor = mPanoramaRenderer.getModelScale() / detector.getScaleFactor();

			// Don't let the object get too small or too large.
			mScaleFactor = Math.max(0.30f, Math.min(mScaleFactor, 1.5f));
			TOUCH_SCALE = DEFAULT_TOUCH_SCALE * mScaleFactor;
 			//     mPanoramaRenderer.mCameraZ *= mScaleFactor;
			mPanoramaRenderer.setModelScale(mScaleFactor);
			invalidate();
			return true;
		}
		@Override
		public boolean onScaleBegin(ScaleGestureDetector detector) {
			beginEvents++;
			return true;
		}

		@Override
		public void onScaleEnd(ScaleGestureDetector detector) {
			beginEvents--;
			super.onScaleEnd(detector);

		}
	}

	private class MyRotateGestureDetector implements RotateGestureDetector.OnRotateGestureListener {

		@Override
		public boolean onRotate(RotateGestureDetector detector) {
			mPanoramaRenderer.rotateZ(detector.getRotationDegreesDelta());
			return true;
		}

		@Override
		public boolean onRotateBegin(RotateGestureDetector detector) {
			beginEvents++;
			return true;
		}

		@Override
		public void onRotateEnd(RotateGestureDetector detector) {
			beginEvents--;
		}
	}



	public void setPanoramaResourceId(int tex_resourceID) {
		mPanoramaRenderer.setTex_resourceID(tex_resourceID);
	}

	public void setTexDrawableResourceID(int tex_resourceID) {
		mPanoramaRenderer.setTex_resourceID(tex_resourceID);
	}

	public void setPanoramaBitmap(Bitmap bitmap) {
		mPanoramaRenderer.setTextureBitmap(bitmap);
	}

	public float[] getMVPMatrix() {
		return mPanoramaRenderer.getMVPMatrix();
	}

	public PointF unProject(double latitude, double longitude) {
		float[] posNew = MatrixCalculator.calculateOnScreenPosNormalized(getMVPMatrix(), latitude, longitude);

		int w = getWidth();
		int h = getHeight();
		final int x = (int) (w * (1f + posNew[0]) / 2f);
		int y = (int) (h * (1f + posNew[1]) / 2f);
		y = h-y;

		return new PointF(x,y);
	}

	private class MoveListener implements MoveGestureDetector.OnMoveGestureListener {
		@Override
		public boolean onMove(MoveGestureDetector detector) {
			if (beginEvents != 0) {
				return true;
			}
			float distanceX = detector.getFocusDelta().x;
			float distanceY = detector.getFocusDelta().y;

			mPanoramaRenderer.rotate(-distanceX*TOUCH_SCALE, -distanceY*TOUCH_SCALE);
			return true;
		}

		@Override
		public boolean onMoveBegin(MoveGestureDetector detector) {
			return true;
		}

		@Override
		public void onMoveEnd(MoveGestureDetector detector) {

		}
	}
}