/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.security.PrivilegedActionException;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BooleanSupplier;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.kerberos.KerberosTicket;
import javax.security.sasl.AuthorizeCallback;
import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.callback.AuthenticationCompleteCallback;
import org.wildfly.security.auth.callback.CachedIdentityAuthorizeCallback;
import org.wildfly.security.auth.callback.IdentityCredentialCallback;
import org.wildfly.security.auth.callback.ServerCredentialCallback;
import org.wildfly.security.auth.principal.NamePrincipal;
import org.wildfly.security.auth.server.SecurityIdentity;
import org.wildfly.security.cache.CachedIdentity;
import org.wildfly.security.cache.IdentityCache;
import org.wildfly.security.credential.GSSKerberosCredential;
import org.wildfly.security.http.HttpAuthenticationException;
import org.wildfly.security.http.HttpConstants;
import org.wildfly.security.http.HttpScope;
import org.wildfly.security.http.HttpServerAuthenticationMechanism;
import org.wildfly.security.http.HttpServerRequest;
import org.wildfly.security.http.HttpServerResponse;
import org.wildfly.security.http.Scope;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism.MechanismUtil;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util._private.Arrays2;

public class SpnegoAuthenticationMechanism
implements HttpServerAuthenticationMechanism {
    private static final String CHALLENGE_PREFIX = "Negotiate ";
    private static final String GSS_CONTEXT_KEY = SpnegoAuthenticationMechanism.class.getName() + ".GSSContext";
    private static final String KERBEROS_TICKET = SpnegoAuthenticationMechanism.class.getName() + ".KerberosTicket";
    private static final String CACHED_IDENTITY_KEY = SpnegoAuthenticationMechanism.class.getName() + ".elytron-identity";
    private final CallbackHandler callbackHandler;
    private final GSSManager gssManager;
    private final Scope[] storageScopes;

    SpnegoAuthenticationMechanism(CallbackHandler callbackHandler, Map<String, ?> properties) {
        Assert.checkNotNullParam((String)"callbackHandler", (Object)callbackHandler);
        Assert.checkNotNullParam((String)"properties", properties);
        this.callbackHandler = callbackHandler;
        this.gssManager = properties.containsKey(HttpConstants.CONFIG_GSS_MANAGER) ? (GSSManager)properties.get(HttpConstants.CONFIG_GSS_MANAGER) : GSSManager.getInstance();
        String scopesProperty = (String)properties.get(HttpConstants.CONFIG_STATE_SCOPES);
        if (scopesProperty == null) {
            this.storageScopes = new Scope[]{Scope.SESSION, Scope.CONNECTION};
        } else {
            String[] names = scopesProperty.split(",");
            this.storageScopes = new Scope[names.length];
            for (int i = 0; i < names.length; ++i) {
                if ("NONE".equals(names[i])) {
                    this.storageScopes[i] = null;
                    continue;
                }
                Scope scope = Scope.valueOf(names[i]);
                if (scope == Scope.APPLICATION || scope == Scope.GLOBAL) {
                    throw ElytronMessages.log.unsuitableScope(scope.name());
                }
                this.storageScopes[i] = scope;
            }
        }
    }

    @Override
    public String getMechanismName() {
        return "SPNEGO";
    }

    @Override
    public void evaluateRequest(HttpServerRequest request) throws HttpAuthenticationException {
        List<String> authorizationValues;
        Optional challenge;
        HttpScope storageScope = this.getStorageScope(request);
        IdentityCache identityCache = null;
        if ((identityCache = this.createIdentityCache(identityCache, storageScope, false)) != null && this.attemptReAuthentication(identityCache, request)) {
            ElytronMessages.log.trace("Successfully authorized using cached identity");
            return;
        }
        boolean scopeIsUsable = storageScope != null && storageScope.exists();
        GSSContext gssContext = scopeIsUsable ? storageScope.getAttachment(GSS_CONTEXT_KEY, GSSContext.class) : null;
        KerberosTicket kerberosTicket = scopeIsUsable ? storageScope.getAttachment(KERBEROS_TICKET, KerberosTicket.class) : null;
        ElytronMessages.log.tracef("Evaluating SPNEGO request: cached GSSContext = %s", gssContext);
        if (gssContext != null && gssContext.isEstablished()) {
            if (this.authorizeSrcName(gssContext, identityCache = this.createIdentityCache(identityCache, storageScope, true))) {
                ElytronMessages.log.trace("Successfully authorized using cached GSSContext");
                request.authenticationComplete();
                return;
            }
            SpnegoAuthenticationMechanism.clearAttachments(storageScope);
            gssContext = null;
            kerberosTicket = null;
        }
        if (gssContext == null) {
            GSSCredential serviceGssCredential;
            ServerCredentialCallback gssCredentialCallback = new ServerCredentialCallback(GSSKerberosCredential.class);
            try {
                ElytronMessages.log.trace("Obtaining GSSCredential for the service from callback handler...");
                this.callbackHandler.handle(new Callback[]{gssCredentialCallback});
                serviceGssCredential = gssCredentialCallback.applyToCredential(GSSKerberosCredential.class, GSSKerberosCredential::getGssCredential);
                kerberosTicket = gssCredentialCallback.applyToCredential(GSSKerberosCredential.class, GSSKerberosCredential::getKerberosTicket);
            }
            catch (IOException | UnsupportedCallbackException e) {
                throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason("SPNEGO", e).toHttpAuthenticationException();
            }
            if (serviceGssCredential == null) {
                throw ElytronMessages.log.unableToObtainServerCredential("SPNEGO").toHttpAuthenticationException();
            }
            try {
                gssContext = this.gssManager.createContext(serviceGssCredential);
                if (ElytronMessages.log.isTraceEnabled()) {
                    ElytronMessages.log.tracef("Using SpnegoAuthenticationMechanism to authenticate %s using the following mechanisms: [%s]", serviceGssCredential.getName(), Arrays2.objectToString(serviceGssCredential.getMechs()));
                }
            }
            catch (GSSException e) {
                throw ElytronMessages.log.mechUnableToCreateGssContext("SPNEGO", e).toHttpAuthenticationException();
            }
        }
        Optional<Object> optional = challenge = (authorizationValues = request.getRequestHeaderValues("Authorization")) != null ? authorizationValues.stream().filter(s -> s.startsWith(CHALLENGE_PREFIX)).limit(1L).map(s -> s.substring(CHALLENGE_PREFIX.length())).findFirst() : Optional.empty();
        if (ElytronMessages.log.isTraceEnabled()) {
            ElytronMessages.log.tracef("Sent HTTP authorizations: [%s]", Arrays2.objectToString(authorizationValues));
        }
        if (challenge.isPresent()) {
            byte[] responseToken;
            ElytronMessages.log.trace("Processing incoming response to a challenge...");
            if (storageScope != null && (storageScope.exists() || storageScope.create())) {
                storageScope.setAttachment(GSS_CONTEXT_KEY, gssContext);
                ElytronMessages.log.tracef("Caching GSSContext %s", gssContext);
                storageScope.setAttachment(KERBEROS_TICKET, kerberosTicket);
                ElytronMessages.log.tracef("Caching KerberosTicket %s", kerberosTicket);
            } else {
                storageScope = null;
                ElytronMessages.log.trace("No usable HttpScope for storage, continuation will not be possible");
            }
            byte[] decodedValue = ByteIterator.ofBytes(((String)challenge.get()).getBytes(StandardCharsets.UTF_8)).base64Decode().drain();
            Subject subject = new Subject(true, Collections.emptySet(), Collections.emptySet(), kerberosTicket != null ? Collections.singleton(kerberosTicket) : Collections.emptySet());
            try {
                GSSContext finalGssContext = gssContext;
                responseToken = Subject.doAs(subject, () -> finalGssContext.acceptSecContext(decodedValue, 0, decodedValue.length));
            }
            catch (PrivilegedActionException e) {
                ElytronMessages.log.trace("Call to acceptSecContext failed.", e.getCause());
                this.handleCallback(AuthenticationCompleteCallback.FAILED);
                SpnegoAuthenticationMechanism.clearAttachments(storageScope);
                request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"));
                return;
            }
            if (gssContext.isEstablished()) {
                GSSCredential gssCredential;
                try {
                    gssCredential = gssContext.getCredDelegState() ? gssContext.getDelegCred() : null;
                }
                catch (GSSException e) {
                    ElytronMessages.log.trace("Unable to access delegated credential despite being delegated.", e);
                    this.handleCallback(AuthenticationCompleteCallback.FAILED);
                    SpnegoAuthenticationMechanism.clearAttachments(storageScope);
                    request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"));
                    return;
                }
                if (gssCredential != null) {
                    ElytronMessages.log.trace("Associating delegated GSSCredential with identity.");
                    this.handleCallback(new IdentityCredentialCallback(new GSSKerberosCredential(gssCredential), true));
                } else {
                    ElytronMessages.log.trace("No GSSCredential delegated from client.");
                }
                ElytronMessages.log.trace("GSSContext established, authorizing...");
                identityCache = this.createIdentityCache(identityCache, storageScope, true);
                if (this.authorizeSrcName(gssContext, identityCache)) {
                    ElytronMessages.log.trace("GSSContext established and authorized - authentication complete");
                    request.authenticationComplete(response -> this.sendChallenge(responseToken, response, 0));
                    return;
                }
                ElytronMessages.log.trace("Authorization of established GSSContext failed");
                this.handleCallback(AuthenticationCompleteCallback.FAILED);
                SpnegoAuthenticationMechanism.clearAttachments(storageScope);
                request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"), this::sendBareChallenge);
                return;
            }
            if (responseToken != null && storageScope != null) {
                ElytronMessages.log.trace("GSSContext establishing - sending negotiation token to the peer");
                request.authenticationInProgress(response -> this.sendChallenge(responseToken, response, 401));
                return;
            }
            ElytronMessages.log.trace("GSSContext establishing - unable to hold GSSContext so continuation will not be possible");
            this.handleCallback(AuthenticationCompleteCallback.FAILED);
            request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"));
            return;
        }
        ElytronMessages.log.trace("Request lacks valid authentication credentials");
        SpnegoAuthenticationMechanism.clearAttachments(storageScope);
        request.noAuthenticationInProgress(this::sendBareChallenge);
    }

    private HttpScope getStorageScope(HttpServerRequest request) throws HttpAuthenticationException {
        for (Scope scope : this.storageScopes) {
            if (scope == null) {
                return null;
            }
            HttpScope httpScope = request.getScope(scope);
            if (httpScope != null && httpScope.supportsAttachments()) {
                if (ElytronMessages.log.isTraceEnabled()) {
                    ElytronMessages.log.tracef("Using HttpScope '%s' with ID '%s'", scope.name(), httpScope.getID());
                }
                return httpScope;
            }
            if (!ElytronMessages.log.isTraceEnabled()) continue;
            ElytronMessages.log.tracef(httpScope == null ? "HttpScope %s not supported" : "HttpScope %s does not support attachments", (Object)scope);
        }
        throw ElytronMessages.log.unableToIdentifyHttpScope();
    }

    private IdentityCache createIdentityCache(IdentityCache existingCache, final HttpScope httpScope, boolean forUpdate) {
        if (!(existingCache == null && httpScope != null && httpScope.supportsAttachments() && (httpScope.exists() || forUpdate && httpScope.create()))) {
            return existingCache;
        }
        return new IdentityCache(){

            @Override
            public CachedIdentity remove() {
                CachedIdentity cachedIdentity = this.get();
                httpScope.setAttachment(CACHED_IDENTITY_KEY, null);
                return cachedIdentity;
            }

            @Override
            public void put(SecurityIdentity identity) {
                httpScope.setAttachment(CACHED_IDENTITY_KEY, new CachedIdentity("SPNEGO", identity));
            }

            @Override
            public CachedIdentity get() {
                return httpScope.getAttachment(CACHED_IDENTITY_KEY, CachedIdentity.class);
            }
        };
    }

    private static void clearAttachments(HttpScope scope) {
        if (scope != null) {
            scope.setAttachment(GSS_CONTEXT_KEY, null);
            scope.setAttachment(KERBEROS_TICKET, null);
        }
    }

    private void sendBareChallenge(HttpServerResponse response) {
        response.addResponseHeader("WWW-Authenticate", "Negotiate");
        response.setStatusCode(401);
    }

    private void sendChallenge(byte[] responseToken, HttpServerResponse response, int statusCode) {
        if (ElytronMessages.log.isTraceEnabled()) {
            ElytronMessages.log.tracef("Sending intermediate challenge: %s", Arrays2.objectToString(responseToken));
        }
        if (responseToken == null) {
            response.addResponseHeader("WWW-Authenticate", "Negotiate");
        } else {
            String responseConverted = ByteIterator.ofBytes(responseToken).base64Encode().drainToString();
            response.addResponseHeader("WWW-Authenticate", CHALLENGE_PREFIX + responseConverted);
        }
        if (statusCode != 0) {
            response.setStatusCode(statusCode);
        }
    }

    private boolean attemptReAuthentication(IdentityCache identityCache, HttpServerRequest request) throws HttpAuthenticationException {
        CachedIdentityAuthorizeCallback authorizeCallback = new CachedIdentityAuthorizeCallback(identityCache);
        try {
            this.callbackHandler.handle(new Callback[]{authorizeCallback});
        }
        catch (IOException | UnsupportedCallbackException e) {
            throw new HttpAuthenticationException(e);
        }
        if (authorizeCallback.isAuthorized()) {
            try {
                this.handleCallback(AuthenticationCompleteCallback.SUCCEEDED);
            }
            catch (IOException e) {
                throw new HttpAuthenticationException(e);
            }
            request.authenticationComplete(null, identityCache::remove);
            return true;
        }
        return false;
    }

    private boolean authorizeSrcName(GSSContext gssContext, IdentityCache identityCache) throws HttpAuthenticationException {
        GSSName srcName;
        try {
            srcName = gssContext.getSrcName();
            if (srcName == null) {
                ElytronMessages.log.trace("Authorization failed - srcName of GSSContext (name of initiator) is null - wrong realm or kdc?");
                return false;
            }
        }
        catch (GSSException e) {
            ElytronMessages.log.trace("Unable to obtain srcName from established GSSContext.", e);
            return false;
        }
        if (gssContext.getCredDelegState()) {
            try {
                GSSCredential credential = gssContext.getDelegCred();
                ElytronMessages.log.tracef("Credential delegation enabled, delegated credential = %s", credential);
                MechanismUtil.handleCallbacks("SPNEGO", this.callbackHandler, new IdentityCredentialCallback(new GSSKerberosCredential(credential), true));
            }
            catch (UnsupportedCallbackException credential) {
            }
            catch (AuthenticationMechanismException e) {
                throw e.toHttpAuthenticationException();
            }
            catch (GSSException e) {
                throw new HttpAuthenticationException(e);
            }
        } else {
            ElytronMessages.log.trace("Credential delegation not enabled");
        }
        boolean authorized = false;
        try {
            Callback authorizeCallBack;
            BooleanSupplier authorizedFunction;
            String clientName = srcName.toString();
            if (identityCache != null) {
                CachedIdentityAuthorizeCallback cacheCallback = new CachedIdentityAuthorizeCallback((Principal)new NamePrincipal(clientName), identityCache, true);
                authorizedFunction = cacheCallback::isAuthorized;
                authorizeCallBack = cacheCallback;
            } else {
                AuthorizeCallback plainCallback = new AuthorizeCallback(clientName, clientName);
                authorizedFunction = plainCallback::isAuthorized;
                authorizeCallBack = plainCallback;
            }
            this.callbackHandler.handle(new Callback[]{authorizeCallBack});
            authorized = authorizedFunction.getAsBoolean();
            ElytronMessages.log.tracef("Authorized by callback handler = %b  clientName = [%s]", authorized, clientName);
        }
        catch (IOException e) {
            ElytronMessages.log.trace("IOException during AuthorizeCallback handling", e);
            throw ElytronMessages.log.mechServerSideAuthenticationFailed("SPNEGO", e).toHttpAuthenticationException();
        }
        catch (UnsupportedCallbackException unsupportedCallbackException) {
            // empty catch block
        }
        if (authorized) {
            this.handleCallback(AuthenticationCompleteCallback.SUCCEEDED);
        }
        return authorized;
    }

    private void handleCallback(Callback callback) throws HttpAuthenticationException {
        try {
            MechanismUtil.handleCallbacks("SPNEGO", this.callbackHandler, callback);
        }
        catch (AuthenticationMechanismException e) {
            throw e.toHttpAuthenticationException();
        }
        catch (UnsupportedCallbackException unsupportedCallbackException) {
            // empty catch block
        }
    }
}

