/*
 * Copyright 2012-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.support;

import java.util.Map;

import org.jspecify.annotations.Nullable;

import org.springframework.boot.EnvironmentPostProcessor;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.PropertySourceInfo;
import org.springframework.boot.origin.Origin;
import org.springframework.boot.origin.OriginLookup;
import org.springframework.boot.origin.SystemEnvironmentOrigin;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertySource;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.core.env.SystemEnvironmentPropertySource;
import org.springframework.util.StringUtils;

/**
 * An {@link EnvironmentPostProcessor} that replaces the systemEnvironment
 * {@link SystemEnvironmentPropertySource} with an
 * {@link OriginAwareSystemEnvironmentPropertySource} that can track the
 * {@link SystemEnvironmentOrigin} for every system environment property.
 *
 * @author Madhura Bhave
 * @author Phillip Webb
 * @since 4.0.0
 */
public class SystemEnvironmentPropertySourceEnvironmentPostProcessor implements EnvironmentPostProcessor, Ordered {

	/**
	 * The default order for the processor.
	 */
	public static final int DEFAULT_ORDER = SpringApplicationJsonEnvironmentPostProcessor.DEFAULT_ORDER - 1;

	private int order = DEFAULT_ORDER;

	@Override
	public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
		postProcessEnvironment(environment, application.getEnvironmentPrefix());
	}

	private void postProcessEnvironment(ConfigurableEnvironment environment, @Nullable String environmentPrefix) {
		String sourceName = StandardEnvironment.SYSTEM_ENVIRONMENT_PROPERTY_SOURCE_NAME;
		PropertySource<?> propertySource = environment.getPropertySources().get(sourceName);
		if (propertySource != null) {
			replacePropertySource(environment, sourceName, propertySource, environmentPrefix);
		}
	}

	@SuppressWarnings("unchecked")
	private void replacePropertySource(ConfigurableEnvironment environment, String sourceName,
			PropertySource<?> propertySource, @Nullable String environmentPrefix) {
		Map<String, Object> originalSource = (Map<String, Object>) propertySource.getSource();
		SystemEnvironmentPropertySource source = new OriginAwareSystemEnvironmentPropertySource(sourceName,
				originalSource, environmentPrefix);
		environment.getPropertySources().replace(sourceName, source);
	}

	@Override
	public int getOrder() {
		return this.order;
	}

	public void setOrder(int order) {
		this.order = order;
	}

	/**
	 * Post-process the given {@link ConfigurableEnvironment} by copying appropriate
	 * settings from a parent {@link ConfigurableEnvironment}.
	 * @param environment the environment to post-process
	 * @param parentEnvironment the parent environment
	 * @since 3.4.12
	 */
	public static void postProcessEnvironment(ConfigurableEnvironment environment,
			ConfigurableEnvironment parentEnvironment) {
		PropertySource<?> parentSystemEnvironmentPropertySource = parentEnvironment.getPropertySources()
			.get(StandardEnvironment.SYSTEM_ENVIRONMENT_PROPERTY_SOURCE_NAME);
		if (parentSystemEnvironmentPropertySource instanceof OriginAwareSystemEnvironmentPropertySource parentOriginAwareSystemEnvironmentPropertySource) {
			new SystemEnvironmentPropertySourceEnvironmentPostProcessor().postProcessEnvironment(environment,
					parentOriginAwareSystemEnvironmentPropertySource.getPrefix());
		}
	}

	/**
	 * {@link SystemEnvironmentPropertySource} that also tracks {@link Origin}.
	 */
	protected static class OriginAwareSystemEnvironmentPropertySource extends SystemEnvironmentPropertySource
			implements PropertySourceInfo, OriginLookup<String> {

		private final @Nullable String prefix;

		OriginAwareSystemEnvironmentPropertySource(String name, Map<String, Object> source,
				@Nullable String environmentPrefix) {
			super(name, source);
			this.prefix = determinePrefix(environmentPrefix);
		}

		private @Nullable String determinePrefix(@Nullable String environmentPrefix) {
			if (!StringUtils.hasText(environmentPrefix)) {
				return null;
			}
			if (environmentPrefix.endsWith(".") || environmentPrefix.endsWith("_") || environmentPrefix.endsWith("-")) {
				return environmentPrefix.substring(0, environmentPrefix.length() - 1);
			}
			return environmentPrefix;
		}

		@Override
		public boolean containsProperty(String name) {
			return super.containsProperty(name);
		}

		@Override
		public @Nullable Object getProperty(String name) {
			return super.getProperty(name);
		}

		@Override
		public @Nullable Origin getOrigin(String key) {
			String property = resolvePropertyName(key);
			if (super.containsProperty(property)) {
				return new SystemEnvironmentOrigin(property);
			}
			return null;
		}

		@Override
		public @Nullable String getPrefix() {
			return this.prefix;
		}

		@Override
		public boolean isImmutable() {
			return (Object) getSource() == System.getenv();
		}

	}

}
