/*
 * Decompiled with CFR 0.152.
 */
package org.openrewrite.gradle.isolated;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.gradle.api.Project;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.openrewrite.ExecutionContext;
import org.openrewrite.SourceFile;
import org.openrewrite.gradle.RewriteExtension;
import org.openrewrite.gradle.TimeUtils;
import org.openrewrite.hcl.HclParser;
import org.openrewrite.json.JsonParser;
import org.openrewrite.properties.PropertiesParser;
import org.openrewrite.protobuf.ProtoParser;
import org.openrewrite.quark.QuarkParser;
import org.openrewrite.text.PlainTextParser;
import org.openrewrite.xml.XmlParser;
import org.openrewrite.yaml.YamlParser;

public class ResourceParser {
    private static final Set<String> DEFAULT_IGNORED_DIRECTORIES = new HashSet<String>(Arrays.asList("build", "target", "out", ".gradle", ".idea", ".project", "node_modules", ".git", ".metadata", ".DS_Store"));
    private static final Logger logger = Logging.getLogger(ResourceParser.class);
    private final Path baseDir;
    private final Collection<PathMatcher> exclusions;
    private final Collection<PathMatcher> plainTextMasks;
    private final int sizeThresholdMb;

    public ResourceParser(Path baseDir, Project project, RewriteExtension extension) {
        this.baseDir = baseDir;
        this.exclusions = this.pathMatchers(baseDir, ResourceParser.mergeExclusions(project, extension));
        this.plainTextMasks = this.pathMatchers(baseDir, extension.getPlainTextMasks());
        this.sizeThresholdMb = extension.getSizeThresholdMb();
    }

    private static Collection<String> mergeExclusions(Project project, RewriteExtension extension) {
        return Stream.concat(project.getSubprojects().stream().map(subproject -> project.getProjectDir().toPath().relativize(subproject.getProjectDir().toPath()).toString()), extension.getExclusions().stream()).collect(Collectors.toList());
    }

    private Collection<PathMatcher> pathMatchers(Path basePath, Collection<String> pathExpressions) {
        return pathExpressions.stream().map(o -> basePath.getFileSystem().getPathMatcher("glob:" + o)).collect(Collectors.toList());
    }

    public List<SourceFile> parse(Path projectDir, Collection<Path> alreadyParsed, ExecutionContext ctx) {
        ArrayList<SourceFile> sourceFiles;
        logger.info("Parsing other sources from {}", (Object)projectDir);
        Instant start = Instant.now();
        try {
            sourceFiles = new ArrayList<SourceFile>(this.parseSourceFiles(projectDir, alreadyParsed, ctx));
        }
        catch (IOException e) {
            logger.error(e.getMessage(), (Throwable)e);
            throw new UncheckedIOException(e);
        }
        if (sourceFiles.size() > 0) {
            Duration duration = Duration.between(start, Instant.now());
            logger.info("Finished parsing {} other sources from {} in {} ({} per source)", new Object[]{sourceFiles.size(), projectDir, TimeUtils.prettyPrint(duration), TimeUtils.prettyPrint(duration.dividedBy(sourceFiles.size()))});
        }
        return sourceFiles;
    }

    public List<Path> listSources(final Path searchDir) throws IOException {
        final JsonParser jsonParser = new JsonParser();
        final XmlParser xmlParser = new XmlParser();
        final YamlParser yamlParser = new YamlParser();
        final PropertiesParser propertiesParser = new PropertiesParser();
        final ProtoParser protoParser = new ProtoParser();
        final HclParser hclParser = HclParser.builder().build();
        final ArrayList<Path> resources = new ArrayList<Path>();
        Files.walkFileTree(searchDir, Collections.emptySet(), 16, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                if (ResourceParser.this.isExcluded(dir) || ResourceParser.this.isIgnoredDirectory(searchDir, dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (attrs.size() != 0L && !attrs.isOther() && !ResourceParser.this.isExcluded(file) && !ResourceParser.this.isOverSizeThreshold(attrs.size()) && (jsonParser.accept(file) || xmlParser.accept(file) || yamlParser.accept(file) || propertiesParser.accept(file) || protoParser.accept(file) || hclParser.accept(file))) {
                    resources.add(file);
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return resources;
    }

    public <S extends SourceFile> List<S> parseSourceFiles(final Path searchDir, final Collection<Path> alreadyParsed, ExecutionContext ctx) throws IOException {
        final ArrayList resources = new ArrayList();
        final ArrayList quarkPaths = new ArrayList();
        final ArrayList plainTextPaths = new ArrayList();
        Files.walkFileTree(searchDir, Collections.emptySet(), 16, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                if (ResourceParser.this.isExcluded(dir) || ResourceParser.this.isIgnoredDirectory(searchDir, dir)) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                if (!(attrs.isOther() || attrs.isSymbolicLink() || alreadyParsed.contains(file) || ResourceParser.this.isExcluded(file))) {
                    if (ResourceParser.this.isOverSizeThreshold(attrs.size())) {
                        logger.info("Parsing as quark " + file + " as its size + " + attrs.size() / 0x100000L + "Mb exceeds size threshold " + ResourceParser.this.sizeThresholdMb + "Mb");
                        quarkPaths.add(file);
                    } else if (ResourceParser.this.isParsedAsPlainText(file)) {
                        System.out.println("Plain Text :" + file);
                        plainTextPaths.add(file);
                    } else {
                        resources.add(file);
                    }
                }
                return FileVisitResult.CONTINUE;
            }
        });
        ArrayList sourceFiles = new ArrayList(resources.size());
        JsonParser jsonParser = new JsonParser();
        ArrayList jsonPaths = new ArrayList();
        XmlParser xmlParser = new XmlParser();
        ArrayList xmlPaths = new ArrayList();
        YamlParser yamlParser = new YamlParser();
        ArrayList yamlPaths = new ArrayList();
        PropertiesParser propertiesParser = new PropertiesParser();
        ArrayList propertiesPaths = new ArrayList();
        ProtoParser protoParser = new ProtoParser();
        ArrayList protoPaths = new ArrayList();
        HclParser hclParser = HclParser.builder().build();
        ArrayList hclPaths = new ArrayList();
        PlainTextParser plainTextParser = new PlainTextParser();
        QuarkParser quarkParser = new QuarkParser();
        resources.forEach(path -> {
            if (jsonParser.accept(path)) {
                jsonPaths.add(path);
            } else if (xmlParser.accept(path)) {
                xmlPaths.add(path);
            } else if (yamlParser.accept(path)) {
                yamlPaths.add(path);
            } else if (propertiesParser.accept(path)) {
                propertiesPaths.add(path);
            } else if (protoParser.accept(path)) {
                protoPaths.add(path);
            } else if (hclParser.accept(path)) {
                hclPaths.add(path);
            } else if (quarkParser.accept(path)) {
                quarkPaths.add(path);
            }
        });
        sourceFiles.addAll(jsonParser.parse(jsonPaths, this.baseDir, ctx));
        alreadyParsed.addAll(jsonPaths);
        sourceFiles.addAll(xmlParser.parse(xmlPaths, this.baseDir, ctx));
        alreadyParsed.addAll(xmlPaths);
        sourceFiles.addAll(yamlParser.parse(yamlPaths, this.baseDir, ctx));
        alreadyParsed.addAll(yamlPaths);
        sourceFiles.addAll(propertiesParser.parse(propertiesPaths, this.baseDir, ctx));
        alreadyParsed.addAll(propertiesPaths);
        sourceFiles.addAll(protoParser.parse(protoPaths, this.baseDir, ctx));
        alreadyParsed.addAll(protoPaths);
        sourceFiles.addAll(hclParser.parse(hclPaths, this.baseDir, ctx));
        alreadyParsed.addAll(hclPaths);
        sourceFiles.addAll(plainTextParser.parse(plainTextPaths, this.baseDir, ctx));
        alreadyParsed.addAll(plainTextPaths);
        sourceFiles.addAll(quarkParser.parse(quarkPaths, this.baseDir, ctx));
        alreadyParsed.addAll(quarkPaths);
        return sourceFiles;
    }

    private boolean isOverSizeThreshold(long fileSize) {
        return this.sizeThresholdMb > 0 && fileSize > (long)this.sizeThresholdMb * 1024L * 1024L;
    }

    private boolean isExcluded(Path path) {
        if (!this.exclusions.isEmpty()) {
            Path relative = this.baseDir.relativize(path);
            for (PathMatcher excluded : this.exclusions) {
                if (!excluded.matches(relative)) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isParsedAsPlainText(Path path) {
        if (!this.plainTextMasks.isEmpty()) {
            Path computed = this.baseDir.relativize(path);
            for (PathMatcher matcher : this.plainTextMasks) {
                if (!matcher.matches(computed)) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isIgnoredDirectory(Path searchDir, Path path) {
        for (Path pathSegment : searchDir.relativize(path)) {
            if (!DEFAULT_IGNORED_DIRECTORIES.contains(pathSegment.toString())) continue;
            return true;
        }
        return false;
    }
}

