/*
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Neo4j is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.neo4j.cypher.internal.runtime.interpreted.commands

import org.neo4j.cypher.internal.runtime.interpreted.ExecutionContext
import org.neo4j.cypher.internal.runtime.interpreted.commands.expressions.Expression
import org.neo4j.cypher.internal.runtime.interpreted.commands.predicates.Predicate
import org.neo4j.cypher.internal.runtime.interpreted.PatternGraphBuilder
import org.neo4j.cypher.internal.runtime.interpreted.pipes.QueryState
import org.neo4j.cypher.internal.runtime.interpreted.pipes.matching.MatchingContext
import org.neo4j.cypher.internal.runtime.interpreted.symbols.{SymbolTable, TypeSafe}
import org.neo4j.cypher.internal.v3_5.util.UnNamedNameGenerator.isNamed
import org.neo4j.cypher.internal.v3_5.util.symbols._
import org.neo4j.values.AnyValue
import org.neo4j.values.storable.Values
import org.neo4j.values.virtual.VirtualValues

/*
This class does pattern matching inside an Expression. It's used as a fallback when the
expression cannot be unnested from inside an expression. It is used for pattern expressions
and pattern comprehension
 */
case class PathExpression(pathPattern: Seq[Pattern], predicate: Predicate,
                          projection: Expression, allowIntroducingNewIdentifiers: Boolean = false)
  extends Expression with PatternGraphBuilder {
  private val variables: Seq[(String, CypherType)] =
    pathPattern.flatMap(pattern => pattern.possibleStartPoints.filter(p => isNamed(p._1)))
  private val symbols2 = SymbolTable(variables.toMap)
  private val variablesInClause = Pattern.variables(pathPattern)

  private val matchingContext = new MatchingContext(symbols2, predicate.atoms, buildPatternGraph(symbols2, pathPattern), variablesInClause)
  private val interestingPoints: Seq[String] = pathPattern.
    flatMap(_.possibleStartPoints.map(_._1)).
    filter(isNamed).
    distinct

  override def apply(ctx: ExecutionContext, state: QueryState): AnyValue = {
    // If any of the points we need is null, the whole expression will return null
    val returnNull = interestingPoints.exists(key => ctx.isNull(key))

    if (returnNull) {
      Values.NO_VALUE
    } else {
      VirtualValues.list(matchingContext.
        getMatches(ctx, state). // find matching subgraphs
        filter(predicate.isTrue(_, state)). // filter out graphs not matching the predicate
        map(projection.apply(_, state)).toArray:_*) // project from found subgraphs
    }
  }

  override def children: Seq[TypeSafe with AstNode[_ >: Expression with Pattern <: TypeSafe with AstNode[_ >: Expression with Pattern]]] =
    pathPattern :+ predicate :+ projection

  override def arguments: Seq[Nothing] = Seq.empty

  override def rewrite(f: Expression => Expression): Expression =
    f(PathExpression(pathPattern.map(_.rewrite(f)), predicate.rewriteAsPredicate(f), projection, allowIntroducingNewIdentifiers))

  override def symbolTableDependencies = {
    val patternDependencies = pathPattern.flatMap(_.symbolTableDependencies).toSet
    val startPointDependencies = pathPattern.flatMap(_.possibleStartPoints).map(_._1).filter(isNamed).toSet
    patternDependencies ++ startPointDependencies
  }

  override def toString() = s"PathExpression(${pathPattern.mkString(",")}, $predicate, $projection)"
}
